package csbase.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;

import tecgraf.javautils.core.io.FileUtils;

/**
 * Unzip: descompacta arquivos zip.
 * 
 * @author lmoreira
 */
public final class Unzip {

  /**
   * Tamanho do buffer.
   */
  private static final int BUFFER_SIZE = 64 * 1024;

  /**
   * O arquivo que est sendo manipulado.
   */
  private File file;

  /**
   * Cria um algoritmo para descompresso de arquivos zip.
   * 
   * @param file O arquivo que ser manipulado (No aceita {@code null}).
   */
  public Unzip(File file) {
    if (file == null) {
      throw new IllegalArgumentException("O parmetro file est nulo.");
    }
    this.file = file;
  }

  /**
   * Descompacta o arquivo no diretrio {@code outputDirectory}, reproduzindo a
   * estrutura de diretrios do arquivo.
   * 
   * @param outputDir O diretrio de sada. (No aceita {@code null}).
   * 
   * @throws IOException Se ocorrer um erro de ES.
   */
  public void decompress(File outputDir) throws IOException {
    decompress(outputDir, true);
  }

  /**
   * Descompacta o arquivo no diretrio {@code outputDirectory}.
   * 
   * @param outputDirectory O diretrio de sada. (No aceita {@code null}).
   * @param useDirectoryStructure se {@code true}, reproduz a estrutura de
   *        diretrios do arquivo compactado dentro do diretrio de sada. Se
   *        {@code false}, ignora qualquer eventual estrutura de diretrios do
   *        arquivo e extrai diretamente todos os arquivos compactados na raiz
   *        do diretrio de sada.
   * 
   * @throws IOException Se ocorrer um erro de ES.
   */
  public void decompress(File outputDirectory, boolean useDirectoryStructure)
    throws IOException {
    if (outputDirectory == null) {
      throw new IllegalArgumentException(
        "O parmetro outputDirectory est nulo.");
    }
    FileInputStream fileInputStream = null;
    BufferedInputStream bufferedInputStream = null;
    ZipInputStream zipInputStream = null;
    try {
      fileInputStream = new FileInputStream(file);
      bufferedInputStream = new BufferedInputStream(fileInputStream);
      zipInputStream = new ZipInputStream(bufferedInputStream);
      ZipEntry zipEntry = zipInputStream.getNextEntry();
      while (zipEntry != null) {
        unzipZipEntry(zipInputStream, zipEntry, outputDirectory,
          useDirectoryStructure);
        zipInputStream.closeEntry();
        zipEntry = zipInputStream.getNextEntry();
      }
    }
    finally {
      closeQuietly(zipInputStream, bufferedInputStream, fileInputStream);
    }
  }

  /**
   * <p>
   * Fornece todos os arquivos contidos no zip.
   * </p>
   * 
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   * 
   * @return A lista de arquivos (se no houver arquivos retornar uma lista
   *         vazia).
   * 
   * @throws IOException Erro de ES.
   */
  public List<ZipEntry> listZipEntries() throws IOException {
    List<ZipEntry> zipEntries = new LinkedList<ZipEntry>();
    ZipInputStream zin = null;
    try {

      FileInputStream fin = new FileInputStream(file);
      zin = new ZipInputStream(fin);

      ZipEntry zipEntry;
      while (null != (zipEntry = zin.getNextEntry())) {
        zipEntries.add(zipEntry);
      }
      return Collections.unmodifiableList(zipEntries);
    }
    finally {
      closeQuietly(zin);
    }
  }

  /**
   * <p>
   * Fecha o arquivo silenciosamente.
   * </p>
   * 
   * @param zipFile O arquivo (No aceita {@code null}).
   */
  private void closeQuietly(ZipFile zipFile) {
    if (zipFile != null) {
      try {
        zipFile.close();
      }
      catch (IOException e) {
        e.printStackTrace();
      }
      zipFile = null;
    }
  }

  /**
   * <p>
   * Fecha os streams aninhados silenciosamente.
   * </p>
   * 
   * <p>
   * Ele fecha o primeiro stream no nulo. Se ocorrer uma exceo, ele gera a
   * pilha de execuo na sada de erro.
   * </p>
   * 
   * @param inputStreams Os streams. A ordem dos stream  do mais externo para o
   *        mais interno.
   */
  private void closeQuietly(InputStream... inputStreams) {
    for (InputStream inputStream : inputStreams) {
      if (inputStream != null) {
        try {
          inputStream.close();
        }
        catch (IOException e) {
          e.printStackTrace();
        }
        return;
      }
    }
  }

  /**
   * <p>
   * Fecha os streams silenciosamente.
   * </p>
   * 
   * <p>
   * Ele fecha o primeiro stream no nulo. Se ocorrer uma exceo, ele gera a
   * pilha de execuo na sada de erro.
   * </p>
   * 
   * @param outputStreams Os streams.
   */
  private void closeQuietly(OutputStream... outputStreams) {
    for (OutputStream outputStream : outputStreams) {
      if (outputStream != null) {
        try {
          outputStream.close();
        }
        catch (IOException e) {
          e.printStackTrace();
        }
      }
    }
  }

  /**
   * Descompata um arquivo do arquivo zip.
   * 
   * @param zipInputStream O stream de dados que aponta para o arquivo zip (No
   *        aceita {@code null}).
   * @param zipEntry o arquivo do zip (No aceita {@code null}).
   * @param outputDirectory O diretrio de sada (No aceita {@code null}).
   * @param useDirectoryStructure se {@code true}, reproduz a estrutura de
   *        diretrios do arquivo compactado dentro do diretrio de sada. Se
   *        {@code false}, ignora qualquer eventual estrutura de diretrios do
   *        arquivo e extrai diretamente todos os arquivos compactados na raiz
   *        do diretrio de sada.
   * 
   * @throws IOException Se ocorrer um erro de ES.
   */
  private void unzipZipEntry(ZipInputStream zipInputStream, ZipEntry zipEntry,
    File outputDirectory, boolean useDirectoryStructure) throws IOException {

    String filePath;
    if (useDirectoryStructure) {
      filePath = zipEntry.getName();
    }
    else {
      String[] zipFilePath = FileUtils.splitPath(zipEntry.getName(), "/");
      filePath = zipFilePath[zipFilePath.length - 1];
    }

    File outputFile = new File(outputDirectory, filePath);
    if (zipEntry.isDirectory()) {
      if (!outputFile.exists()) {
        outputFile.mkdirs();
      }
    }
    else {
      File parentFile = new File(outputFile.getParent());
      if (!parentFile.exists()) {
        parentFile.mkdirs();
      }

      int size;
      byte[] buffer = new byte[BUFFER_SIZE];

      FileOutputStream fileOutputStream = null;
      BufferedOutputStream bufferedOutputStream = null;
      try {
        fileOutputStream = new FileOutputStream(outputFile);
        bufferedOutputStream = new BufferedOutputStream(fileOutputStream);
        while ((size = zipInputStream.read(buffer, 0, buffer.length)) != -1) {
          bufferedOutputStream.write(buffer, 0, size);
        }
        bufferedOutputStream.flush();
      }
      finally {
        closeQuietly(bufferedOutputStream, fileOutputStream);
      }
    }
  }
}
