package csbase.util.rmi;

import java.io.IOException;
import java.io.Serializable;
import java.net.ServerSocket;
import java.net.Socket;
import java.rmi.server.RMISocketFactory;

/**
 * <p>
 * Fbrica de socket que mantm referncia para o ltimo {@link ServerSocket}
 * gerado.
 * </p>
 * <p>
 * Atravs desta referncia, podemos obter a porta escolhida para o ltimo
 * objeto exportado utilizando um socket gerado por ela, usando a porta zero.<br>
 * Exemplo:<br>
 * <code>
 * Remote obj = ...
 * 
 * SocketFactory socketFactory = new SocketFactory();
 * UnicastRemoteObject.exportObject(obj, 0, socketFactory, socketFactory);
 * 
 * int port = socketFactory.getServerSocket().getLocalPort();
   * </code>
 * </p>
 * 
 * @author Tecgraf
 */
class SocketFactory extends RMISocketFactory implements Serializable {
  /**
   * A porta na qual o ltimo {@link ServerSocket} criado pelo mtodo
   * {@link #createServerSocket(int)} est escutando.
   */
  private int localPort = -1;

  /**
   * {@inheritDoc}
   */
  public ServerSocket createServerSocket(int port) throws IOException {
    ServerSocket serverSocket = getDelegate().createServerSocket(port);
    localPort = serverSocket .getLocalPort();
    return serverSocket;
  }

  /**
   * {@inheritDoc}
   */
  public Socket createSocket(String host, int port) throws IOException {
    return getDelegate().createSocket(host, port);
  }

  /**
   * Obtm a porta na qual o ltimo {@link ServerSocket} criado pelo mtodo
   * {@link #createServerSocket(int)} est escutando.
   * 
   * @return a porta na qual o ltimo {@link ServerSocket} criado pelo mtodo
   * {@link #createServerSocket(int)} est escutando.
   */
  public int getLocalPort() {
    return localPort;
  }

  /**
   * Obtm o {@link RMISocketFactory} para onde sero delegadas as chamadas.
   * 
   * @return o {@link RMISocketFactory} para onde sero delegadas as chamadas.
   */
  public RMISocketFactory getDelegate() {
    RMISocketFactory delegate = getSocketFactory();
    if (null == delegate) {
      delegate = getDefaultSocketFactory();
    }
    return delegate;
  }
}