package csdk.v2.api.application.manager;

import csdk.v2.api.application.ApplicationException;
import csdk.v2.api.application.IMessage;
import csdk.v2.api.core.IContext;

/**
 * Contexto de gerncia de aplicaes.
 */
public interface IApplicationManagerContext extends IContext {

  /**
   * Inicia a execuo de outra aplicao.
   *
   * @param appId identificador da aplicao.
   * @return o identificador da nova instncia da aplicao ou {@code null}
   * caso no exista aplicao com o identificador definido.
   * @throws ApplicationException caso ocorra falha ao iniciar a nova
   * aplicao.
   */
  public String runApplication(String appId) throws ApplicationException;

  /**
   * Verifica se a aplicao especificada est sendo executada.
   *
   * @param instanceId o identificador da instncia da aplicao.
   * @return {@code true} se a aplicao estiver em execuo ou {@code
   * false}, caso constrrio.
   */
  public boolean isApplicationRunning(String instanceId);

  /**
   * Finaliza a execuo de outra aplicao.
   *
   * @param instanceId o identificador da instncia da aplicao.
   * @throws ApplicationException caso ocorra falha ao fechar a aplicao.
   */
  public void killApplication(String instanceId) throws ApplicationException;

  /**
   * Envia uma mensagem sncrona a uma instncia de aplicao em execuo.
   *
   * @param instanceId o identificador da instncia da aplicao.
   * @param message a mensagem a ser enviada.
   * @return a resposta da mensagem.
   * @throws ApplicationException em caso de erro ao tentar enviar a mensagem.
   */
  public Object sendSyncMessage(String instanceId,
    IMessage message) throws ApplicationException;

  /**
   * Envia uma mensagem assncrona a todas instncias de aplicao em execuo.
   *
   * @param message a mensagem a ser enviada.
   * @throws ApplicationException em caso de erro ao tentar enviar a mensagem.
   */
  public void broadcastAsyncMessage(
    IMessage message) throws ApplicationException;

  /**
   * Envia uma mensagem assncrona a uma instncia de aplicao em execuo.
   *
   * @param instanceId o identificador da instncia da aplicao.
   * @param message a mensagem a ser enviada.
   * @throws ApplicationException em caso de erro ao tentar enviar a mensagem.
   */
  public void sendAsyncMessage(String instanceId,
    IMessage message) throws ApplicationException;

}
