package csdk.v1_0.runner.user;

import java.awt.Window;
import java.util.HashMap;
import java.util.Map;

import csdk.v1_0.api.user.IUser;
import csdk.v1_0.api.user.IUserContext;
import csdk.v1_0.api.user.UserException;
import csdk.v1_0.runner.CSDKLogger;
import csdk.v1_0.runner.core.AbstractContext;

/**
 * Implementao do contexto de usurios para o Runner.
 * 
 * @author Tecgraf
 */
public class RunnerUserContext extends AbstractContext implements IUserContext {

  /**
   * O mapa de usurios.
   */
  private final Map<String, IUser> users;
  /**
   * O usurio corrente.
   */
  private IUser currentUser;

  /**
   * Contrutor.
   */
  public RunnerUserContext() {
    this.users = new HashMap<String, IUser>();
  }

  /**
   * Contrutor.
   * 
   * @param users mapa de usurios do sistema.
   */
  public RunnerUserContext(Map<String, IUser> users) {
    this();
    this.users.putAll(users);
  }

  /**
   * Contrutor.
   * 
   * @param users mapa de usurios do sistema.
   * @param currentUser login do usurio corrente.
   * 
   * @throws UserException caso o usurio corrente no esteja definido no mapa
   *         de usurios.
   */
  public RunnerUserContext(Map<String, IUser> users, String currentUser)
    throws UserException {
    this(users);
    this.currentUser = getUserByLogin(currentUser, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final void contextActivated() {
    if (this.currentUser == null) {
      String login = System.getProperty("user.name");
      try {
        this.currentUser = getUserByLogin(login, null);
      }
      catch (UserException e) {
        CSDKLogger logger = CSDKLogger.getInstance();
        logger.logSevere("Current user not defined!");
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IUser getCurrentUser() throws UserException {
    return currentUser;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IUser[] getAllUsers(Window window) throws UserException {
    return users.values().toArray(new IUser[users.size()]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IUser getUserByLogin(String login, Window window) throws UserException {
    return users.get(login);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void contextDeactivated() {
    this.currentUser = null;
    this.users.clear();
  }
}
