package tecgraf.javautils.concurrent.locks;

import java.lang.ref.WeakReference;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.logging.Logger;

import tecgraf.javautils.concurrent.locks.exceptions.InvalidLockIdException;
import tecgraf.javautils.concurrent.locks.exceptions.ObjectNotLockedException;

/**
 * Classe responsvel por gerenciar os locks sobre objetos compartilhados.
 * <p>
 * Os locks podem ser obtidos de duas formas:
 * <ul>
 * <li><b>Requisio imediata</b>: o <code>LockManager</code> tenta adquirir o
 * lock no momento em que a chamada  feita e, nesse caso, retorna o
 * identificador do lock. Ver
 * {@link #acquireLock(LockPolicy, Object, SharedAccessObject, LockDependency...)}
 * <li><b>Requisio futura</b>: o <code>LockManager</code> tenta adquirir o
 * lock no momento da chamada mas, no caso de no conseguir, coloca a requisio
 * em uma lista de espera. As requisies estabelecem um tempo de espera limite
 * pelo lock. Dessa forma, o consumidor no fica bloqueado enquanto o lock no 
 * concedido. A sincronizao ocorre com o uso de um objeto de callback
 * implementado como um <code>LockFuture</code>. Ver
 * {@link #tryAcquireLock(LockPolicy, Object, SharedAccessObject, LockFuture, long, LockDependency...)}
 * </ul>
 * <p>
 * Os pedidos de lock so adquiridos ou no de acordo com a poltica de acesso
 * requisitada. As polticas atualmente aceitas so:
 * <ul>
 * <li><b>Concurrent Read (CR)</b>: Permite acesso compartilhado ao recurso,
 * exceto se houver um consumidor com lock <code>Exclusive</code>. Consumidores
 * nesse modo aceitam ler um recurso mesmo sabendo que ele pode estar sendo
 * alterado por um escritor <code>PW</code>.
 * <li><b>Protected Read (PR)</b>: Permite acesso compartilhado ao recurso,
 * exceto com o <code>PW</code> e com o <code>E</code>. Corresponde o modo
 * shared tradicional ou ao leitor tradicional na poltica leitor/escritor.
 * <li><b>Protected Write (PW)</b>: Permite acesso compartilhado apenas com o
 * modo <code>CR</code>. Se aplica bem ao caso onde o consumidor precisa
 * exclusividade na escrita de um recurso em relao a outros escritores, mas
 * no aos leitores <code>CR</code>.
 * <li><b>Exclusive (E)</b>: Garante o acesso exclusivo ao recurso, isso ,
 * apenas um consumidor pode ter acesso ao recurso. Corresponde ao escritor
 * tradicional na poltica leitor/escritor.
 * </ul>
 * <p>
 * Opcionalmente, os pedidos de lock podem definir uma cadeia de dependncia de
 * locks sobre outros objetos da aplicao. Um lock sobre um determinado objeto
 * somente pode ser obtido se os locks dependentes tambm o forem. Ver
 * {@link LockDependency}.
 * <p>
 * Utiliza duas threads: uma para gerenciar as requisies de lock por timeout e
 * uma outra para enviar os eventos aos listeners desse <code>LockManager</code>
 * . As threads so iniciadas no momento da instanciao do
 * <code>LockManager</code> e so finalizadas com a chamada ao mtodo
 * {@link #close()}.
 * 
 * @author Tecgraf PUC-Rio
 */
public class LockManager {

  /**
   * Define se o gerenciador aceita reentrncia de pedidos de lock ou no
   * 
   */
  public enum ManagerType {
    /** Aceita reentrncia */
    ALLOWS_REENTRANCE,
    /** No aceita reentrncia (default) */
    NO_REENTRANCE
  }

  /**
   * Determina se gerenciador permite reentrncia ou no
   */
  private ManagerType managerType;

  /**
   * Tempo mximo que a thread de processamento da fila dorme: 1 minuto.
   */
  private final static long MAX_SLEEP_INTERVAL = 60000;

  /**
   * Constante que representa a espera, sem timeout, por um lock de um objeto.
   */
  public final static int INFINITE_TIMEOUT = -1;

  /**
   * Mapa com os locks para objetos. Os locks possuem referncias fracas para os
   * requisitantes (donos) dos locks; se elas expiram, os locks so liberados.
   */
  private final Map<Object, LockInfo> lockedObjects;

  /**
   * Mapa com os identificadores de lock e seus respectivos objetos, utilizado
   * para relacionamento no momento do release
   */
  private final Map<LockId, SharedAccessObject> lockIdMap;

  /**
   * Lista de observadores para todos os tipos de objetos
   */

  /** Indica o trmino do processamento da fila de locks */
  private boolean exitManagerThread = false;

  /**
   * Fila de requisies de lock.
   */
  private final LockRequestQueue lockRequestQueue;

  /**
   * Fila de eventos de lock.
   */
  private final LockEventQueue lockEventQueue;

  /**
   * Gerenciador de observadores dos locks
   */
  private LockListenerManager listenerManager;

  /**
   * Log do mecanismo de lock
   */
  private LockLogger logger;

  /**
   * Requisio de lock de um objeto.
   */
  private class LockRequest {

    /**
     * Indica se a requisio de lock foi solicitado por um servio, a partir de
     * uma <code>Thread</code> do servidor. Quando o lock  obtido por uma
     * <code>Thread</code> que no vem diretamente do cliente, no h sesso
     * associada.
     */
    boolean serverRequested = true;

    /**
     * Referncia fraca para identificador usurio que solitou o lock. Pode ser
     * null se o pedido de lock veio de uma chamada que no foi originada pela
     * thread do cliente.
     */
    WeakReference<Object> ownerKeyRef;

    /** Observador a ser notificado quando lock for obtido ou expirado. */
    LockFuture future;

    /** Data limite para que o lock seja obtido */
    long expirationDate;

    /** Poltica de requisio do lock */
    LockPolicy lockPolicy;

    /** Objeto sobre o qual o lock foi solicitado. */
    SharedAccessObject objectToLock;

    /** Dependncia do objeto referente ao lock */
    LockDependency<SharedAccessObject, SharedAccessObject>[] dependencies;

    /**
     * Constri uma requisio de lock de uma thread do servidor
     * 
     * @param future observador a ser notificado quando for obtido ou expirado
     * @param expirationDate data limite para que o lock seja obtido
     * @param lockPolicy poltica de requisio do lock
     * @param object objeto sobre o qual foi solicitado o lock
     * @param dependencies dependncias do lock a ser obtido
     */
    LockRequest(LockFuture future, long expirationDate, LockPolicy lockPolicy,
      SharedAccessObject object,
      LockDependency<SharedAccessObject, SharedAccessObject>... dependencies) {
      this.future = future;
      this.expirationDate = expirationDate;
      this.lockPolicy = lockPolicy;
      this.objectToLock = object;
      this.dependencies = dependencies;
    }

    /**
     * Constri uma requisio de lock de um cliente.
     * 
     * @param ownerKey identificador do usurio
     * @param future observador a ser notificado quando for obtido ou expirado
     * @param expirationDate data limite para que o lock seja obtido
     * @param lockPolicy poltica de requisio do lock
     * @param object objeto sobre o qual foi solicitado o lock
     * @param dependencies dependncias do lock a ser obtido
     */
    LockRequest(Object ownerKey, LockFuture future, long expirationDate,
      LockPolicy lockPolicy, SharedAccessObject object,
      LockDependency<SharedAccessObject, SharedAccessObject>... dependencies) {
      this(future, expirationDate, lockPolicy, object, dependencies);
      this.ownerKeyRef = new WeakReference<Object>(ownerKey);
      this.serverRequested = false;
    }

    /**
     * Indica se todas a referncia  requisio de lock expirou.
     * 
     * @return flag indicando se todas as referncias ao lock expiraram
     */
    boolean isInvalid() {
      if (serverRequested) {
        return false;
      }
      if (ownerKeyRef == null) {
        return true;
      }
      Object ownerKey = ownerKeyRef.get();
      return ownerKey == null;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
      StringBuilder str = new StringBuilder();
      str.append("Objeto solicitado para lock: " + objectToLock.toString());
      if (serverRequested) {
        str.append("\n -Requisio por thread de servidor");
      }
      else {
        if (ownerKeyRef.get() != null) {
          str.append("\n -Identificador do Usurio: " + ownerKeyRef.get());
        }
        else {
          str.append("\n -Referncia ao usurio invlida.");
        }
      }
      str.append("\n -Data de expirao: " + new Date(expirationDate));
      str.append("\n -Poltica de Lock: " + lockPolicy.toString());
      return str.toString();
    }
  }

  /**
   * Representa a fila de requisies de lock.
   */
  private class LockRequestQueue {
    /**
     * Mapa com as filas de requisies de locks.
     */
    private final Map<Object, List<LockRequest>> lockRequests;

    /**
     * Requisies ordenadas pela data de expirao.
     */
    private final List<LockRequest> sortedLockRequests;

    /**
     * Constri a fila de requisio de locks.
     */
    LockRequestQueue() {
      this.lockRequests = new Hashtable<Object, List<LockRequest>>();
      this.sortedLockRequests = new ArrayList<LockRequest>();
    }

    /**
     * Obtm o prximo pedido de lock a expirar
     * 
     * @return o prximo pedido de lock a expirar
     */
    LockRequest getNextRequest() {
      synchronized (sortedLockRequests) {
        if (sortedLockRequests.size() > 0) {
          return sortedLockRequests.get(0);
        }
      }
      return null;
    }

    /**
     * Adiciona a requisio o pedido de lock na fila.
     * 
     * @param request pedido de lock
     */
    synchronized void addRequest(LockRequest request) {
      List<LockRequest> requests;
      synchronized (lockRequests) {
        requests = lockRequests.get(request.objectToLock);
        if (requests == null) {
          requests = new ArrayList<LockRequest>();
          lockRequests.put(request.objectToLock, requests);
        }
      }
      requests.add(request);
      addSortedRequest(request);
      notify();
    }

    /**
     * Adiciona uma requisio  fila ordenada por prioridade.
     * 
     * @param request requisio de lock
     */
    private void addSortedRequest(LockRequest request) {
      synchronized (sortedLockRequests) {
        if (sortedLockRequests.size() <= 0) {
          sortedLockRequests.add(request);
          return;
        }
        for (int i = 0; i < sortedLockRequests.size(); i++) {
          if (request.expirationDate < sortedLockRequests.get(i).expirationDate) {
            sortedLockRequests.add(i, request);
            return;
          }
        }
        /*
         * Se chegou aqui, a data de expirao  maior do que todos que j est
         * na fila.
         */
        sortedLockRequests.add(request);
      }
    }

    /**
     * Obtm as requisies de lock para um determinado objeto.
     * 
     * @param object objeto requisitado
     * @return as requisies de lock para um determinado objeto
     */
    synchronized LockRequest[] getRequests(Object object) {
      synchronized (lockRequests) {
        List<LockRequest> requests = lockRequests.get(object);
        return requests.toArray(new LockRequest[requests.size()]);
      }
    }

    /**
     * Remove as requisies de um determinado objeto.
     * 
     * @param object objeto solicitado presente na fila de espera
     * @param requests requisies a serem removidas
     * @return true se os pedidos de locks estavam na fila
     */
    synchronized boolean remove(Object object, List<LockRequest> requests) {
      synchronized (lockRequests) {
        List<LockRequest> objectQueue = lockRequests.get(object);
        if (objectQueue == null) {
          return false;
        }
        objectQueue.removeAll(requests);
        if (objectQueue.isEmpty()) {
          lockRequests.remove(object);
        }
      }
      synchronized (sortedLockRequests) {
        return sortedLockRequests.removeAll(requests);
      }
    }

    /**
     * Remove a requisio de lock de um determinado objeto.
     * 
     * @param object objeto solicitado presente na fila de espera
     * @param request requisio a ser removidas
     * @return true se o pedido de lock estava na fila
     */
    synchronized boolean remove(Object object, LockRequest request) {
      List<LockRequest> requests = new ArrayList<LockRequest>();
      requests.add(request);
      return this.remove(object, requests);
    }

    /**
     * Obtm os objetos das filas de lock. Este mtodo chama o mtodo
     * <code>wait</code> caso no haja requisies de lock. O
     * <code>notify</code>  chamado quando um lock  requirido pelo mtodo
     * <code>addRequest</code> ou quando um lock  liberado.
     * 
     * @return os objetos das filas de lock
     */
    synchronized Object[] getAndWaitForQueuesKeys() {
      if (size() <= 0) {
        try {
          wait();
        }
        catch (InterruptedException e) {
        }
      }
      synchronized (lockRequests) {
        return lockRequests.keySet().toArray(new Object[lockRequests.size()]);
      }
    }

    /**
     * Obtm o nmero de filas de requisies de lock.
     * 
     * @return o nmero de filas de requisies de lock
     */
    synchronized int size() {
      return lockRequests.size();
    }

    /**
     * Dorme durante o tempo especificado ou at que seja acordada por outra
     * thread que invoque <code>notify</code>.
     * 
     * @param interval tempo mximo de interrupo da execuo
     */
    synchronized void sleep(long interval) {
      try {
        wait(interval);
      }
      catch (InterruptedException e) {
      }
    }

    /**
     * Mtodo chamado quando um lock  liberado.
     */
    synchronized void lockReleased() {
      notify();
    }

    /**
     * Processa as filas de requisies de locks. Antes de obter o lock,
     * verifica se a requisio expirou pelo timeout. No final, notifica os
     * <code>LockFuture</code> sobre os locks que foram obtidos ou que
     * expiraram.
     */
    void processRequests() {
      Object[] objects = getAndWaitForQueuesKeys();
      for (Object objectToLock : objects) {
        //Lista de locks adquiridos durante processamento
        List<LockRequest> acquiredLocks = new ArrayList<LockRequest>();
        //Lista de pedidos de lock expirados durante processamento
        List<LockRequest> expiredAndInvalidLockRequests =
          new ArrayList<LockRequest>();
        LockRequest[] requests = getRequests(objectToLock);
        long currentDate = System.currentTimeMillis();
        // Percorre a fila de solicities de lock. Se no consegue obter o lock,
        // verifica se expirou
        for (LockRequest lockRequest : requests) {
          //Se a requisio ainda  vlida (no ser se cliente cair durante espera do lock) tenta obter o lock
          if (!lockRequest.isInvalid()) {
            //Adiciona falha na tentativa de obter o lock
            lockRequest.future.addAttempt();
            //
            LockAcquireResult lockAcquireResult =
              getLock(lockRequest.lockPolicy, lockRequest.ownerKeyRef.get(),
                lockRequest.objectToLock, null, lockRequest.dependencies);
            // Se lock foi obtido, ento identificador  diferente de null
            if (lockAcquireResult.lockResult.acquiredLock()) {
              acquiredLocks.add(lockRequest);
              //Notifica o requisitor de que o mesmo adquiriu o lock
              notifyObjectLocked(lockRequest, lockAcquireResult.lockResult
                .getLockId());
            }
            else {
              //Verifica se o tempo de espera pelo lock excedeu o tempo limite
              if ((lockRequest.expirationDate != INFINITE_TIMEOUT)
                && (currentDate >= lockRequest.expirationDate)) {
                //Adiciona a requisio na lista de requisies a serem removidas devido a mesma estar expirada
                expiredAndInvalidLockRequests.add(lockRequest);
                //Notifica o requisitor de que o pedido de lock expirou
                notifyLockExpired(lockRequest);
              }
            }
          }
          else {
            //Adiciona a requisio na lista de requisies a serem removidas devido a mesma estar invlida
            expiredAndInvalidLockRequests.add(lockRequest);
          }
        }
        // Remove as requisies que obtiveram o lock
        remove(objectToLock, acquiredLocks);
        // Remove as requisies expiradas e invlidas
        remove(objectToLock, expiredAndInvalidLockRequests);
      }
    }

    /**
     * Notifica o listener que o lock foi obtido.
     * 
     * @param lockRequest requisio de lock
     * @param lockId identificador do novo lock
     */
    void notifyObjectLocked(final LockRequest lockRequest, final LockId lockId) {
      new Thread(new Runnable() {
        @Override
        public void run() {
          try {
            //Avisa ao requisitor que obteve o lock solicitado
            lockRequest.future.objectLocked(lockId);
          }
          catch (RemoteException e) {
            //Remove o lock recm-adquirido
            releaseLock(lockId);
            //Insere a requisio novamente na fila de requisies para tentar 
            //obter o lock e avisar o requisitor em um momento posterior
            lockRequestQueue.addRequest(lockRequest);
          }
        }
      }).start();
    }

    /**
     * Notifica o listener que o lock foi expirado.
     * 
     * @param lockRequest requisio de lock
     */
    void notifyLockExpired(final LockRequest lockRequest) {
      new Thread(new Runnable() {
        @Override
        public void run() {
          try {
            //Avisa ao requisitor que seu pedido de lock expirou
            lockRequest.future.objectLockExpired();
          }
          catch (RemoteException e) {
            //Insere a requisio novamente na fila de requisies para tentar 
            //avisar o requisitor em um momento posterior 
            lockRequestQueue.addRequest(lockRequest);
          }
        }
      }).start();
    }
  }

  /**
   * Tipo de evento de lock
   */
  private enum LockEventType {
    /**
     * Determina que um lock sobre objeto foi adquirido
     */
    OBJECT_LOCKED,
    /**
     * Determina que o lock sobre o objeto foi removido
     */
    LOCK_RELEASED
  }

  /**
   * Aviso sobre estado de lock de um objeto.
   */
  private class LockEvent {

    /** Tipo de evento */
    LockEventType eventType;

    /** Identificador usurio que adquiriu ou removeu o lock */
    Object ownerKey;

    /** Poltica do lock adquirido ou liberado */
    LockPolicy lockPolicy;

    /** Objeto sobre o qual o lock foi adquirido ou removido. */
    SharedAccessObject object;

    /**
     * Constri um aviso de mudana de lock.
     * 
     * @param eventType tipo de aviso
     * @param ownerKey identificador do usurio
     * @param lockPolicy poltica do lock
     * @param object objeto sobre o qual foi o lock foi adquirido ou removido
     */
    LockEvent(LockEventType eventType, Object ownerKey, LockPolicy lockPolicy,
      SharedAccessObject object) {
      this.eventType = eventType;
      this.ownerKey = ownerKey;
      this.lockPolicy = lockPolicy;
      this.object = object;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
      StringBuilder str = new StringBuilder();
      str.append("\n #Event Type: " + eventType.toString());
      str.append("\n -Object: " + object.toString());
      str.append("\n -UniqueCode: " + object.getUniqueCode().toString());
      str.append("\n -Owner identification: " + ownerKey.toString());
      str.append("\n -Policy: " + lockPolicy.toString());
      str.append("\n");
      return str.toString();
    }
  }

  /**
   * Representa a fila de eventos de lock.
   */
  private class LockEventQueue {
    /**
     * Requisies ordenadas pela data de expirao.
     */
    private final List<LockEvent> lockEvents;

    /**
     * Constri a fila de eventos de locks.
     */
    LockEventQueue() {
      this.lockEvents = new ArrayList<LockEvent>();
    }

    /**
     * Adiciona o evento de lock na fila.
     * 
     * @param lockEvent aviso de lock
     */
    synchronized void addEvent(LockEvent lockEvent) {
      synchronized (this.lockEvents) {
        this.lockEvents.add(lockEvent);
      }
      notify();
    }

    /**
     * Adiciona uma coleo de eventos de locks na fila.
     * 
     * @param lockEvents avisos de lock
     */
    synchronized void addEvents(List<LockEvent> lockEvents) {
      synchronized (this.lockEvents) {
        this.lockEvents.addAll(lockEvents);
      }
      notify();
    }

    /**
     * Dorme at que seja acordada por outra thread que invoque
     * <code>notify</code>.
     */
    synchronized void sleep() {
      try {
        wait();
      }
      catch (InterruptedException e) {
      }
    }

    /**
     * Processa a fila de eventos de locks. Itera sobre cada evento e o repassa
     * para ser utilizado na notificao aos observadores do objeto relacionado
     * ao lock.
     */
    void processEvents() {
      //Processa cada evento da lista de eventos
      synchronized (lockEvents) {
        while (lockEvents.size() > 0) {
          LockEvent lockEvent = lockEvents.get(0);
          notifyListeners(lockEvent);
          //Retira da lista
          lockEvents.remove(0);
        }
      }
    }

    /**
     * Notifica os observadores sobre o lock de um objeto
     * 
     * @param lockEvent evento de lock
     */
    private void notifyListeners(final LockEvent lockEvent) {
      new Thread(new Runnable() {
        @Override
        public void run() {
          //Recupera lista de listeners interessados no objeto
          Set<LockListener> listeners;
          synchronized (listenerManager) {
            listeners = listenerManager.getListeners(lockEvent.object);
          }
          //Notifica os listeners de que o lock foi obtido
          for (LockListener listener : listeners) {
            try {
              //Informa que lock de objeto foi obtido
              if (lockEvent.eventType == LockEventType.OBJECT_LOCKED) {
                listener.onObjectLocked(lockEvent.object, lockEvent.ownerKey,
                  lockEvent.lockPolicy);
              }
              //Informa que o lock sobre o objeto foi removido
              else if (lockEvent.eventType == LockEventType.LOCK_RELEASED) {
                listener.onLockReleased(lockEvent.object, lockEvent.ownerKey,
                  lockEvent.lockPolicy);
              }
            }
            catch (RemoteException e) {
              e.printStackTrace();
            }
          }
        }
      }).start();
    }
  }

  /**
   * Representa o resultado da aquisio do lock
   * 
   * 
   * @author Tecgraf
   */
  private class LockAcquireResult {

    /**
     * Identificador do resultado
     */
    LockResult lockResult;

    /**
     * Lista de eventos
     */
    List<LockEvent> eventList;

    /**
     * Construtor utilizado quando o lock  obtido
     * 
     * @param lockResult identificador do lock resultante
     */
    LockAcquireResult(LockResult lockResult) {
      this.lockResult = lockResult;
      this.eventList = new ArrayList<LockEvent>();
    }
  }

  /**
   * Constri o gerenciador de locks do tipo no reentrante
   * 
   */
  public LockManager() {
    this(ManagerType.NO_REENTRANCE, null);
  }

  /**
   * Constri o gerenciador de locks informando o tipo do gerenciador
   * 
   * @param managerType o tipo do gerenciador
   */
  public LockManager(ManagerType managerType) {
    this(managerType, null);
  }

  /**
   * Constri o gerenciador de locks do tipo no reentrante
   * 
   * @param logger logger utilizado para mensagens do gerenciador de lock
   */
  public LockManager(Logger logger) {
    this(ManagerType.NO_REENTRANCE, logger);
  }

  /**
   * Constri o gerenciador de locks que informa o tipo do gerenciador
   * 
   * @param managerType o tipo do gerenciador
   * @param logger logger utilizado para mensagens do gerenciador de lock
   */
  public LockManager(ManagerType managerType, Logger logger) {
    //Define tipo do gerenciador - padro no permite reentrncia    
    this.managerType = managerType;
    //Recebe a instncia do logger
    this.logger = LockLogger.getInstance();
    if (logger != null) {
      this.setLogger(logger);
    }
    //Cria mapa de objetos com locks
    this.lockedObjects = new Hashtable<Object, LockInfo>();
    //Cria relacionamento entre identificadores de lock e respectivos objetos 
    this.lockIdMap = new Hashtable<LockId, SharedAccessObject>();
    //Cria lista de requisio de locks
    this.lockRequestQueue = new LockRequestQueue();
    //Cria lista de eventos de locks
    this.lockEventQueue = new LockEventQueue();
    //Cria gerenciador de observadores de locks
    this.listenerManager = new LockListenerManager();
    //Inicia thread para tratamento das requisies de locks
    startRequestThread();
    //Inicia thread para tratamento dos eventos de locks
    startEventThread();
    //Inicia thread para tratamento de locks invlidos
    startInvalidLocksCollectorThread();
  }

  /**
   * Retorna o tipo de gerenciador
   * 
   * @return tipo do gerenciador
   */
  public ManagerType getManagerType() {
    return this.managerType;
  }

  /**
   * Mtodo para fechar o gerenciador de locks
   */
  public void close() {
    this.stopThreads();
  }

  /**
   * Recupera o logger utilizado
   * 
   * @return o logger utilizado
   */
  public Logger getLogger() {
    return logger.getLogger();
  }

  /**
   * Determina o logger a ser utilizado
   * 
   * @param theLogger o logger a ser utilizado
   */
  public void setLogger(Logger theLogger) {
    logger.setLogger(theLogger);
  }

  /**
   * Tentativa de aquisio do lock de um objeto passando as dependncias que
   * devem ser obtidas para que o lock seja obtido com sucesso.
   * 
   * @param lockPolicy poltica de lock
   * @param ownerKey identificador do requisitante do lock
   * @param object objeto referente  requisio de lock
   * @param lockDependencies dependncia do objeto solicitado
   * @return identificador do lock, se o mesmo foi obtido com sucesso, ou null
   *         caso contrrio
   */
  public LockId acquireLock(LockPolicy lockPolicy, Object ownerKey,
    SharedAccessObject object, LockDependency... lockDependencies) {
    //Testa se objeto a ter o lock obtido  nulo
    if (object == null) {
      throw new IllegalArgumentException(
        "Objeto requisitado no pode ser nulo.");
    }
    //Tenta obter o lock do objeto
    LockAcquireResult acquire =
      this.getLock(lockPolicy, ownerKey, object, null, lockDependencies);
    if (!acquire.lockResult.acquiredLock()) {
      return null;
    }
    //Retorna o identificador para o cliente
    return acquire.lockResult.getLockId();
  }

  /**
   * Tentativa de aquisio do lock de um objeto passando as dependncias que
   * devem ser obtidas para que o lock seja obtido com sucesso. <br>
   * <br>
   * <i>Caso o lock <b>no</b> seja obtido, retorno conter um mapa com os
   * identificadores dos donos dos locks existentes que so conflitantes com o
   * lock solicitado sobre o objeto alvo.</i>
   * 
   * @param lockPolicy poltica de lock
   * @param ownerKey identificador do requisitante do lock
   * @param object objeto referente  requisio de lock
   * @param lockDependencies dependncia do objeto solicitado
   * @return resultado da tentativa do lock, contendo identificador caso o mesmo
   *         foi obtido com sucesso, ou mapa com os identificadores dos donos
   *         dos locks existentes que so conflitantes com o lock solicitado
   *         sobre o objeto alvo.
   */
  public LockResult acquireLockWithResult(LockPolicy lockPolicy,
    Object ownerKey, SharedAccessObject object,
    LockDependency... lockDependencies) {
    //Testa se objeto a ter o lock obtido  nulo
    if (object == null) {
      throw new IllegalArgumentException(
        "Objeto requisitado no pode ser nulo.");
    }
    //Tenta obter o lock do objeto
    LockAcquireResult acquire =
      this.getLock(lockPolicy, ownerKey, object, null, lockDependencies);
    //Retorna resultado da aquisio do lock
    return acquire.lockResult;
  }

  /**
   * Tenta obter um lock exclusivo para um objeto compartilhado. Caso este se
   * encontre bloqueado, uma requisio  cadastrada na fila.
   * 
   * @param lockPolicy poltica do lock solicitado
   * @param ownerKey identificador do usurio
   * @param object objeto referente  requisio do lock
   * @param future observador a ser notificado quando o lock  obtido ou o
   *        timeout atingido
   * @param timeout tempo de espera pelo lock
   * @param lockDependencies dependncias do lock requisitado
   */
  public void tryAcquireLock(LockPolicy lockPolicy, Object ownerKey,
    SharedAccessObject object, LockFuture future, long timeout,
    LockDependency... lockDependencies) {
    //Testa se objeto  nulo
    if (object == null) {
      throw new IllegalArgumentException(
        "Objeto requisitado no pode ser nulo.");
    }
    //Registra requisio de lock na fila de requisies
    this.registerLockRequest(lockPolicy, ownerKey, object, future, timeout,
      lockDependencies);
  }

  /**
   * Cria um lock para um objeto passando as dependncias que devem ser obtidas
   * para que o lock seja obtido com sucesso.
   * 
   * @param lockPolicy poltica de lock
   * @param ownerKey identificador do requisitante do lock
   * @param object objeto referente  requisio de lock
   * @param originatorLockId identificador do lock origem
   * @param lockDependencies dependncia do objeto
   * @return resultado com identificador do lock obtido e lista de eventos do
   *         mesmo
   */
  synchronized private LockAcquireResult getLock(LockPolicy lockPolicy,
    Object ownerKey, SharedAccessObject object, LockId originatorLockId,
    LockDependency<SharedAccessObject, SharedAccessObject>... lockDependencies) {
    //Verifica se j existe info de lock do objeto requisitado
    LockInfo lockInfo;
    synchronized (lockedObjects) {
      //Tenta obter o lock a partir do indetificador nico
      lockInfo = lockedObjects.get(object.getUniqueCode());
      if (lockInfo == null) {
        //Se no existir um info de lock, cria para o objeto requisitado
        lockInfo =
          new LockInfo(object,
            this.managerType == ManagerType.ALLOWS_REENTRANCE);
      }
      //Verifica se a poltica solicitada permite a aquisio do lock
      //TODO Retornar motivo da no obteno
      if (!lockPolicy.allowsLock(lockInfo, ownerKey, originatorLockId)) {
        boolean releasedLock = false;
        List<Lock> locks = lockInfo.getLocks(originatorLockId);
        for (Lock lock : locks) {
          logger.fine("Verifica lock invlido");
          // Caso lock seja de primeira ordem e esteja invlido, remove o mesmo
          if (lock.isFirstOrder() && lock.isInvalid()) {
            logger.fine("Lock invlido! Liberando o lock");
            this.releaseLock(lock.getId());
            releasedLock = true;
          }
        }
        if (releasedLock) {
          return this.getLock(lockPolicy, ownerKey, object, originatorLockId,
            lockDependencies);
        }
        else {
          logger.fine("% " + ownerKey + " no pode obter lock do objeto "
            + object);
          return new LockAcquireResult(new LockResult(this.getLockOwnersKeys(
            object, lockPolicy.getIncompatibleLockPolicies())));
        }
      }
      //Insere info de lock na lista de locks
      lockedObjects.put(object.getUniqueCode(), lockInfo);
    }
    //Cria lock para objeto solicitado
    Lock lock = lockInfo.newLock(lockPolicy, ownerKey, originatorLockId);
    logger.fine(ownerKey + " obtm lock " + lockPolicy + " do objeto: "
      + object);
    //Cria objeto resultante da aquisio
    LockAcquireResult result =
      new LockAcquireResult(new LockResult(lock.getId()));
    //Adiciona ao resultado
    result.eventList.add(new LockEvent(LockEventType.OBJECT_LOCKED, ownerKey,
      lockPolicy, object));
    //Insere no mapa de identificadores e objetos
    synchronized (lockIdMap) {
      this.lockIdMap.put(lock.getId(), object);
    }
    //Trata dependncias se existirem
    if (lockDependencies != null) {
      for (LockDependency<SharedAccessObject, ?> dependency : lockDependencies) {
        //Se a dependncia for nula passa para a prxima
        if (dependency == null) {
          continue;
        }
        for (SharedAccessObject depObject : dependency.getDependency(object)) {
          //Testa se objeto dependente  nulo
          if (depObject == null) {
            continue;
          }
          //Tenta adquirir o lock das dependncias    
          LockAcquireResult depLockAcquireResult =
            getLock(dependency.getPolicy(), ownerKey, depObject, lock
              .getOriginatorLockId(), dependency.getChain());
          //Verifica se o lock foi obtido
          if (depLockAcquireResult.lockResult.acquiredLock()) {
            //Se sim, adiciona o identificador como uma dependncia
            lock.addIdDependency(depLockAcquireResult.lockResult.getLockId());
            //Adiciona lista de eventos
            result.eventList.addAll(depLockAcquireResult.eventList);
          }
          else {
            //Caso contrrio, solicita a remoo do lock
            this.releaseLock(lock.getId(), false);
            //Limpa os eventos
            result.eventList.clear();
            //Retorna null
            return depLockAcquireResult;
          }
        }
      }
    }
    //Se o lock  de primeira ordem, notifica os observadores
    if (lock.isFirstOrder()) {
      //Adiciona os eventos  lista
      this.lockEventQueue.addEvents(result.eventList);
    }
    //Caso todas as dependncias estejam satisfeitas retorna o identificador
    return result;
  }

  /**
   * Tenta obter um lock para um objeto compartilhado. Caso este se encontre
   * bloqueado, uma requisio  cadastrada na fila.
   * 
   * @param lockPolicy poltica do lock solicitado
   * @param ownerKey identificador do usurio
   * @param object objeto referente  requisio do lock
   * @param future observador a ser notificado quando o lock  obtido ou o
   *        timeout atingido
   * @param timeout tempo de espera pelo lock
   * @param lockDependencies dependncias do lock a ser obtido
   */
  private void registerLockRequest(LockPolicy lockPolicy, Object ownerKey,
    SharedAccessObject object, LockFuture future, long timeout,
    LockDependency<SharedAccessObject, SharedAccessObject>... lockDependencies) {
    //Define o tempo mximo de espera da requisio
    long expirationDate =
      (timeout == INFINITE_TIMEOUT) ? timeout : System.currentTimeMillis()
        + timeout;
    //Cria a requisio de lock
    LockRequest request =
      new LockRequest(ownerKey, future, expirationDate, lockPolicy, object,
        lockDependencies);
    //Insere na fila de requisio
    this.lockRequestQueue.addRequest(request);
  }

  /**
   * Remove o lock de um objeto. Quando o contador de locks para o objeto chegar
   * a zero, remove o info de lock do objeto.
   * <p>
   * O lock s  removido se objeto estiver contido na lista de objetos com
   * locks, e o identificador do lock  igual ao parmetro <code>lockId</code>.
   * <p>
   * 
   * @param lockId
   * @return contagem de referncias para o lock
   */
  public int releaseLock(LockId lockId) {
    //Remove o lock
    return this.releaseLock(lockId, true);
  }

  /**
   * Remove o lock de um objeto. Quando o contador de locks para o objeto chegar
   * a zero, remove o info de lock do objeto.
   * <p>
   * O lock s  removido se objeto estiver contido na lista de objetos com
   * locks, e o identificador do lock  igual ao parmetro <code>lockId</code>.
   * <p>
   * Os observadores s sero informados sobre os releases referentes a este
   * identificador se a flag <code>broadcastRelease</code> fr igual a
   * <code>true</code>
   * 
   * @param lockId identificador do lock
   * @param broadcastRelease flag que indica se o release ser divugado aos
   *        observadores
   * @return contagem de referncias para o lock
   */
  private int releaseLock(LockId lockId, boolean broadcastRelease) {
    if (lockId == null) {
      throw new IllegalArgumentException(
        "Identificador do lock a ser liberado no pode ser nulo.");
    }
    //Recupera objeto do mapa de identificadores e respectivos objetos
    SharedAccessObject objectLocked;
    synchronized (lockIdMap) {
      objectLocked = this.lockIdMap.remove(lockId);
    }
    if (objectLocked == null) {
      throw new InvalidLockIdException(
        "Identificador de lock no est relacionado com nenhum objeto.");
    }
    logger.fine("Solicitando info para remoo de " + objectLocked);
    //Encontra o info de lock do objeto
    LockInfo lockInfo = null;
    synchronized (lockedObjects) {
      lockInfo = lockedObjects.get(objectLocked.getUniqueCode());
      //Verifica se encontrou o info de lock
      if (lockInfo == null) {
        throw new ObjectNotLockedException("No existem locks sobre o objeto "
          + objectLocked);
      }
      //Remove lock do info de lock
      Lock lockRemoved = lockInfo.removeLock(lockId);
      logger.fine("Liberado lock de " + objectLocked);
      if (lockRemoved == null) {
        logger.fine("Lock removido no existe,  igual a null!");
      }
      //Pede para fila analisar requisies
      lockRequestQueue.lockReleased();
      //Notifica os observadores
      if (broadcastRelease) {
        //Adiciona um evento  lista
        this.lockEventQueue.addEvent(new LockEvent(LockEventType.LOCK_RELEASED,
          lockRemoved.getOwnerKey(), lockRemoved.getPolicy(), objectLocked));
      }
      //Remove o lock dos objetos dependentes
      for (LockId idRemove : lockRemoved.getIdDependencies()) {
        this.releaseLock(idRemove, broadcastRelease);
      }
      //Se no existem mais locks sobre o objeto, remove seu info
      if (lockInfo.getLocks().size() <= 0) {
        lockedObjects.remove(objectLocked.getUniqueCode());
        logger.fine("Removido info de lock de " + objectLocked);
        return 0;
      }
    }
    //Retorna o nmero de locks ainda existentes sobre objeto    
    return lockInfo.getLocks().size();
  }

  /**
   * Fora a remoo dos locks de um objeto. S deve ser usado pelo
   * administrador se acontencer alguma situao de ter locks "presos".
   * 
   * @param objectLocked objeto sobre o qual se deseja remover o lock
   */
  public synchronized void forceReleaseLock(SharedAccessObject objectLocked) {
    //Objeto passado como parmetro no pode ser nulo
    if (objectLocked == null) {
      throw new IllegalArgumentException(
        "Objeto a ter o lock foradamente liberado no pode ser nulo [objectLocked == null].");
    }
    //Recupera do mapa de objetos o seu info    
    LockInfo lockInfo;
    synchronized (lockedObjects) {
      lockInfo = lockedObjects.get(objectLocked.getUniqueCode());
    }
    if (lockInfo == null) {
      throw new ObjectNotLockedException("No existem locks sobre o objeto.");
    }
    //Percorrendo lista de locks sobre objeto para liberao "forada"
    for (Lock lock : lockInfo.getLocks()) {
      this.releaseLock(lock.getId(), true);
    }
  }

  /**
   * Verifica se usurio possui determinado tipo de lock de um objeto. Caso o
   * usurio possua um tipo de lock mais alto que o tipo a ser verificado o
   * resultado  verdadeiro. Caso no possua nenhum lock ou possua um lock mais
   * baixo que o tipo a ser verificado, o resultado  falso.
   * 
   * @param policy a poltica a ser verificada
   * @param object o objeto a ser verificado
   * @param ownerKey a identificao do usurio
   * @return se usurio possui determinado tipo de lock de um objeto
   */
  public boolean hasLock(LockPolicy policy, SharedAccessObject object,
    Object ownerKey) {
    //Objetos passados como parmetros no podem ser nulos
    if ((policy == null) || (object == null) || (ownerKey == null)) {
      throw new IllegalArgumentException(
        "Parmetros para verificao de lock no podem ser nulos.");
    }
    //Recupera do mapa de objetos o seu info
    LockInfo lockInfo;
    synchronized (lockedObjects) {
      lockInfo = lockedObjects.get(object.getUniqueCode());
    }
    //Se no foi encontrada info de lock retorna falso
    if (lockInfo == null) {
      return false;
    }
    return lockInfo.hasLock(policy, ownerKey);
  }

  /**
   * Retorna a poltica de lock mais alta que o usurio obtm sobre o objeto, se
   * houver
   * 
   * @param object o objeto a ser verificado
   * @param ownerKey a identificao do usurio
   * @return a poltica deste lock mais alta, caso contrrio retorna nulo
   */
  public LockPolicy hasLock(SharedAccessObject object, Object ownerKey) {
    //Objetos passados como parmetros no podem ser nulos
    if ((object == null) || (ownerKey == null)) {
      throw new IllegalArgumentException(
        "Parmetros para verificao de lock no podem ser nulos.");
    }
    //Recupera do mapa de objetos o seu info
    LockInfo lockInfo;
    synchronized (lockedObjects) {
      lockInfo = lockedObjects.get(object.getUniqueCode());
    }
    //Se no foi encontrada info de lock retorna falso
    if (lockInfo == null) {
      return null;
    }
    return lockInfo.getLockPolicy(ownerKey);
  }

  /**
   * Retorna o conjunto de chaves de usurios detentores dos determinados tipos
   * de locks sobre o objeto passados como parmetro
   * 
   * @param objectToVerify objeto a ser verificado
   * @param policies polticas de lock
   * 
   * @return um mapa com a poltica e o conjunto de chaves de usurios
   *         detentores dos determinados tipos de locks
   */
  public Map<LockPolicy, Set<Object>> getLockOwnersKeys(
    SharedAccessObject objectToVerify, LockPolicy... policies) {
    //Objeto passado como parmetro no pode ser nulos
    if (objectToVerify == null) {
      throw new IllegalArgumentException(
        "Parmetros para verificao de lock no podem ser nulos.");
    }
    //Recupera do mapa de objetos o seu info
    LockInfo lockInfo;
    synchronized (lockedObjects) {
      lockInfo = lockedObjects.get(objectToVerify.getUniqueCode());
    }
    //Se no foi encontrada info de lock retorna falso
    if (lockInfo == null) {
      return null;
    }
    if (policies.length <= 0) {
      return lockInfo.getLockOwners();
    }
    else {
      return lockInfo.getLockOwners(policies);
    }
  }

  /**
   * Listener para observar o estado de todos os locks obtidos e liberados
   * 
   * @param listener observador utilizado para receber eventos
   */
  public void addLockListener(LockListener listener) {
    //Adiciona o listener para o gerenciador
    this.listenerManager.addLockListener(listener);
  }

  /**
   * Listener para observar o estado de todos os locks de um determinado tipo de
   * objeto
   * 
   * @param theClass o tipo do objeto a ser observado
   * @param listener observador utilizado para receber eventos
   */
  public void addLockListener(Class<?> theClass, LockListener listener) {
    //Adiciona o listener para o gerenciador
    this.listenerManager.addLockListener(theClass, listener);
  }

  /**
   * Listener para observar o estado de todos os locks de um determinado objeto
   * 
   * @param object o objeto a ser observado
   * @param listener observador utilizado para receber eventos
   */
  public void addLockListener(Object object, LockListener listener) {
    //Adiciona o listener para o gerenciador
    this.listenerManager.addLockListener(object, listener);
  }

  /**
   * Remove o listener de observao do estado de locks
   * 
   * @param listener observador utilizado para receber eventos
   */
  public void removeLockListener(LockListener listener) {
    //Solicita ao gerenciador que remova o listener
    this.listenerManager.removeLockListener(listener);
  }

  /**
   * Inicia a thread de limpeza de locks invlidos
   */
  private void startInvalidLocksCollectorThread() {
    final Object collector = new Object();
    new Thread(new Runnable() {
      @Override
      public void run() {
        logger.config("Inicia a limpeza de locks invlidos");
        while (!exitManagerThread) {
          //Remove os locks invlidos          
          clearInvalidLocks();
          synchronized (collector) {
            try {
              collector.wait(MAX_SLEEP_INTERVAL);
            }
            catch (InterruptedException e) {
            }
          }
        }
      }

    }).start();
  }

  /**
   * Inicia a thread de processamento das requisies de lock.
   */
  private void startRequestThread() {
    new Thread(new Runnable() {
      @Override
      public void run() {
        logger.config("Inicia processamento de fila de requisies");
        //Fila de requisio deve estar ativa sempre
        while (!exitManagerThread) {
          //Remove os locks invlidos          
          clearInvalidLocks();
          //Processa os pedidos de locks
          lockRequestQueue.processRequests();
          //Intervalo para thread "dormir" 
          long interval = MAX_SLEEP_INTERVAL;
          //Recupera o prximo pedido de lock em termos de tempo solicitado
          LockRequest next = lockRequestQueue.getNextRequest();
          if (next != null) {
            long now = System.currentTimeMillis();
            /*
             * O intervalo  tempo que falta para o prximo comando expirar
             * menos 1 segundo. Fazemos isto para que a fila seja acordada antes
             * do comando expirar.
             */
            long nextTimeout = next.expirationDate - now;
            interval =
              ((nextTimeout < 0) || (MAX_SLEEP_INTERVAL < nextTimeout)) ? MAX_SLEEP_INTERVAL
                : nextTimeout;
          }
          lockRequestQueue.sleep(interval);
        }
      }

    }).start();
  }

  /**
   * Remove da tabela de infos de locks invlidos, ou seja, aqueles que cujos
   * requisitantes (owner) perderam sua referncia fraca. Por exemplo, a sesso
   * do usurio detentor do lock pode ter cado e o lock estar preso.
   */
  private void clearInvalidLocks() {
    logger.fine("Limpa locks invlidos");
    Set<Lock> invalidLocks = new HashSet<Lock>();
    synchronized (lockedObjects) {
      Set<Entry<Object, LockInfo>> locks = lockedObjects.entrySet();
      for (Entry<Object, LockInfo> lockInfoEntry : locks) {
        for (Lock lock : lockInfoEntry.getValue().getLocks()) {
          logger.fine("Verifica lock invlido");
          //Caso lock seja de primeira ordem e esteja invlido, adiciona  lista de locks para remoo
          if (lock.isFirstOrder() && lock.isInvalid()) {
            logger.fine("Lock invlido!");
            invalidLocks.add(lock);
          }
        }
      }
    }
    // Remove os locks a partir da lista de locks invlidos
    for (Lock lock : invalidLocks) {
      logger.fine("Libera lock invlido");
      this.releaseLock(lock.getId(), true);
    }
  }

  /**
   * Inicia a thread de aviso sobre locks.
   */
  private void startEventThread() {
    new Thread(new Runnable() {
      @Override
      public void run() {
        logger.config("Inicia processamento de fila de eventos");
        while (!exitManagerThread) {
          //Processa os eventos de locks
          lockEventQueue.processEvents();
          //Intervalo para thread "dormir" 
          lockEventQueue.sleep();
        }
      }
    }).start();
  }

  /**
   * Interrompe a execuo das threads de requisio e avisos
   */
  private void stopThreads() {
    this.exitManagerThread = true;
  }
}
