package tecgraf.javautils.excel.v1;

import java.util.Date;

/**
 * Interface responsvel por popular o contedo de uma planilha
 * 
 * 
 * @author bbreder
 */
public interface ExcelDataTool extends ExcelTool {

  /**
   * Posiciona o cursor para a celula especificada
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param row linha (1..Short.MAX_VALUE)
   * @return prpria estrutura
   */
  public ExcelDataTool setCell(int column, int row);

  /**
   * Posiciona o cursor para a linha especificada
   * 
   * @param row linha (1..Short.MAX_VALUE)
   * @return prpria estrutura
   */
  public ExcelDataTool setRow(int row);

  /**
   * Posiciona o cursor para a coluna especificada
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @return prpria estrutura
   */
  public ExcelDataTool setColumn(int column);

  /**
   * Atribui o contedo de uma clula no formato de texto.
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param row linha (1..Short.MAX_VALUE)
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCell(int column, int row, String value);

  /**
   * Atribui o contedo de uma clula no formato de numero.
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param row linha (1..Short.MAX_VALUE)
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCell(int column, int row, double value);

  /**
   * Atribui o contedo de uma clula no formato de valor logico.
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param row linha (1..Short.MAX_VALUE)
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCell(int column, int row, boolean value);

  /**
   * Atribui o contedo de uma clula no formato de data.
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param row linha (1..Short.MAX_VALUE)
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCell(int column, int row, Date value);

  /**
   * Atribui o contedo de uma clula em qualquer formato
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param row linha (1..Short.MAX_VALUE)
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCell(int column, int row, Object value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma linha anteriormente,
   * no formato de texto.
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellSameLine(int column, String value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma linha anteriormente,
   * no formato de nmero.
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellSameLine(int column, double value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma linha anteriormente,
   * no formato de valor lgico.
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellSameLine(int column, boolean value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma linha anteriormente,
   * no formato de data.
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellSameLine(int column, Date value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma coluna anteriormente,
   * no formato de texto.
   * 
   * @param row linha (1..Short.MAX_VALUE)
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellSameColumn(int row, String value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma coluna anteriormente,
   * no formato de nmero.
   * 
   * @param row linha (1..Short.MAX_VALUE)
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellSameColumn(int row, double value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma coluna anteriormente,
   * no formato lgico.
   * 
   * @param row linha (1..Short.MAX_VALUE)
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellSameColumn(int row, boolean value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma coluna anteriormente,
   * no formato de data.
   * 
   * @param row linha (1..Short.MAX_VALUE)
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellSameColumn(int row, Date value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma linha anteriormente e
   * mudando a coluna para a prxima clula, no formato de texto.
   * 
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellHorizontal(String value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma linha anteriormente e
   * mudando a coluna para a prxima clula, no formato de nmero.
   * 
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellHorizontal(double value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma linha anteriormente e
   * mudando a coluna para a prxima clula, no formato de valor lgico.
   * 
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellHorizontal(boolean value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma linha anteriormente e
   * mudando a coluna para a prxima clula, no formato de data.
   * 
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellHorizontal(Date value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma linha anteriormente e
   * mudando a coluna para a prxima clula, no formato generico.
   * 
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellHorizontal(Object value);

  /**
   * Atribui o contedo de vrias clulas, utilizando a mesma linha
   * anteriormente e mudando a coluna para a prxima clula, no formato
   * generico.
   * 
   * @param values valores a serem atribuidos
   * @return prpria estrutura
   */
  public ExcelDataTool setCellHorizontal(Object... values);

  /**
   * Atribui o contedo de vrias clulas, utilizando a mesma linha
   * anteriormente e mudando a coluna para a prxima clula, no formato
   * generico.
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param row linha (1..Short.MAX_VALUE)
   * @param values valores a serem atribuidos
   * @return prpria estrutura
   */
  public ExcelDataTool setCellHorizontal(int column, int row, Object... values);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma coluna anteriormente e
   * mudando a linha para a prxima clula, no formato de texto.
   * 
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellVertical(String value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma coluna anteriormente e
   * mudando a linha para a prxima clula, no formato de nmero.
   * 
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellVertical(double value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma coluna anteriormente e
   * mudando a linha para a prxima clula, no formato de valor lgico.
   * 
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellVertical(boolean value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma coluna anteriormente e
   * mudando a linha para a prxima clula, no formato de data.
   * 
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellVertical(Date value);

  /**
   * Atribui o contedo de uma clula, utilizando a mesma coluna anteriormente e
   * mudando a linha para a prxima clula, no formato generico.
   * 
   * @param value valor a ser atribuido
   * @return prpria estrutura
   */
  public ExcelDataTool setCellVertical(Object value);

  /**
   * Atribui o contedo de vrias clulas, utilizando a mesma coluna
   * anteriormente e mudando a linha para a prxima clula, no formato generico.
   * 
   * @param values valores a serem atribuidos
   * @return prpria estrutura
   */
  public ExcelDataTool setCellVertical(Object... values);

  /**
   * Atribui o contedo de vrias clulas, utilizando a mesma coluna
   * anteriormente e mudando a linhas para a prxima clula, no formato
   * generico.
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param row linha (1..Short.MAX_VALUE)
   * @param values valores a serem atribuidos
   * @return prpria estrutura
   */
  public ExcelDataTool setCellVertical(int column, int row, Object... values);

  /**
   * Incrementa uma coluna
   * 
   * @return prpria estrutura
   */
  public ExcelDataTool incColumn();

  /**
   * Incrementa uma linha
   * 
   * @return prpria estrutura
   */
  public ExcelDataTool incRow();

  /**
   * Decrementa uma coluna
   * 
   * @return prpria estrutura
   */
  public ExcelDataTool decColumn();

  /**
   * Decrementa uma linha
   * 
   * @return prpria estrutura
   */
  public ExcelDataTool decRow();

  /**
   * Retorna o formato para data corrente
   * 
   * @return formato para data
   */
  public String getDateFormat();

  /**
   * Atribui um formato para data
   * 
   * @param format
   */
  public void setDateFormat(String format);

  /**
   * Retorna a coluna corrente
   * 
   * @return coluna corrente
   */
  public int getColumn();

  /**
   * Retorna a linha corrente
   * 
   * @return linha corrente
   */
  public int getRow();

  /**
   * Retorna o nmero de casas decimais que um nmero de ponto flutuante ir
   * exibir
   * 
   * @return numero de casas decimais que ser exibido
   */
  public int getNumberPrecision();

  /**
   * Atribui o nmero de casas decimais que um nmero de ponto flutuante ir
   * exibir
   * 
   * @param precision numero de casas decimais que ser exibido
   */
  public void setNumberPrecision(int precision);

  /**
   * Retorna o numero mximo de profundidade de cabealho de linha de uma tabela
   * 
   * @return numero maximo
   */
  public int getMaxRowDeep();

  /**
   * Retorna o numero mximo de profundidade de cabealho de linha de uma tabela
   * 
   * @param maxRowDeep
   */
  public void setMaxRowDeep(int maxRowDeep);

  /**
   * Retorna a ltima ou a maior coluna utilizada na planilha.
   * 
   * @return ltima coluna utilizada
   */
  public int getMaxColumnUsed();

  /**
   * Retorna a ltima ou a maior linha utilizada na planilha.
   * 
   * @return ltima linha utilizada
   */
  public int getMaxRowUsed();

  /**
   * Retorna o contedo da clula
   * 
   * @param col coluna (1..Short.MAX_VALUE)
   * @param row linha (1..Short.MAX_VALUE)
   * @return valor da clula
   */
  public Object getCell(int col, int row);

  /**
   * Retorna o comprimento da clula se ela existir.
   * 
   * @param col coluna (1..Short.MAX_VALUE)
   * @param row linha (1..Short.MAX_VALUE)
   * @return comprimento da clula
   */
  public int getCellWidth(int col, int row);

}
