package tecgraf.javautils.excel.v1;

/**
 * Interface responsvel pelos Merges das celulas
 * 
 * 
 * @author Tecgraf
 */
public interface ExcelStructureTool extends ExcelTool {

  /**
   * Realiza um merge entre celulas delimitado por um intervalo de linhas e
   * colunas.
   * 
   * @param columnBegin inicio da coluna (1..Short.MAX_VALUE) inclusive
   * @param columnEnd fim da coluna (1..Short.MAX_VALUE) inclusive
   * @param rowBegin inicio da linha (1..Short.MAX_VALUE) inclusive
   * @param rowEnd fim da linha (1..Short.MAX_VALUE) inclusive
   * @return prpria estrutura
   */
  public ExcelStructureTool merge(int columnBegin, int columnEnd, int rowBegin,
    int rowEnd);

  /**
   * Realiza um merge entre celulas delimitado por um intervalo de linhas e uma
   * coluna.
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param rowBegin inicio da linha (1..Short.MAX_VALUE) inclusive
   * @param rowEnd fim da linha (1..Short.MAX_VALUE) inclusive
   * @return prpria estrutura
   */
  public ExcelStructureTool mergeRow(int column, int rowBegin, int rowEnd);

  /**
   * Realiza um merge entre celulas delimitado por um intervalo de colunas e uma
   * linha.
   * 
   * @param columnBegin inicio da coluna (1..Short.MAX_VALUE) inclusive
   * @param columnEnd fim da coluna (1..Short.MAX_VALUE) inclusive
   * @param row linha (1..Short.MAX_VALUE)
   * @return prpria estrutura
   */
  public ExcelStructureTool mergeColumn(int columnBegin, int columnEnd, int row);

  /**
   * Realiza um merge entre celulas delimitado pelas linhas utilizadas
   * anteriormente e um conjunto de colunas. Para cada coluna passada como
   * parmetro, ser aplicado um merge com as linhas utilizadas anteriormente
   * com a coluna corrente.
   * 
   * @param columns colunas (1..Short.MAX_VALUE)
   * @return prpria estrutura
   */
  public ExcelStructureTool repeatLineMerge(int... columns);

  /**
   * Realiza um merge entre celulas delimitado pelas colunas utilizadas
   * anteriormente e um conjunto de linhas. Para cada linha passada como
   * parmetro, ser aplicado um merge com as colunas utilizadas anteriormente
   * com a linha corrente.
   * 
   * @param rows linha (1..Short.MAX_VALUE)
   * @return prpria estrutura this.helper = helper;
   */
  public ExcelStructureTool repeatColumnMerge(int... rows);

  /**
   * Atribui uma espessura para a coluna na unidade Excel (in units of 1/256th
   * of a character width)
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param width "in units of 1/256th of a character width"
   * @return prpria estrutura
   */
  public ExcelStructureTool setColumnWidth(int column, int width);

  /**
   * Retorna o comprimento da coluna
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @return comprimento da coluna
   */
  public int getColumnWidth(int column);

  /**
   * Atribui uma altura de uma linha
   * 
   * @param row linha (1..Short.MAX_VALUE)
   * @param width altura a ser atribuda
   * @return prpria estrutura
   */
  public ExcelStructureTool setRowHeight(int row, int width);

  /**
   * Indica a ocorrencia de um merge na celula. Caso no haja, ser retornado
   * nulo.
   * 
   * @param column coluna (1..Short.MAX_VALUE)
   * @param row linha (1..Short.MAX_VALUE)
   * @return estrutura de merge se existir ou nulo
   */
  public ExcelMerge hasMerge(int column, int row);

  /**
   * Retorna o facilitador que constroi sheet.
   * 
   * @return facilitador de sheet
   */
  public ExcelSheet getHelper();

  /**
   * Atribui um intervalo de colunas corrente
   * 
   * @param columnBegin inicio da coluna (1..Short.MAX_VALUE)
   * @param columnEnd fim da coluna (1..Short.MAX_VALUE)
   */
  public void setColumn(int columnBegin, int columnEnd);

  /**
   * Atribui um intervalo de linhas corrente
   * 
   * @param rowBegin inicio da linha (1..Short.MAX_VALUE)
   * @param rowEnd fim da linha (1..Short.MAX_VALUE)
   */
  public void setRow(int rowBegin, int rowEnd);

  /**
   * Retorna a coluna inicial corrente
   * 
   * @return coluna inicial corrente
   */
  public int getColumnBegin();

  /**
   * Atribui uma coluna inicial corrente
   * 
   * @param columnBegin inicio da coluna (1..Short.MAX_VALUE)
   */
  public void setColumnBegin(int columnBegin);

  /**
   * Retorna a coluna final corrente
   * 
   * @return coluna final corrente
   */
  public int getColumnEnd();

  /**
   * Atribui uma coluna final corrente
   * 
   * @param columnEnd fim da coluna (1..Short.MAX_VALUE)
   */
  public void setColumnEnd(int columnEnd);

  /**
   * Retorna a linha inicial corrente
   * 
   * @return linha inicial corrente
   */
  public int getRowBegin();

  /**
   * Atribui uma linha inicial corrente
   * 
   * @param rowBegin inicio da linha (1..Short.MAX_VALUE)
   */
  public void setRowBegin(int rowBegin);

  /**
   * Retorna a linha final corrente
   * 
   * @return linha final corrente
   */
  public int getRowEnd();

  /**
   * Atribui uma linha final corrente
   * 
   * @param rowEnd fim da linha (1..Short.MAX_VALUE)
   */
  public void setRowEnd(int rowEnd);

}
