package tecgraf.javautils.excel.v1.util;

import java.awt.Component;
import java.awt.Point;

import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.JViewport;
import javax.swing.ListModel;
import javax.swing.table.TableModel;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import tecgraf.javautils.excel.v1.ExcelColor;
import tecgraf.javautils.excel.v1.ExcelConfig;
import tecgraf.javautils.excel.v1.ExcelDataTool;
import tecgraf.javautils.excel.v1.ExcelExportable;
import tecgraf.javautils.excel.v1.ExcelExportableAdapter;
import tecgraf.javautils.excel.v1.ExcelStroke;
import tecgraf.javautils.excel.v1.ExcelStructureTool;
import tecgraf.javautils.excel.v1.ExcelStyle;
import tecgraf.javautils.excel.v1.ExcelStyleTool;
import tecgraf.javautils.excel.v1.ExcelTable;
import tecgraf.javautils.excel.v1.style.DefaultExcelStyleSet;

/**
 * Facilitadores de Excel
 * 
 * 
 * @author bbreder
 */
public abstract class ExcelTableUtil {

  /**
   * Atribui uma tabela a uma planilha excel
   * 
   * @param head
   * @param data
   * @param style
   * @param column
   * @param row
   * @param table
   */
  public static void setTable(ExcelStructureTool head, ExcelDataTool data,
    ExcelStyleTool style, int column, int row, JTable table) {
    setTable(head, data, style, column, row, table, null);
  }

  /**
   * Atribui uma tabela a uma planilha excel
   * 
   * @param head
   * @param data
   * @param style
   * @param column
   * @param row
   * @param table
   * @param format
   */
  public static void setTable(ExcelStructureTool head, ExcelDataTool data,
    ExcelStyleTool style, int column, int row, JTable table, ExcelTable format) {
    setTable(head, data, style, column, row, table, format, false);
  }

  /**
   * Atribui uma tabela a uma planilha excel
   * 
   * @param head
   * @param data
   * @param style
   * @param column
   * @param row
   * @param table
   * @param format
   * @param firstIsRowHeader
   */
  public static void setTable(ExcelStructureTool head, ExcelDataTool data,
    ExcelStyleTool style, int column, int row, JTable table, ExcelTable format,
    boolean firstIsRowHeader) {
    if (format == null) {
      throw new IllegalArgumentException("formatador de tabela  nulo");
    }
    // Escreve o cabealho de linha
    data.setCell(column, row);
    // Modelo da tabela
    TableModel model = table.getModel();
    // Recupera o scroll da tabela
    JScrollPane scroll = getScroll(table);
    if (scroll instanceof ExcelExportable) {
      ExcelExportable excelExportable = (ExcelExportable) scroll;
      excelExportable.exportExcel(head, data, style, format);
    }
    else {
      // Calcula aonde ir comear o contedo da tabela
      Point dataPoint = getDataPoint(table, head, data, style);
      // Indica aonde termina o cabealho de linha
      int dataRow = row + dataPoint.y;
      // Indica aonde termina o cabealho de coluna
      int dataColumn = column + dataPoint.x;
      if (scroll != null) {
        // Exibe o Corner
        showCorner(head, data, style, table, row, column, dataRow, dataColumn,
          format);
        // Se existir um cabealho de coluna
        showAdvanceColumnHeader(head, data, style, table, row, column, dataRow,
          dataColumn, format);
        // Se existir um cabealho de linha
        showRowHeader(head, data, style, table, dataRow, column, format);
      }
      else {
        showBasicColumnHeader(head, data, style, table, dataRow, dataColumn,
          format);
      }
      exportData(data, style, table, format, firstIsRowHeader, dataRow,
        dataColumn);
      if (firstIsRowHeader) {
        style.addBoxRow(dataColumn, dataRow, dataRow + model.getRowCount() - 1,
          ExcelStroke.MEDIUM, ExcelColor.BLACK);
      }
      if (model.getRowCount() > 0) {
        data.decColumn();
      }
      style.addBox(column, data.getColumn(), row, data.getRow(),
        ExcelStroke.MEDIUM, ExcelColor.BLACK);
      data.incRow();
      data.setColumn(column);
    }
  }

  /**
   * Exporta o contedo da tabela
   * 
   * @param data
   * @param style
   * @param table
   * @param format
   * @param firstIsRowHeader
   * @param dataRow
   * @param dataColumn
   */
  public static void exportData(ExcelDataTool data, ExcelStyleTool style,
    JTable table, ExcelTable format, boolean firstIsRowHeader, int dataRow,
    int dataColumn) {
    // Escreve o contedo da tabela
    for (int r = 0; r < table.getRowCount(); r++) {
      data.setCell(dataColumn, dataRow + r);
      for (int c = 0; c < table.getColumnCount(); c++) {
        Object valueAt = table.getValueAt(r, c);
        // Se a 1a coluna for cabealho
        if (firstIsRowHeader && c == 0) {
          style.setStyle(dataColumn, dataRow + r, DefaultExcelStyleSet
            .buildRowHeader(style));
        }
        if (format != null) {
          valueAt = format.getValue(table, r, c, valueAt);
          ExcelStyle cellStyle = format.getStyle(table, r, c, style);
          if (cellStyle != null) {
            style.setStyle(dataColumn + c, dataRow + r, cellStyle);
          }
        }
        else {
          valueAt = fixTableValue(table, r, c, valueAt);
        }
        data.setCellHorizontal(valueAt);
      }
    }
  }

  /**
   * Modifica o valor de uma celula da tabela
   * 
   * @param table
   * @param row
   * @param column
   * @param value
   * @return valor modificado
   */
  private static Object fixTableValue(JTable table, int row, int column,
    Object value) {
    //    if (value instanceof Date) { TODO lbarros
    Component comp =
      table.getCellRenderer(row, column).getTableCellRendererComponent(table,
        value, false, false, row, column);
    if (comp instanceof JLabel) {
      value = ((JLabel) comp).getText();
    }
    //    }
    return value;
  }

  /**
   * Imprime o cabealho de linha
   * 
   * @param head
   * @param data
   * @param style
   * @param table
   * @param row
   * @param column
   * @param format
   */
  public static void showRowHeader(ExcelStructureTool head, ExcelDataTool data,
    ExcelStyleTool style, JTable table, int row, int column, ExcelTable format) {
    // Recupera o scroll da tabela
    JScrollPane scroll = getScroll(table);
    Component corner = getCorner(scroll);
    data.setCell(column, row);
    if (corner != null) {
      data.incRow();
    }
    // Indica que existe cabealho de linha 
    boolean hasRowHeader =
      scroll.getRowHeader() != null && scroll.getRowHeader().getView() != null;
    // Componente de cabealho de linha
    Component rowHeader = hasRowHeader ? scroll.getRowHeader().getView() : null;
    if (hasRowHeader) {
      // Trata o cabealho de linha no modo JList
      if (JList.class.isInstance(rowHeader)) {
        JList list = (JList) rowHeader;
        ListModel rowModel = list.getModel();
        for (int r = 0; r < rowModel.getSize(); r++) {
          Object valueAt = rowModel.getElementAt(r);
          ExcelStyle cellStyle = null;
          if (format != null) {
            valueAt = format.getRowHeaderValue(table, r, valueAt);
            cellStyle = format.getRowHeaderStyle(table, row + r, style);
          }
          if (cellStyle == null) {
            cellStyle = DefaultExcelStyleSet.buildRowHeader(style);
          }
          style.setStyle(data.getColumn(), data.getRow(), cellStyle);
          data.setCellVertical(valueAt);
        }
        style.addBox(column, column, row, row + rowModel.getSize() - 1,
          ExcelStroke.MEDIUM, ExcelColor.BLACK);
      }
      // Tratar o cabealho de linha no modo JTree
      else if (JTree.class.isInstance(rowHeader)) {
        JTree tree = (JTree) rowHeader;
        TreeModel rowModel = tree.getModel();
        if (TreeNode.class.isInstance(rowModel.getRoot())) {
          TreeNode node = (TreeNode) rowModel.getRoot();
          setTable(head, data, style, table, tree, node, 0, format);
          style.addBox(column, column, row, data.getRow() - 1,
            ExcelStroke.MEDIUM, ExcelColor.BLACK);
          style.addBox(column, column + table.getColumnCount(), row + 1, data
            .getRow() - 1, ExcelStroke.MEDIUM, ExcelColor.BLACK);
        }
      }
    }
  }

  /**
   * Imprime o cabealho de coluna no modo avanado
   * 
   * @param head
   * @param data
   * @param style
   * @param table
   * @param row
   * @param column
   * @param dataRow
   * @param dataColumn
   * @param format
   */
  private static void showAdvanceColumnHeader(ExcelStructureTool head,
    ExcelDataTool data, ExcelStyleTool style, JTable table, int row,
    int column, int dataRow, int dataColumn, ExcelTable format) {
    JScrollPane scroll = getScroll(table);
    Component columnHeader =
      scroll.getColumnHeader() != null ? scroll.getColumnHeader().getView()
        : null;
    ExcelExportableAdapter exportableAdapter =
      ExcelConfig.getExportable(columnHeader);
    if (columnHeader instanceof ExcelExportable) {
      ExcelExportable excelExportable = (ExcelExportable) columnHeader;
      data.setCell(column, row);
      excelExportable.exportExcel(head, data, style, format);
    }
    else if (exportableAdapter != null) {
      data.setCell(column, row);
      exportableAdapter.exportExcel(columnHeader, head, data, style, format);
    }
    else {
      showBasicColumnHeader(head, data, style, table, row, dataColumn, format);
    }
  }

  /**
   * Imprime o cabealho de linha no modo basico
   * 
   * @param head
   * @param data
   * @param style
   * @param table
   * @param row
   * @param column
   * @param format
   */
  private static void showBasicColumnHeader(ExcelStructureTool head,
    ExcelDataTool data, ExcelStyleTool style, JTable table, int row,
    int column, ExcelTable format) {
    // Modelo da tabela
    TableModel model = table.getModel();
    // Escreve o cabealho de coluna    
    data.setCell(column, row);

    for (int c = 0; c < model.getColumnCount(); c++) {
      String valueAt = model.getColumnName(c).replace("\n", " ");
      if (format != null) {
        valueAt = format.getColumnHeaderValue(table, c, valueAt).toString();
        ExcelStyle cellStyle =
          format.getColumnHeaderStyle(table, column + c, style);
        if (cellStyle != null) {
          style.setStyle(column + c, row, cellStyle);
        }
        else {
          style.setStyle(column + c, row, DefaultExcelStyleSet
            .buildColumnHeader(style));
        }
      }
      else {
        style.setStyle(column + c, row, DefaultExcelStyleSet
          .buildColumnHeader(style));
      }
      data.setCell(column + c, row, valueAt);
    }
    style.addBoxColumn(column, column + model.getColumnCount() - 1, row,
      ExcelStroke.MEDIUM, ExcelColor.BLACK);
  }

  /**
   * Retorna o quanto de linhas e colunas o cabealho da tabela est ocupando
   * 
   * @param table
   * @param data
   * @param head
   * @param style
   * @return ponto
   */
  private static Point getDataPoint(JTable table, ExcelStructureTool head,
    ExcelDataTool data, ExcelStyleTool style) {
    int headerRow = 0;
    int headerColumn = 0;
    // Recupera o scroll da tabela
    JScrollPane scroll = getScroll(table);
    if (scroll == null && table.getColumnCount() > 0) {
      headerRow = 1;
    }
    else {
      // Indica que existe cabealho de linha 
      boolean hasRowHeader =
        scroll.getRowHeader() != null
          && scroll.getRowHeader().getView() != null;
      // Componente de cabealho de linha
      Component rowHeader =
        hasRowHeader ? scroll.getRowHeader().getView() : null;
      // Indica que existe cabealho de coluna
      boolean hasColumnHeader =
        scroll.getColumnHeader() != null
          && scroll.getColumnHeader().getView() != null;
      // Componente de cabealho de coluna
      Component columnHeader = scroll.getColumnHeader().getView();
      {
        // Etapa que calcula a variavel headerRow e headerColumn
        if (scroll != null) {
          headerRow = 1;
          if (hasCorner(scroll)) {
            // Se tiver corner, entao pelo menos anda 1 para linha e coluna
            headerColumn = 1;
            headerRow = 1;
          }
          if (hasRowHeader) {
            if (JList.class.isInstance(rowHeader)) {
              headerColumn = 1;
            }
            // Tratar o cabealho de linha no modo JTree
            else if (JTree.class.isInstance(rowHeader)) {
              //              JTree tree = (JTree) rowHeader;
              //              TreeNode node = (TreeNode) tree.getModel().getRoot();
              //TODO lbarros remover?
              //              getTreeNodeDeep(head, data, style, tree, node);
              headerColumn = 1;
            }
          }
          if (hasColumnHeader) {
            // TODO bbredertratar
            //            if (GroupableTableHeader.class.isInstance(columnHeader)) {
            //              GroupableTableHeader groupableHeader =
            //                (GroupableTableHeader) columnHeader;
            //              headerRow = 1;//TODO lbarros recuperar a altura atual (em linhas) do cabealho
            //              headerRow = 2;
            //            }
          }
          else {
            headerRow = 1;
          }
        }
      }
    }
    return new Point(headerColumn, headerRow);
  }

  /**
   * Funo responsvel por exibir o corner da tabela
   * 
   * @param head
   * @param data
   * @param style
   * @param table
   * @param row
   * @param column
   * @param dataColumn
   * @param dataRow
   * @param format
   */
  private static void showCorner(ExcelStructureTool head, ExcelDataTool data,
    ExcelStyleTool style, JTable table, int row, int column, int dataRow,
    int dataColumn, ExcelTable format) {
    // Recupera o scroll da tabela
    JScrollPane scroll = getScroll(table);
    Component corner = scroll.getCorner(JScrollPane.UPPER_LEFT_CORNER);
    style.setStyle(column, row, DefaultExcelStyleSet.buildColumnHeader(style));
    style.addBox(column, row, ExcelStroke.MEDIUM, ExcelColor.BLACK);
    if (corner != null && format != null) {
      Object value = format.getCorner(corner);
      if (value != null) {
        data.setCell(column, row, value.toString());
      }
    }
    if (dataRow - row > 1 || dataColumn - column > 1) {
      head.merge(column, dataColumn, row, dataRow - 1);
    }
  }

  /**
   * Funo responsvel por exibir o corner da tabela
   * 
   * @param scroll
   * @return tem corner
   */
  private static boolean hasCorner(JScrollPane scroll) {
    Component corner = scroll.getCorner(JScrollPane.UPPER_LEFT_CORNER);
    //TODO lbarros restaurar
    //    if (corner != null && Container.class.isInstance(corner)) {
    //      Container container = (Container) corner;
    //      if (container.getComponentCount() > 0) {
    //        Component cornerChild = ((Container) corner).getComponent(0);
    //        if (cornerChild != null && JTextComponent.class.isInstance(cornerChild)) {
    //          return true;
    //        }
    //      }
    //    }
    //    return false;
    return corner != null;
  }

  /**
   * Escreve os ns de uma arvore, no cabealho de linha.
   * 
   * @param head
   * @param data
   * @param style
   * @param tree
   * @param node
   * @param deep
   * @param format
   */
  private static void setTable(ExcelStructureTool head, ExcelDataTool data,
    ExcelStyleTool style, JTable table, JTree tree, TreeNode node, int deep,
    ExcelTable format) {

    for (int r = 0; r < node.getChildCount(); r++) {
      TreeNode child = node.getChildAt(r);
      String text = child.toString().replace('\n', ' ');
      if (format != null) {
        text = format.getRowHeaderValue(table, r, text).toString();
      }
      for (int n = 0; n < deep; n++) {
        text = "     " + text;
      }
      data.setCellVertical(text);
      int row = data.getRow() - 1;
      style.setStyle(data.getColumn(), row, DefaultExcelStyleSet
        .buildRowHeader(style));
      if (format != null) {
        ExcelStyle cellStyle = format.getRowHeaderStyle(table, row, style);
        if (cellStyle != null) {
          style.setStyle(data.getColumn(), row, cellStyle);
        }
        else {
          style.setStyle(data.getColumn(), row, DefaultExcelStyleSet
            .buildRowHeader(style));
        }
      }
      else {
        style.setStyle(data.getColumn(), row, DefaultExcelStyleSet
          .buildRowHeader(style));
      }
      TreePath path = buildTreePath(child);
      if (child.getChildCount() > 0 && tree.isExpanded(path)
        && deep < data.getMaxRowDeep()) {
        data.incColumn();
        data.decColumn();
        setTable(head, data, style, table, tree, child, deep + 1, format);
      }
    }
  }

  /**
   * Constroi o TreePath correspondente ao n
   * 
   * @param child
   * @return TreePath do n
   */
  private static TreePath buildTreePath(TreeNode child) {
    if (child.getParent() != null) {
      TreePath parent = buildTreePath(child.getParent());
      int size = parent.getPath().length;
      Object[] paths = new Object[size + 1];
      System.arraycopy(parent.getPath(), 0, paths, 0, size);
      paths[size] = child;
      return new TreePath(paths);
    }
    else {
      return new TreePath(child);
    }
  }

  /**
   * Escreve os ns de uma arvore, no cabealho de linha.
   * 
   * @param head
   * @param data
   * @param style
   * @param tree
   * @param node
   * @return profundidade de uma arvore
   */
  private static int getTreeNodeDeep(ExcelStructureTool head,
    ExcelDataTool data, ExcelStyleTool style, JTree tree, TreeNode node) {
    int count = 1;
    for (int r = 0; r < node.getChildCount(); r++) {
      TreeNode child = node.getChildAt(r);
      TreePath path = buildTreePath(child);
      if (child.getChildCount() > 0 && tree.isExpanded(path)) {
        count = getTreeNodeDeep(head, data, style, tree, child) + 1;
      }
    }
    return Math.min(count, data.getMaxRowDeep());
  }

  /**
   * Retorna o scroll da tabela
   * 
   * @param table
   * @return scroll ou nulo
   */
  private static JScrollPane getScroll(JTable table) {
    if (table.getParent() != null
      && JViewport.class.isInstance(table.getParent())
      && table.getParent().getParent() != null
      && JScrollPane.class.isInstance(table.getParent().getParent())) {
      return (JScrollPane) table.getParent().getParent();
    }
    else {
      return null;
    }
  }

  /**
   * Retorna o Corner do ScrollPane
   * 
   * @param scroll
   * @return corner
   */
  public static Component getCorner(JScrollPane scroll) {
    return scroll.getCorner(JScrollPane.UPPER_LEFT_CORNER);
  }

}
