package tecgraf.javautils.excel.v1.poi;

import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;

import tecgraf.javautils.excel.v1.ExcelColor;
import tecgraf.javautils.excel.v1.ExcelStroke;
import tecgraf.javautils.excel.v1.ExcelStyle;

/**
 * Classe responsvel por criar um estilo para ser aplicado em celulas de uma
 * planilha.
 *
 *
 * @author bbreder
 */
public class PoiExcelStyle implements ExcelStyle, Cloneable {

  /** Referencia para a planilha excel */
  private HSSFWorkbook workbook;
  /** Referencia para o estilo excel */
  private HSSFCellStyle style;
  /** Referencia para a fonte do estilo */
  private HSSFFont font;
  /** Alinhamento */
  private Short align;
  /** Negrito */
  private boolean bold;
  /** Italico */
  private boolean italic;
  /** Sublinhado */
  private boolean strikeout;
  /** Cor do texto */
  private Short forecolor;
  /** Cor do fundo */
  private Short backcolor;
  /** Tamanho do texto */
  private Short size;
  /** Espessura da Borda */
  private ExcelStroke borderTopStroke;
  /** Espessura da Borda */
  private ExcelStroke borderBottomStroke;
  /** Espessura da Borda */
  private ExcelStroke borderLeftStroke;
  /** Espessura da Borda */
  private ExcelStroke borderRightStroke;
  /** Cor da Borda */
  private ExcelColor borderTopColor;
  /** Cor da Borda */
  private ExcelColor borderBottomColor;
  /** Cor da Borda */
  private ExcelColor borderLeftColor;
  /** Cor da Borda */
  private ExcelColor borderRightColor;
  /** Cache do Adpter */
  private Object adapter;

  /**
   * Construtor padro
   *
   * @param workbook
   */
  public PoiExcelStyle(HSSFWorkbook workbook) {
    this.workbook = workbook;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyle left() {
    this.align = HSSFCellStyle.ALIGN_LEFT;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyle center() {
    this.align = HSSFCellStyle.ALIGN_CENTER;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyle right() {
    this.align = HSSFCellStyle.ALIGN_RIGHT;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcelStyle bold() {
    this.bold = true;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcelStyle italic() {
    this.italic = true;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcelStyle strikeout() {
    this.strikeout = true;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcelStyle forecolor(ExcelColor color) {
    short scolor = PoiExcelColor.get(color);
    this.forecolor = scolor;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcelStyle backcolor(ExcelColor color) {
    short scolor = PoiExcelColor.get(color);
    this.backcolor = scolor;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcelStyle size(int height) {
    short sheight = (short) Math.min(Short.MAX_VALUE, height);
    this.size = sheight;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyle borderTop(ExcelStroke stroke, ExcelColor color) {
    this.borderTopStroke = stroke;
    this.borderTopColor = color;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyle borderBottom(ExcelStroke stroke, ExcelColor color) {
    this.borderBottomStroke = stroke;
    this.borderBottomColor = color;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyle borderLeft(ExcelStroke stroke, ExcelColor color) {
    this.borderLeftStroke = stroke;
    this.borderLeftColor = color;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExcelStyle borderRight(ExcelStroke stroke, ExcelColor color) {
    this.borderRightStroke = stroke;
    this.borderRightColor = color;
    return this;
  }

  /**
   * Retorna o estilo criado
   *
   * @return estilo criado
   */
  protected HSSFCellStyle getStyle() {
    if (this.style == null) {
      this.style = this.getWorkbook().createCellStyle();
    }
    return style;
  }

  /**
   * Retorna a fonte
   *
   * @return prpria estrutura
   */
  protected HSSFFont getFont() {
    if (this.font == null) {
      this.font = this.getWorkbook().createFont();
      this.getStyle().setFont(this.font);
    }
    return font;
  }

  /**
   * Cria um adpter novo
   *
   * @return novo adpter
   */
  public Object buildAdapter() {
    HSSFCellStyle style = this.getStyle();
    HSSFFont font = this.getFont();
    style.setVerticalAlignment(HSSFCellStyle.VERTICAL_CENTER);
    if (this.align != null) {
      style.setAlignment(this.align);
    }
    if (this.bold) {
      font.setBoldweight(HSSFFont.BOLDWEIGHT_BOLD);
    }
    if (this.italic) {
      font.setItalic(true);
    }
    if (this.strikeout) {
      font.setStrikeout(true);
    }
    if (this.forecolor != null) {
      font.setColor(this.forecolor);
      style.setFillForegroundColor(this.forecolor);
    }
    if (this.backcolor != null) {
      style.setFillForegroundColor(this.backcolor);
      style.setFillPattern(HSSFCellStyle.SOLID_FOREGROUND);
    }
    if (this.size != null) {
      font.setFontHeight((short) Math.min(this.size * 20, Short.MAX_VALUE));
    }
    if (this.borderTopStroke != null) {
      style.setBorderTop(PoiExcelStroke.get(this.borderTopStroke));
      style.setTopBorderColor(PoiExcelColor.get(this.borderTopColor));
    }
    if (this.borderBottomStroke != null) {
      style.setBorderBottom(PoiExcelStroke.get(this.borderBottomStroke));
      style.setBottomBorderColor(PoiExcelColor.get(this.borderBottomColor));
    }
    if (this.borderLeftStroke != null) {
      style.setBorderLeft(PoiExcelStroke.get(this.borderLeftStroke));
      style.setLeftBorderColor(PoiExcelColor.get(this.borderLeftColor));
    }
    if (this.borderRightStroke != null) {
      style.setBorderRight(PoiExcelStroke.get(this.borderRightStroke));
      style.setRightBorderColor(PoiExcelColor.get(this.borderRightColor));
    }
    this.adapter = style;
    return style;
  }

  /**
   * Retorna a planilha excel
   *
   * @return planilha excel
   */
  protected HSSFWorkbook getWorkbook() {
    return workbook;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getAdapter() {
    if (this.adapter == null) {
      this.adapter = this.buildAdapter();
    }
    return this.adapter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PoiExcelStyle clone() {
    try {
      PoiExcelStyle clone = (PoiExcelStyle) super.clone();
      clone.style = null;
      clone.font = null;
      return clone;
    }
    catch (CloneNotSupportedException e) {
      e.printStackTrace();
      return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((align == null) ? 0 : align.hashCode());
    result = prime * result + ((backcolor == null) ? 0 : backcolor.hashCode());
    result = prime * result + (bold ? 1231 : 1237);
    result =
      prime * result
      + ((borderBottomColor == null) ? 0 : borderBottomColor.hashCode());
    result =
      prime * result
      + ((borderBottomStroke == null) ? 0 : borderBottomStroke.hashCode());
    result =
      prime * result
      + ((borderLeftColor == null) ? 0 : borderLeftColor.hashCode());
    result =
      prime * result
      + ((borderLeftStroke == null) ? 0 : borderLeftStroke.hashCode());
    result =
      prime * result
      + ((borderRightColor == null) ? 0 : borderRightColor.hashCode());
    result =
      prime * result
      + ((borderRightStroke == null) ? 0 : borderRightStroke.hashCode());
    result =
      prime * result
      + ((borderTopColor == null) ? 0 : borderTopColor.hashCode());
    result =
      prime * result
      + ((borderTopStroke == null) ? 0 : borderTopStroke.hashCode());
    result = prime * result + ((forecolor == null) ? 0 : forecolor.hashCode());
    result = prime * result + (italic ? 1231 : 1237);
    result = prime * result + ((size == null) ? 0 : size.hashCode());
    result = prime * result + (strikeout ? 1231 : 1237);
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    PoiExcelStyle other = (PoiExcelStyle) obj;
    if (align == null) {
      if (other.align != null) {
        return false;
      }
    }
    else if (!align.equals(other.align)) {
      return false;
    }
    if (backcolor == null) {
      if (other.backcolor != null) {
        return false;
      }
    }
    else if (!backcolor.equals(other.backcolor)) {
      return false;
    }
    if (bold != other.bold) {
      return false;
    }
    if (borderBottomColor != other.borderBottomColor) {
      return false;
    }
    if (borderBottomStroke != other.borderBottomStroke) {
      return false;
    }
    if (borderLeftColor != other.borderLeftColor) {
      return false;
    }
    if (borderLeftStroke != other.borderLeftStroke) {
      return false;
    }
    if (borderRightColor != other.borderRightColor) {
      return false;
    }
    if (borderRightStroke != other.borderRightStroke) {
      return false;
    }
    if (borderTopColor != other.borderTopColor) {
      return false;
    }
    if (borderTopStroke != other.borderTopStroke) {
      return false;
    }
    if (forecolor == null) {
      if (other.forecolor != null) {
        return false;
      }
    }
    else if (!forecolor.equals(other.forecolor)) {
      return false;
    }
    if (italic != other.italic) {
      return false;
    }
    if (size == null) {
      if (other.size != null) {
        return false;
      }
    }
    else if (!size.equals(other.size)) {
      return false;
    }
    if (strikeout != other.strikeout) {
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return "PoiExcelStyle [align=" + align + ", bold=" + bold + ", italic="
      + italic + ", strikeout=" + strikeout + ", forecolor=" + forecolor
      + ", backcolor=" + backcolor + ", size=" + size + ", borderTopStroke="
      + borderTopStroke + ", borderBottomStroke=" + borderBottomStroke
      + ", borderLeftStroke=" + borderLeftStroke + ", borderRightStroke="
      + borderRightStroke + ", borderTopColor=" + borderTopColor
      + ", borderBottomColor=" + borderBottomColor + ", borderLeftColor="
      + borderLeftColor + ", borderRightColor=" + borderRightColor + "]";
  }

}
