package tecgraf.javautils.excel.v1.util;

import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.JTable;

import tecgraf.javautils.excel.v1.AbstractExcelModel;
import tecgraf.javautils.excel.v1.ExcelColor;
import tecgraf.javautils.excel.v1.ExcelDataTool;
import tecgraf.javautils.excel.v1.ExcelExportable;
import tecgraf.javautils.excel.v1.ExcelMerge;
import tecgraf.javautils.excel.v1.ExcelStroke;
import tecgraf.javautils.excel.v1.ExcelStructureTool;
import tecgraf.javautils.excel.v1.ExcelStyleTool;
import tecgraf.javautils.excel.v1.ExcelTable;
import tecgraf.javautils.excel.v1.poi.PoiExcel;
import tecgraf.javautils.excel.v1.style.DefaultExcelStyleSet;

/**
 * Classe utilitaria para gerar relatorio do tipo bandeirabr
 *
 *
 * @author bbreder
 */
public abstract class VerticalExcelModel extends AbstractExcelModel {

  /** Lista de itens */
  private List<AbstractItem> list = new ArrayList<AbstractItem>();
  /** Preciso do float */
  private int precision = 1;

  /**
   * Mtodo que ir calcular as operaes para gerar o relatorio
   */
  public abstract void preBuild();

  /**
   * {@inheritDoc}
   */
  @Override
  public void build(ExcelStructureTool head, ExcelDataTool data,
    ExcelStyleTool style) {
    this.preBuild();
    data.setNumberPrecision(precision);
    // Executa 2 vezes para primeiro calcular o nmero de colunas totais e depois realmente imprimir
    for (int n = 0; n < 2; n++) {
      data.setCell(1, 1);
      int row = 1;
      for (AbstractItem item : list) {
        item.build(head, data, style, n == 0);
        item.setRowHeight(data.getRow() - row);
        row = data.getRow();
      }
    }
    { // Aplicando as bordas
      Integer index = null;
      int row = 1;
      for (int n = 0; n < list.size(); n++) {
        AbstractItem item = list.get(n);
        if (HeaderItem.class.isInstance(item) && index == null) {
          index = row;
        }
        if (!HeaderItem.class.isInstance(item) && index != null) {
          int columnBegin = 1;
          int columnEnd = data.getMaxColumnUsed();
          int rowBegin = index;
          int rowEnd = row - item.getRowHeight();
          style.addBox(columnBegin, columnEnd, rowBegin, rowEnd,
            ExcelStroke.MEDIUM, ExcelColor.BLACK);
          index = null;
        }
        row += item.getRowHeight();
      }
    }
    { // Aplicando Merge de Cabealho
      // Subdivido em 2 etapas
      {
        // Calcula as celulas que no possuem merges
        for (int col = 1; col <= data.getMaxColumnUsed(); col++) {
          int width = 0;
          for (int row = 1; row <= data.getMaxRowUsed(); row++) {
            ExcelMerge merge = head.hasMerge(col, row);
            if (merge == null) {
              width = Math.max(width, data.getCellWidth(col, row));
            }
          }
          head.setColumnWidth(col, width);
        }
      }
      {
        // Aumenta o tamanho das celulas que possui merge e esto apertadas
        Set<ExcelMerge> mergeUsedSet = new HashSet<ExcelMerge>();
        for (int col = 1; col <= data.getMaxColumnUsed(); col++) {
          for (int row = 1; row <= data.getMaxRowUsed(); row++) {
            ExcelMerge merge = head.hasMerge(col, row);
            if (merge != null && !mergeUsedSet.contains(merge)) {
              mergeUsedSet.add(merge);
              int dataWidth = data.getCellWidth(col, row);
              int headWidth = 0;
              int beginColumn = merge.getColumnBeginIndex();
              int endColumn = merge.getColumnEndIndex();
              for (int n = beginColumn; n <= endColumn; n++) {
                headWidth += head.getColumnWidth(n);
              }
              int columns = endColumn - beginColumn;
              if (dataWidth > headWidth) {
                int delta = (dataWidth - headWidth) / (columns + 1);
                for (int n = beginColumn; n <= endColumn; n++) {
                  head.setColumnWidth(n, head.getColumnWidth(n) + delta);
                }
              }
            }
          }
        }
      }
    }
  }

  /**
   * Adiciona um contedo do tipo cabealho
   *
   * @param text
   */
  public void addHeader(String text) {
    this.list.add(new HeaderItem(text));
  }

  /**
   * Adiciona uma tabela
   *
   * @param table
   */
  public void addTable(JTable table) {
    this.addTable(table, new DefaultExcelTable());
  }

  /**
   * Adiciona uma tabela
   *
   * @param table
   * @param format
   */
  public void addTable(JTable table, ExcelTable format) {
    this.list.add(new TableItem(table, format, false));
  }

  /**
   * Adiciona uma tabela
   *
   * @param table
   * @param format
   * @param firstIsRowHeader
   */
  public void addTable(JTable table, ExcelTable format, boolean firstIsRowHeader) {
    this.list.add(new TableItem(table, format, firstIsRowHeader));
  }

  /**
   * Adiciona uma linha
   */
  public void addLine() {
    this.list.add(new LineItem());
  }

  /**
   * Adiciona o item de reservado
   */
  public void addReserved() {
    this.list.add(new ReservedItem());
  }

  /**
   * Adiciona o item
   *
   * @param item
   */
  public void addItem(AbstractItem item) {
    this.list.add(item);
  }

  /**
   * @param exportable Exportvel para Excel
   * @param format Leitor de Estrutura de Dados
   */
  public void addExportable(ExcelExportable exportable, ExcelTable format) {
    list.add(new ExportableItem(exportable, format));
  }

  /**
   * Atribui a preciso do ponto flutuante
   *
   * @param value
   */
  public void setPrecision(int value) {
    this.precision = value;
  }

  /**
   * Item de linha
   *
   *
   * @author bbreder
   */
  private class LineItem extends AbstractItem {

    /**
     * {@inheritDoc}
     */
    @Override
    public void build(ExcelStructureTool head, ExcelDataTool data,
      ExcelStyleTool style, boolean testing) {
      data.incRow();
    }

  }

  /**
   * Item de cabealho
   *
   *
   * @author Tecgraf
   */
  private class HeaderItem extends AbstractItem {

    /** Text */
    private String text;

    /**
     * Construtor padro
     *
     * @param text
     */
    public HeaderItem(String text) {
      this.text = text;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void build(ExcelStructureTool head, ExcelDataTool data,
      ExcelStyleTool style, boolean testing) {
      if (!testing) {
        head.mergeColumn(1, data.getMaxColumnUsed(), data.getRow());
        style.setStyleColumn(1, data.getMaxColumnUsed(), data.getRow(),
          DefaultExcelStyleSet.buildTopHeader(style));
      }
      data.setCellVertical(text);
    }

  }

  /**
   * Item de tabela
   *
   *
   * @author bbreder
   */
  private class TableItem extends AbstractItem {

    /** Tabela */
    private JTable table;
    /** Leitor de Estrutura de Dados */
    private ExcelTable format;
    /** Indica que a 1a coluna  cabealho de linha */
    private boolean firstIsRowHeader;

    /**
     * Construtor padro
     *
     * @param table
     * @param getValue
     * @param firstIsRowHeader
     */
    public TableItem(JTable table, ExcelTable getValue, boolean firstIsRowHeader) {
      this.table = table;
      this.format = getValue;
      this.firstIsRowHeader = firstIsRowHeader;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void build(ExcelStructureTool head, ExcelDataTool data,
      ExcelStyleTool style, boolean testing) {
      ExcelTableUtil.setTable(head, data, style, data.getColumn(), data
        .getRow(), table, format, firstIsRowHeader);
    }

  }

  /**
   * Item de revervado
   *
   *
   * @author bbreder
   */
  private class ReservedItem extends AbstractItem {

    /**
     * {@inheritDoc}
     */
    @Override
    public void build(ExcelStructureTool head, ExcelDataTool data,
      ExcelStyleTool style, boolean testing) {
      data.setCell(data.getColumn(), data.getRow(), "RESERVADA");
      if (!testing) {
        head.mergeColumn(1, data.getMaxColumnUsed(), data.getRow());
        style.setStyle(data.getColumn(), data.getRow(), DefaultExcelStyleSet
          .buildReserved(style));
      }
      data.incRow();
    }
  }

  /** Item de um exportvel para Excel */
  private class ExportableItem extends AbstractItem {
    /** Exportvel para Excel */
    private final ExcelExportable exportable;
    /** Leitor de Estrutura de Dados */
    private final ExcelTable format;

    /**
     * @param exportable Exportvel para Excel
     * @param format Leitor de Estrutura de Dados
     */
    public ExportableItem(ExcelExportable exportable, ExcelTable format) {
      this.exportable = exportable;
      this.format = format;
    }

    /** {@inheritDoc} */
    @Override
    public void build(ExcelStructureTool head, ExcelDataTool data,
      ExcelStyleTool style, boolean testing) {
      exportable.exportExcel(head, data, style, format);
    }
  }

  /**
   * Testador
   *
   * @param args
   * @throws IOException
   */
  public static void main(String[] args) throws IOException {
    new PoiExcel(new VerticalExcelModel() {
      @Override
      public void preBuild() {
        this.addHeader("Linha 1");
        this.addHeader("Linha 2");
        this.addHeader("Linha 3");
        this.addLine();
        this.addTable(new JTable(5, 3));
        this.addLine();
        this.addTable(new JTable(10, 5));
        this.addLine();
        this.addReserved();
      }
    }).save(new FileOutputStream("test.xls"));
  }

}
