package tecgraf.javautils.excel.v2;

import java.awt.Color;
import java.awt.Font;
import java.awt.Rectangle;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

import tecgraf.javautils.excel.v1.util.XmlNode;

/**
 * Estrutura que realiza toda a escrita de uma planilha excel e converte em
 * operaes primitivas capaz de ser representado em qualquer tipo de documento.
 * 
 * @author Tecgraf
 */
public class TecExcel {

  /** Sequence de Tag */
  private int tabSequence = 1;

  /** Documento */
  private final XmlNode document = new XmlNode("excel");

  /** Aba corrente */
  private XmlNode tab = new XmlNode("tab").setAttribute("name", "")
    .setAttribute("order", toString(tabSequence++));

  /**
   * Construtor
   */
  public TecExcel() {
    this.document.addNode(tab);
  }

  /**
   * @param row
   * @param column
   * @param text
   * @return this
   */
  public TecExcel setCell(int row, int column, String text) {
    if (row < 0 || column < 0 || text == null) {
      return this;
    }
    this.removeRectIntersects(row, column, "cell");
    this.removeCellEqual(row, column, "cell");
    this.tab.addNode(new XmlNode("cell").setAttribute("row", toString(row))
      .setAttribute("col", toString(column)).setAttribute("text", text));
    return this;
  }

  /**
   * @param row
   * @param column
   * @param color
   * @return this
   */
  public TecExcel setBackColor(int row, int column, Color color) {
    if (row < 0 || column < 0 || color == null) {
      return this;
    }
    this.removeRectIntersects(row, column, "backcolor");
    this.removeCellEqual(row, column, "backcolor");
    this.tab.addNode(new XmlNode("backcolor")
      .setAttribute("row", toString(row)).setAttribute("col", toString(column))
      .setAttribute("color", toString(color)));
    return this;
  }

  /**
   * @param row
   * @param column
   * @param color
   * @return this
   */
  public TecExcel setForeColor(int row, int column, Color color) {
    if (row < 0 || column < 0 || color == null) {
      return this;
    }
    this.removeRectIntersects(row, column, "forecolor");
    this.removeCellEqual(row, column, "forecolor");
    this.tab.addNode(new XmlNode("forecolor")
      .setAttribute("row", toString(row)).setAttribute("col", toString(column))
      .setAttribute("color", toString(color)));
    return this;
  }

  /**
   * @param row
   * @param column
   * @param font
   * @return this
   */
  public TecExcel setFont(int row, int column, Font font) {
    if (row < 0 || column < 0 || font == null) {
      return this;
    }
    this.removeRectIntersects(row, column, "font");
    this.removeCellEqual(row, column, "font");
    this.tab.addNode(new XmlNode("font").setAttribute("row", toString(row))
      .setAttribute("col", toString(column)).setAttribute("name",
        toString(font)));
    return this;
  }

  /**
   * @param row
   * @param column
   * @param align
   * @return this
   */
  public TecExcel setAlign(int row, int column, AlignEnum align) {
    if (row < 0 || column < 0 || align == null) {
      return this;
    }
    this.removeRectIntersects(row, column, "align");
    this.removeCellEqual(row, column, "align");
    this.tab.addNode(new XmlNode("align").setAttribute("row", toString(row))
      .setAttribute("col", toString(column)).setAttribute("value",
        toString(align)));
    return this;
  }

  /**
   * @param name
   * @return this
   */
  public TecExcel addTab(String name) {
    this.tab =
      new XmlNode("tab").setAttribute("name", name).setAttribute("order",
        toString(tabSequence++));
    this.document.addNode(tab);
    return this;
  }

  /**
   * @param rowBegin
   * @param rowEnd
   * @param columnBegin
   * @param columnEnd
   * @return this
   */
  public TecExcel setMerge(int rowBegin, int columnBegin, int rowEnd,
    int columnEnd) {
    if (rowBegin > rowEnd || columnBegin > rowEnd) {
      return this;
    }
    this
      .removeRectIntersects(rowBegin, columnBegin, rowEnd, columnEnd, "merge");
    this.tab.addNode(new XmlNode("merge").setAttribute("rowBegin",
      toString(rowBegin)).setAttribute("rowEnd", toString(rowEnd))
      .setAttribute("colBegin", toString(columnBegin)).setAttribute("colEnd",
        toString(columnEnd)));
    return this;
  }

  /**
   * @param rowBegin
   * @param rowEnd
   * @param columnBegin
   * @param columnEnd
   * @param size
   * @param color
   * @return this
   */
  public TecExcel setBox(int rowBegin, int columnBegin, int rowEnd,
    int columnEnd, int size, Color color) {
    if (rowBegin > rowEnd || columnBegin > rowEnd) {
      return this;
    }
    this.removeRectEqual(rowBegin, columnBegin, rowEnd, columnEnd, "box");
    this.tab.addNode(new XmlNode("box").setAttribute("rowBegin",
      toString(rowBegin)).setAttribute("rowEnd", toString(rowEnd))
      .setAttribute("colBegin", toString(columnBegin)).setAttribute("colEnd",
        toString(columnEnd)).setAttribute("size", toString(size)).setAttribute(
        "color", toString(color)));
    return this;
  }

  /**
   * @param name
   * @return this
   */
  public TecExcel setTabName(String name) {
    this.tab.setAttribute("name", name);
    return this;
  }

  /**
   * @return XmlNode
   */
  public XmlNode toXmlNode() {
    try {
      ByteArrayOutputStream output = new ByteArrayOutputStream();
      this.document.write(output);
      return new XmlNode(new ByteArrayInputStream(output.toByteArray()), false);
    }
    catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Remove a ocorrencia de todas as tags de nome 'tagName' que possua o mesmo
   * 'row' e 'column'.
   * 
   * @param row
   * @param column
   * @param tagName
   */
  protected void removeCellEqual(int row, int column, String tagName) {
    for (XmlNode node : this.tab.getNodesByTagName(tagName)) {
      int oRow = Integer.parseInt(node.getAttribute("row"));
      int oCol = Integer.parseInt(node.getAttribute("col"));
      if (row == oRow && column == oCol) {
        this.tab.getNodes().remove(node);
      }
    }
  }

  /**
   * @param rowBegin
   * @param columnBegin
   * @param rowEnd
   * @param columnEnd
   * @param tagName
   */
  protected void removeRectEqual(int rowBegin, int columnBegin, int rowEnd,
    int columnEnd, String tagName) {
    for (XmlNode node : this.tab.getNodesByTagName(tagName)) {
      int row0 = Integer.parseInt(node.getAttribute("rowBegin"));
      int row1 = Integer.parseInt(node.getAttribute("rowEnd"));
      int col0 = Integer.parseInt(node.getAttribute("colBegin"));
      int col1 = Integer.parseInt(node.getAttribute("colEnd"));
      if (rowBegin == row0 && rowEnd == row1 && columnBegin == col0
        && columnEnd == col1) {
        this.tab.getNodes().remove(node);
      }
    }
  }

  /**
   * Remove a ocorrencia de todas as tags de nome 'tagName' que possua o mesmo
   * 'row' e 'column' numa regio de merge.
   * 
   * @param row
   * @param column
   * @param tagName
   */
  protected void removeRectIntersects(int row, int column, String tagName) {
    Rectangle tRect = new Rectangle(column, row, 1, 1);
    for (XmlNode mNode : this.tab.getNodesByTagName("merge")) {
      int row0 = Integer.parseInt(mNode.getAttribute("rowBegin"));
      int row1 = Integer.parseInt(mNode.getAttribute("rowEnd"));
      int col0 = Integer.parseInt(mNode.getAttribute("colBegin"));
      int col1 = Integer.parseInt(mNode.getAttribute("colEnd"));
      Rectangle mRect =
        new Rectangle(col0, row0, col1 - col0 + 1, row1 - row0 + 1);
      if (tRect.intersects(mRect)) {
        for (XmlNode oNode : this.tab.getNodesByTagName(tagName)) {
          int oRow = Integer.parseInt(oNode.getAttribute("row"));
          int oCol = Integer.parseInt(oNode.getAttribute("col"));
          Rectangle oRect = new Rectangle(oCol, oRow, 1, 1);
          if (oRect.intersects(mRect)) {
            this.tab.getNodes().remove(oNode);
          }
        }
      }
    }
  }

  /**
   * @param rowBegin
   * @param columnBegin
   * @param rowEnd
   * @param columnEnd
   * @param tagName
   */
  protected void removeRectIntersects(int rowBegin, int columnBegin,
    int rowEnd, int columnEnd, String tagName) {
    Rectangle tRect =
      new Rectangle(columnBegin, rowBegin, columnEnd - columnBegin + 1, rowEnd
        - rowBegin + 1);
    for (XmlNode node : this.tab.getNodesByTagName(tagName)) {
      int row0 = Integer.parseInt(node.getAttribute("rowBegin"));
      int row1 = Integer.parseInt(node.getAttribute("rowEnd"));
      int col0 = Integer.parseInt(node.getAttribute("colBegin"));
      int col1 = Integer.parseInt(node.getAttribute("colEnd"));
      Rectangle oRect =
        new Rectangle(col0, row0, col1 - col0 + 1, row1 - row0 + 1);
      if (tRect.intersects(oRect)) {
        this.tab.getNodes().remove(node);
      }
    }
  }

  /**
   * @param color
   * @return string
   */
  protected static String toString(Color color) {
    StringBuilder sb = new StringBuilder();
    sb.append("0x00");
    String value =
      Integer.toHexString(
        (color.getRed() << 16) + (color.getGreen() << 8) + color.getBlue())
        .toUpperCase();
    while (sb.length() + value.length() != 10) {
      sb.append('0');
    }
    sb.append(value);
    return sb.toString();
  }

  /**
   * @param font
   * @return string
   */
  protected static String toString(Font font) {
    StringBuilder sb = new StringBuilder();
    sb.append(font.getName());
    if ((font.getStyle() & Font.BOLD) == Font.BOLD) {
      sb.append(' ');
      sb.append("bold");
    }
    if ((font.getStyle() & Font.ITALIC) == Font.ITALIC) {
      sb.append(' ');
      sb.append("italic");
    }
    sb.append(' ');
    sb.append(font.getSize());
    return sb.toString();
  }

  /**
   * @param align
   * @return string
   */
  protected static String toString(AlignEnum align) {
    switch (align) {
      case LEFT:
        return "left";
      case CENTER:
        return "center";
      case RIGHT:
        return "right";
      default:
        return "unknown";
    }
  }

  /**
   * @param value
   * @return string
   */
  protected static String toString(int value) {
    return Integer.toString(value);
  }

  /**
   * Alinhamento
   * 
   * @author Tecgraf
   */
  public static enum AlignEnum {

    /** Left */
    LEFT,
    /** Center */
    CENTER,
    /** Right */
    RIGHT;

  }

  /**
   * @param args
   * @throws Exception
   */
  public static void main(String[] args) throws Exception {
    TecExcel c = new TecExcel();
    c.setTabName("a");
    c.setCell(-1, 1, "aaa");
    c.setCell(1, 1, "bbb");
    c.setBackColor(0, 0, Color.ORANGE);
    c.setCell(1, 1, "ccc");
    c.setBackColor(0, 0, Color.CYAN);
    c.addTab("b");
    c.setCell(2, 3, "___");
    c.setMerge(1, 1, 2, 2);
    c.setMerge(0, 0, 1, 1);
    c.setCell(0, 0, "aaa");
    c.setCell(1, 1, "bbb");
    c.setBackColor(0, 0, Color.ORANGE);
    c.setBackColor(1, 1, Color.CYAN);
    c.setBox(1, 1, 3, 3, 1, Color.YELLOW);
    c.setBox(1, 1, 3, 3, 1, Color.RED);
    c.setForeColor(0, 0, Color.ORANGE);
    c.setForeColor(1, 1, Color.CYAN);
    c.setFont(0, 0, new Font("Arial", Font.BOLD | Font.ITALIC, 16));
    c.setAlign(1, 1, AlignEnum.CENTER);
    System.out.println(c.toXmlNode());
    System.out.println(new XmlNode(new ByteArrayInputStream(c.toXmlNode()
      .getBytes()), true));
  }
}
