package tecgraf.javautils.gui.table;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.GridBagLayout;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.Serializable;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;

import tecgraf.javautils.gui.GBC;

/**
 * Classe para renderizao do cabealho das colunas da tabela como um panel.
 */
public class MultiLinePanelHeaderRenderer implements TableCellRenderer,
  Serializable {

  /**
   * Identificador de verso da classe utilizada no processo de serializao.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Estilos para formatao do texto do cabealho.
   * 
   * @see MultiLinePanelHeaderRenderer#setTextStyleEnabled(TextStyle, boolean)
   */
  public enum TextStyle {
    /** Negrito */
    BOLD,
    /** Itlico */
    ITALIC,
    /** Sublinhado */
    UNDERLINE,
    /** Riscado */
    STRIKETHROUGH
  }

  /** Icone a ser exibido */
  protected Icon icon;

  /** Cor de fundo do HEADER */
  protected Color backgroundColor = UIManager
    .getColor("TableHeader.background");

  /** Cor das letras do HEADER */
  protected Color foregroundColor;

  /** Indica se o Header deve forar a existencia de duas linhas */
  protected boolean multilineForce = false;

  /** Borda do cabealho */
  protected Border headerBorder;

  /** Fonte utilizada na tabela */
  protected Font font = UIManager.getFont("TableHeader.font");

  /** Indica se o texto deve ser exibido em negrito */
  private boolean boldOn;

  /** Indica se o texto deve ser exibido em itlico */
  private boolean italicOn;

  /** Indica se o texto deve ser sublinhado */
  private boolean underlineOn;

  /** Indica se o texto deve ser riscado */
  private boolean strikeOn;

  /** Espaamento vertical entre as bordas e o texto */
  private int verticalGap = 8;

  /** Espao entre as linhas. */
  private int lineSpace = 2;

  /** Espaamento entre o texto e o cone */
  private int iconTextGap = 4;

  /** Posicionamento horizontal do texto. */
  private int horizontalTextPosition;

  /**
   * Posicionamento do texto. Para evitar que seja redefinido nas classes que
   * extendem desta classe.
   */
  protected static int LEFT = SwingConstants.LEFT;
  /** Posicionamento do texto. */
  protected static int RIGHT = SwingConstants.RIGHT;

  /**
   * Construtor padro
   */
  public MultiLinePanelHeaderRenderer() {
    super();
    this.horizontalTextPosition = LEFT;
    this.foregroundColor = UIManager.getColor("TableHeader.foreground");
    this.backgroundColor = UIManager.getColor("TableHeader.background");
    this.headerBorder = UIManager.getBorder("TableHeader.cellBorder");
  }

  /**
   * Construtor padro
   * 
   * @param icon o cone a ser exibido no cabealho do header
   */
  public MultiLinePanelHeaderRenderer(ImageIcon icon) {
    this();
    this.icon = icon;
  }

  /**
   * Configura o icone
   * 
   * @param icon cone.
   */
  public void setIcon(Icon icon) {
    this.icon = icon;
  }

  /**
   * Icone configurado
   * 
   * @return Icone
   */
  public Icon getIcon() {
    return this.icon;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getTableCellRendererComponent(JTable table, Object value,
    boolean isSelected, boolean hasFocus, int row, int column) {
    String str = (value == null) ? "" : value.toString();
    BufferedReader br = new BufferedReader(new StringReader(str));
    String line;
    List<String> v = new ArrayList<String>();
    int lines = 0;
    try {
      while ((line = br.readLine()) != null) {
        v.add(line);
        lines++;
      }
      if (multilineForce) {
        if (lines < 2) {
          v.add(" ");
          lines++;
        }
      }
    }
    catch (IOException e) {
      e.printStackTrace();
    }
    Object[] values = v.toArray();
    int length = lines;
    JPanel labelPanel = makeTextPanel(table, values, length);
    labelPanel.setAlignmentY(RIGHT);
    JPanel headerPanel = new JPanel(new GridBagLayout());
    GBC gbcLabel = new GBC(0, 0);
    if (isRightHorizontalTextPosition()) {
      gbcLabel = new GBC(1, 0);
    }
    headerPanel.add(labelPanel, gbcLabel);
    if (hasIconPanel()) {
      JPanel iconPanel = makeIconPanel(length);
      GBC gbcIcon = new GBC(1, 0).left(this.getIconTextGap());
      if (isRightHorizontalTextPosition()) {
        gbcIcon = new GBC(0, 0).right(this.getIconTextGap());
      }
      headerPanel.add(iconPanel, gbcIcon);
    }
    if (backgroundColor != null) {
      headerPanel.setBackground(backgroundColor);
    }
    headerPanel.setBorder(headerBorder);
    headerPanel.setOpaque(true);
    return headerPanel;
  }

  /**
   * Retorna se o painel ir ter ou no cones
   * 
   * @return boolean se tiver false se no
   */
  protected boolean hasIconPanel() {
    return icon != null;
  }

  /**
   * Cria o painel que ter os labels com os nomes da coluna.
   * 
   * @param table tabela a ter os labels definidos.
   * @param values valores dos labels.
   * @param length nmero de valores.
   * 
   * @return painel.
   */
  private JPanel makeTextPanel(JTable table, Object[] values, int length) {
    JPanel labelPanel = new JPanel(new GridBagLayout());
    labelPanel.setFont(table.getFont());
    FontMetrics metrics = labelPanel.getFontMetrics(labelPanel.getFont());
    // Configura as cores de foreground e o background 
    // de acordo com o header da tabela         
    if (table != null) {
      JTableHeader header = table.getTableHeader();
      if (header != null) {
        labelPanel.setForeground(header.getForeground());
        labelPanel.setBackground(header.getBackground());
      }
    }
    int maxHeight = (metrics.getHeight() + lineSpace) * (length) + verticalGap;
    // Configura cada linha do Header usando um JLabel
    for (int i = 0; i < length; i++) {
      JLabel l = new JLabel();
      setValue(l, values[i]);
      labelPanel.add(l, new GBC(0, i).gridheight(1).gridwidth(1));
    }
    Dimension d = labelPanel.getPreferredSize();
    d.height = Math.max(d.height, maxHeight);
    labelPanel.setPreferredSize(d);
    labelPanel.setOpaque(false);
    return labelPanel;
  }

  /**
   * Cria o painel que ter o icone a ser exibido.
   * 
   * @param length nmero de labels.
   * 
   * @return painel.
   */
  protected JPanel makeIconPanel(int length) {
    JLabel label = makeIconLabel(this.icon);

    JPanel iconPanel = new JPanel(new GridBagLayout());
    iconPanel.setOpaque(false);
    iconPanel.add(label, new GBC(0, 0).none());
    return iconPanel;
  }

  /**
   * Mtodo que cria uma label de um cone
   * 
   * @param icon
   * @return Icone da label
   */
  protected JLabel makeIconLabel(Icon icon) {
    JLabel label = new JLabel();
    label.setOpaque(false);
    label.setForeground(getForegroundColor());
    label.setBackground(getBackgroundColor());
    label.setFont(font);
    label.setIcon(icon);
    return label;
  }

  /**
   * Verifica se o posicionamento do texto  a esquerda do panel
   * 
   * @return True se o posiconamento do texto  a esquerda do panel. False caso
   *         contrrio.
   */
  private boolean isRightHorizontalTextPosition() {
    return horizontalTextPosition == RIGHT;
  }

  /**
   * Configura o JLabel para a exibio de uma linha do cabealho
   * 
   * @param l label a ser alterado.
   * @param value valor do label.
   */
  protected void setValue(JLabel l, Object value) {
    StringBuilder labelText = new StringBuilder();
    if (boldOn || italicOn || underlineOn || strikeOn) {
      labelText.append("<html>");
    }
    if (boldOn) {
      labelText.append("<b>");
    }
    if (italicOn) {
      labelText.append("<i>");
    }
    if (underlineOn) {
      labelText.append("<u>");
    }
    if (strikeOn) {
      labelText.append("<strike>");
    }
    if (value != null) {
      labelText.append(value.toString());
    }
    if (strikeOn) {
      labelText.append("</strike>");
    }
    if (underlineOn) {
      labelText.append("</u>");
    }
    if (italicOn) {
      labelText.append("</i>");
    }
    if (boldOn) {
      labelText.append("</b>");
    }
    if (boldOn || italicOn || underlineOn || strikeOn) {
      labelText.append("</html>");
    }
    l.setText(labelText.toString());
    l.setForeground(getForegroundColor());
    l.setBackground(getBackgroundColor());
    l.setHorizontalAlignment(RIGHT);
    l.setHorizontalTextPosition(JLabel.RIGHT);
    l.setOpaque(false);
    l.setFont(font);
  }

  /**
   * Configura a cor de fundo do header
   * 
   * @param color a cor de fundo
   */
  public void setBackgroundColor(Color color) {
    this.backgroundColor = color;
  }

  /**
   * Obtm a cor de fundo do header
   * 
   * @return a cor de fundo
   */
  public Color getBackgroundColor() {
    return this.backgroundColor;
  }

  /**
   * Configura a cor das letras do header
   * 
   * @param color a cor das letras
   */
  public void setForegroundColor(Color color) {
    this.foregroundColor = color;
  }

  /**
   * Obtm a cor das letras do header
   * 
   * @return a cor das letras
   */
  public Color getForegroundColor() {
    return this.foregroundColor;
  }

  /**
   * Indica se o header deve forar a existncia de duas linhas.
   * 
   * @param multiline verdadeiro se tiver que forar a existncia.
   */
  public void setMultilineForce(boolean multiline) {
    this.multilineForce = multiline;
  }

  /**
   * Liga ou desliga um determinado estilo de formatao para o texto do
   * cabealho.
   * 
   * @param style estilo a ser ligado/desligado.
   * @param enable se true, liga. Se false, desliga.
   */
  public void setTextStyleEnabled(TextStyle style, boolean enable) {
    switch (style) {
      case BOLD:
        boldOn = enable;
        break;
      case ITALIC:
        italicOn = enable;
        break;
      case UNDERLINE:
        underlineOn = enable;
        break;
      case STRIKETHROUGH:
        strikeOn = enable;
        break;
    }
  }

  /**
   * Atualiza o nmero de pixels adicionais para "centralizar" verticalmente o
   * texto no componente. Metade desse valor  inserido acima do texto e a outra
   * metade  inserida abaixo.
   * 
   * @param vertGap - novo nmero de pixels adicionais.
   */
  public void setVerticalGap(int vertGap) {
    this.verticalGap = vertGap;
  }

  /**
   * Define o espao entre as linhas.
   * 
   * @param space Espao entre as linhas.
   */
  public void setLineSpace(int space) {
    this.lineSpace = space;
  }

  /**
   * Retorna
   * 
   * @return iconTextGap
   */
  public int getIconTextGap() {
    return iconTextGap;
  }

  /**
   * @param iconTextGap
   */
  public void setIconTextGap(int iconTextGap) {
    this.iconTextGap = iconTextGap;
  }

  /**
   * Retorna
   * 
   * @return horizontalTextPosition
   */
  public int getHorizontalTextPosition() {
    return horizontalTextPosition;
  }

  /**
   * @param horizontalTextPosition
   */
  public void setHorizontalTextPosition(int horizontalTextPosition) {
    if (horizontalTextPosition != LEFT && horizontalTextPosition != RIGHT) {
      throw new IllegalArgumentException(
        "Os parametros aceitos so: LEFT e RIGHT");
    }
    this.horizontalTextPosition = horizontalTextPosition;
  }
}
