/*
 * $Id$
 */
package tecgraf.javautils.gui.print;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.print.PageFormat;
import java.awt.print.Paper;
import java.util.List;

/**
 * Grfico a ser impresso. Como o JFreeChart desenha o grfico de acordo com o
 * espao disponvel, a definio da rea a ser desenhada pode ser determinada
 * de duas formas:
 * <ol>
 * <li>Passando um array com os demais componentes que iro ocupar a mesma
 * pgina: o grfico ocupa toda a largura da pgina e a altura remanescente,
 * descontando a altura dos outros componentes;</li>
 * <li>Passando dois percentuais de ocupao das dimenses da pgina, variando
 * de 0 a 1. Exemplo: se forem passados 0.2 para a largura e 0.5 para a altura,
 * o componente ir ocupar 20% da largura disponvel e 50% da altura disponvel.
 * </li>
 * </ol>
 */
public class PrintableChart implements PrintableReportItem {
  /** Grfico a ser impresso */
  private PrintableChartPanel chartPanel;
  /** Altura absoluta do componente */
  private float height;
  /** Largura absoluta do componente */
  private float width;
  /** Outros componentes a serem impressos na pgina */
  private List<PrintableReportItem> otherPageItems;
  /**
   * Altura proporcional do componente em relao  pgina, variando de 0 a 1
   * (100%).
   */
  private float heightRatio;
  /**
   * Largura proporcional do componente em relao  pgina, variando de 0 a 1
   * (100%).
   */
  private float widthRatio;
  /**
   * Coordenada y para o papel, quando em orientao LANDSCAPE.
   */
  private float yPaperLandscape;

  /**
   * Construtor.
   * 
   * @param chartPanel grfico a ser impresso.
   * @param otherPageItems outros itens a serem impressos na pgina.
   */
  public PrintableChart(PrintableChartPanel chartPanel,
    List<PrintableReportItem> otherPageItems) {
    if (otherPageItems == null) {
      throw new IllegalArgumentException("otherPageItems == null");
    }
    this.chartPanel = chartPanel;
    this.otherPageItems = otherPageItems;
  }

  /**
   * Construtor que calcula as dimenses do grfico a partir de dois parmetros
   * fracionrios, representando o percentual de utilizao do espao disponvel
   * em ambos os eixos vertical e horizontal.
   * 
   * @param chartPanel grfico a ser impresso.
   * @param heightRatio percentual de ocupao da altura da pgina (0 a 1).
   * @param widthRatio percentual de ocupao da largura da pgina (0 a 1).
   */
  public PrintableChart(PrintableChartPanel chartPanel, float heightRatio,
    float widthRatio) {
    this.chartPanel = chartPanel;
    if (heightRatio < 0 || heightRatio > 1) {
      throw new IllegalArgumentException("heightRatio  uma frao entre 0 e 1");
    }
    if (widthRatio < 0 || widthRatio > 1) {
      throw new IllegalArgumentException("widthRatio  uma frao entre 0 e 1");
    }
    this.heightRatio = heightRatio;
    this.widthRatio = widthRatio;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean simulatePrint(Graphics2D g2, PageFormat format, int pageIndex) {
    return print(g2, format, pageIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean print(Graphics2D g2, PageFormat pageFormat, int pageIndex) {
    if (otherPageItems == null) {
      height = (float) pageFormat.getImageableHeight() * heightRatio;
      width = (float) pageFormat.getImageableWidth() * widthRatio;
      yPaperLandscape =
        (float) pageFormat.getPaper().getImageableY()
          + (float) pageFormat.getImageableWidth() - width;
    }
    else {
      yPaperLandscape = (float) pageFormat.getImageableX();
      height =
        (float) pageFormat.getImageableHeight()
          - getOthersHeight(g2, pageFormat, pageIndex) - 1;
      width = (float) pageFormat.getImageableWidth();
    }

    double x =
      (pageFormat.getOrientation() != PageFormat.LANDSCAPE) ? pageFormat
        .getImageableX() : pageFormat.getImageableY();
    double y =
      (pageFormat.getOrientation() != PageFormat.LANDSCAPE) ? pageFormat
        .getImageableY() : yPaperLandscape;
    double paperWidth =
      (pageFormat.getOrientation() != PageFormat.LANDSCAPE) ? width : height;
    double paperHeight =
      (pageFormat.getOrientation() != PageFormat.LANDSCAPE) ? height : width;
    Paper paper = pageFormat.getPaper();
    paper.setImageableArea(x, y, paperWidth, paperHeight);
    PageFormat graphicPageFormat = (PageFormat) pageFormat.clone();
    graphicPageFormat.setPaper(paper);

    Paint oldColor = chartPanel.getBackgroundPaint();
    chartPanel.setBackgroundPaint(Color.white);

    AffineTransform affineTransform = g2.getTransform();
    Shape clip = g2.getClip();
    chartPanel.print(g2, graphicPageFormat);
    g2.setTransform(affineTransform);
    g2.setClip(clip);

    chartPanel.setBackgroundPaint(oldColor);

    return true;
  }

  /**
   * Obtm a altura dos outros componentes a serem impressos na pgina.
   * 
   * @param g2 contexto grfico.
   * @param pageFormat formato da pgina.
   * @param pageIndex ndice da pgina.
   * 
   * @return altura dos outros componentes a serem impressos na pgina.
   */
  private float getOthersHeight(Graphics2D g2, PageFormat pageFormat,
    int pageIndex) {
    float height = 0;
    for (PrintableReportItem printable : otherPageItems) {
      printable.simulatePrint(g2, pageFormat, pageIndex);
      height += printable.getHeight();
    }
    return height;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public float getHeight() {
    return height;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initPrinting(PrintConfiguration configuration) {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public float getWidth() {
    return width;
  }
}
