package tecgraf.javautils.gui.table;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.Insets;

import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.UIManager;
import javax.swing.table.DefaultTableCellRenderer;

/**
 * Classe para renderizao do cabealho das colunas da tabela, sendo que o
 * texto pode [ou no] conter mltiplas linhas ['\n']. Nessa implementao
 * usamos um JLabel como componente, e nele inserimos o texto formatado em HTML.
 */
public class MultiLineLabelHeaderRenderer extends DefaultTableCellRenderer {

  /**
   * Identificador de verso da classe utilizada no processo de serializao.
   */
  private static final long serialVersionUID = 1L;

  /** Espaamento vertical entre as bordas e o texto */
  private int verticalGap = 8;
  /** Espao entre as linhas. */
  private int lineSpace = 2;

  /**
   * Construtor.
   */
  public MultiLineLabelHeaderRenderer() {
    setOpaque(true);
    setForeground(UIManager.getColor("TableHeader.foreground"));
    setBackground(UIManager.getColor("TableHeader.background"));
    setBorder(UIManager.getBorder("TableHeader.cellBorder"));
    setHorizontalAlignment(JLabel.CENTER);
    setVerticalAlignment(JLabel.CENTER);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getTableCellRendererComponent(JTable table, Object value,
    boolean isSelected, boolean hasFocus, int row, int column) {
    setTitle(table, (String) value);
    return this;
  }

  /**
   * Mtodo que pode ser chamado externamente para "forar" um determinado texto
   * para o componente. Esse componente  ento modificado para que fique com a
   * borda e com a fonte iguais aos da tabela passada como referncia. Esse
   * mtodo pode ser usado diretamente se esse componente no for ser usado como
   * um cell-renderer, e sim como um JLabel. Isso  devido ao fato de que como
   * um renderer, o mtodo 'getTableCellRendererComponent' ser chamado,
   * sobrescrevendo o que foi alterado por esse mtodo.
   * 
   * @param table - tabela de referncia.
   * @param title - novo ttulo, com '\n's opcionais.
   */
  public void setTitle(JTable table, String title) {
    setFont(table.getFont());

    setText("<html><center>" + title.replaceAll("\n", "<br>")
      + "</center></html>");

    Insets insets = getInsets();
    FontMetrics metrics = getFontMetrics(getFont());
    String[] lines = title.split("\n");

    // Calcula as dimenses do texto.
    int width = 0;
    for (String aLine : lines) {
      width = Math.max(width, metrics.stringWidth(aLine));
    }
    int height =
      +((metrics.getHeight() + lineSpace) * lines.length);
    
    // Inclui o gap e os insets
    width += insets.left + insets.right;
    height += verticalGap + insets.top + insets.bottom;

    // Inclui espao ocupado pelo o cone.
    Icon icon = getIcon();
    if (icon != null) {
      // Adiciona a largura do cone e o espao entre ele e o texto.
      width += getIconTextGap() + getIcon().getIconWidth();
      // Caso o cone seja mais alto que uma linha de texto, adiciona a diferena.
      height +=
        Math.max(metrics.getHeight(), icon.getIconHeight())
          - metrics.getHeight();
    }

    Dimension size = new Dimension(width, height);
    setPreferredSize(size);
    setMinimumSize(size);
  }

  /**
   * Atualiza o nmero de pixels adicionais para "centralizar" verticalmente o
   * texto no componente. Metade desse valor  inserido acima do texto e a outra
   * metade  inserida abaixo.
   * 
   * @param vertGap - novo nmero de pixels adicionais.
   */
  public void setVerticalGap(int vertGap) {
    this.verticalGap = vertGap;
  }
  
  /**
   * Define o espao entre as linhas.
   * 
   * @param space Espao entre as linhas.
   */
  public void setLineSpace(int space) {
    this.lineSpace = space;
  }
}
