package tecgraf.javautils.gui.crud.gui.main;

import java.awt.BorderLayout;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import javax.swing.JPanel;
import javax.swing.JSplitPane;

import tecgraf.javautils.gui.crud.IRegistrationDescriptor;
import tecgraf.javautils.gui.crud.RegistrationBundle;
import tecgraf.javautils.gui.crud.gui.editbuttons.RegistrationEditButtonsPanel;
import tecgraf.javautils.gui.crud.gui.edition.RegistrationEditionPanel;
import tecgraf.javautils.gui.crud.gui.table.RegistrationTablePanel;

/**
 * Painel principal para cadastro.
 * 
 * @author Tecgraf
 * @param <M> tipo do objeto do modelo.
 * @param <I> tipo do objeto da interface.
 */
public class RegistrationMainPanel<M, I> extends JPanel {

  /**
   * Modo
   * 
   * @author Tecgraf
   */
  public enum Mode {
    /**
     * Adio de novo objeto.
     */
    ADD,

    /**
     * Visualizao
     */
    VIEW,

    /**
     * Edio.
     */
    EDIT,
  }

  /**
   * Listeners de modo.
   */
  final private List<RegistrationModeListener> modeListeners =
    new ArrayList<RegistrationModeListener>();

  /**
   * Descritor
   */
  final private IRegistrationDescriptor<M, I> descriptor;

  /**
   * Painel de tabela.
   */
  final private RegistrationTablePanel<M, I> tablePanel;

  /**
   * Modo
   */
  private Mode mode;

  /**
   * Painel de botes.
   */
  final private RegistrationEditButtonsPanel<M, I> buttonsPanel;

  /**
   * Bundle
   */
  final private ResourceBundle bundle;

  /**
   * Consulta texto para internacionalizao.
   * 
   * @param tag tag
   * @return texto.
   */
  public String getString(final String tag) {
    if (bundle == null) {
      return "!!! " + tag;
    }
    try {
      return bundle.getString(tag);
    }
    catch (MissingResourceException mre) {
      return "[[" + tag + "]]";
    }
  }

  /**
   * Construtor.
   * 
   * @param descriptor descritor
   * @param locale locale
   */
  public RegistrationMainPanel(final IRegistrationDescriptor<M, I> descriptor,
    final Locale locale) {
    this(descriptor, new RegistrationBundle(locale));
  }

  /**
   * Construtor.
   * 
   * @param descriptor descritor
   * @param bundle
   */
  public RegistrationMainPanel(final IRegistrationDescriptor<M, I> descriptor,
    final ResourceBundle bundle) {
    if (bundle == null) {
      throw new RuntimeException("null bundle not allowed!");
    }

    this.descriptor = descriptor;
    this.bundle = bundle;

    this.tablePanel = new RegistrationTablePanel<M, I>(this, descriptor);
    final RegistrationEditionPanel<M, I> edtPanel =
      descriptor.getRegistrationEditPanel(this);
    this.buttonsPanel =
      new RegistrationEditButtonsPanel<M, I>(this, edtPanel, descriptor);

    this.mode = Mode.VIEW;

    final JPanel rightPanel = new JPanel();
    rightPanel.setLayout(new BorderLayout());
    rightPanel.add(BorderLayout.CENTER, edtPanel);
    rightPanel.add(BorderLayout.SOUTH, buttonsPanel);

    final JSplitPane split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
    split.add(tablePanel);
    split.add(rightPanel);
    split.setOneTouchExpandable(true);

    setLayout(new BorderLayout());
    add(BorderLayout.CENTER, split);

    edtPanel.clearIHC();
    edtPanel.setReadOnlyIHC(true);

    setMode(Mode.VIEW);
  }

  /**
   * Consulta objeto selecionado.
   * 
   * @return objeto
   */
  public M getSelectedObject() {
    return tablePanel.getSelectedObject();
  }

  /**
   * Consulta descritor.
   * 
   * @return descritor
   */
  public IRegistrationDescriptor<M, I> getDescriptor() {
    return descriptor;
  }

  /**
   * Consulta a viso IHC do objeto selecionado.
   * 
   * @return objeto
   */
  public I getSelectedIHCObject() {
    return descriptor.toIHCObject(getSelectedObject());
  }

  /**
   * Consulta de modo.
   * 
   * @return modo
   */
  public Mode getMode() {
    return this.mode;
  }

  /**
   * Ajuste de modo.
   * 
   * @param mode modo
   */
  public void setMode(Mode mode) {
    final Mode oldMode = this.mode;
    this.mode = mode;
    notifyModeListeners(oldMode, mode);
  }

  /**
   * Notifica listerners.
   * 
   * @param oldMode modo antigo
   * @param newMode modo novo.
   */
  private void notifyModeListeners(Mode oldMode, Mode newMode) {
    for (RegistrationModeListener l : modeListeners) {
      l.modeChanged(oldMode, newMode);
    }
  }

  /**
   * Adio de listener de modo
   * 
   * @param listener listener
   */
  public void addModeListener(final RegistrationModeListener listener) {
    modeListeners.add(listener);
  }

  /**
   * Remoo de listener de modo
   * 
   * @param listener listener
   */
  public void delModeListener(final RegistrationModeListener listener) {
    modeListeners.remove(listener);
  }

  /**
   * Atualizao de interface
   */
  public void refreshGUI() {
    tablePanel.updateTable();
  }

  /**
   * Sinalizao de objeto editado.
   * 
   * @param flag indicativo
   */
  public void signalObjectEdited(boolean flag) {
    buttonsPanel.signalObjectEdited(flag);
  }
}
