package tecgraf.javautils.gui.table;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.swing.event.TableModelEvent;
import javax.swing.table.AbstractTableModel;

/**
 * <p>
 * Modelo para tabelas em que cada objeto representando uma linha da tabela.
 * Este modelo utiliza instncias da interface
 * {@link tecgraf.javautils.gui.table.ObjectTableProvider} para obter os valores
 * das clulas.
 * </p>
 * <p>
 * INSTRUES PARA USO:<br>
 * Cada objeto da lista a ser passada para o modelo deve ter propriedades que
 * representem as colunas da tabela. O construtor recebe como parmetro os nomes
 * das colunas e um "provedor de valores" (instncia da
 * <code>ObjectTableProvider</code>) para obter a propriedade adequada para cada
 * coluna.
 * </p>
 * <p>
 * Antes de utilizar essa classe, de uma olhada na classe
 * {@link ColumnsObjectTableModel}. Elas so bem parecidas, mas a
 * {@link ColumnsObjectTableModel} substitui o {@link ObjectTableProvider} por
 * um array de {@link IColumn}. Essa mudana na arquitetura,
 * <ul>
 * <li>facilita o reso do cdigo da coluna,</li>
 * <li>exime o desenvolvedor de trabalhar com o ndice das colunas como no
 * {@link ObjectTableProvider},</li>
 * <li>facilita verificar se uma clula  editvel, pois o mtodo
 * {@link IColumn#isEditable(Object)} recebe apenas o objeto-linha, diferente do
 * metodo {@link ModifiableObjectTableProvider#isCellEditable(int, int)} que
 * recebe os ndices da linha e coluna da clula.</li>
 * </ul>
 * </p>
 * 
 * @param <T> tipo do objeto que representa cada linha da tabela
 * 
 * @see ObjectTableModelSample
 * 
 * @author Leonardo Barros
 */
public class ObjectTableModel<T> extends AbstractTableModel {
  /** Lista de objetos representando linhas da tabela */
  private List<T> rows;

  /** Provedor que extrai os valores das clulas dos objetos-linha */
  private ObjectTableProvider<T> provider;

  /**
   * Cria o modelo.
   * 
   * @param rows lista de objetos-linha.
   * @param provider provedor para extrair os valores das clulas dos
   *        objetos-linha.
   * 
   * @throws IllegalArgumentException se a lista de linhas ou o provedor forem
   *         iguais a null, ou se a lista de nomes das colunas retornada pelo
   *         provedor for null
   */
  public ObjectTableModel(List<T> rows, ObjectTableProvider<T> provider) {
    if (rows == null) {
      throw new IllegalArgumentException("rows == null");
    }
    if (provider == null) {
      throw new IllegalArgumentException("provider == null");
    }
    if (provider.getColumnNames() == null) {
      throw new IllegalArgumentException("colNames == null");
    }
    this.rows = rows;
    this.provider = provider;
  }

  /**
   * Obtm o <code>ObjectTableProvider</code>
   * 
   * @return ObjectTableProvider
   */
  public ObjectTableProvider<T> getProvider() {
    return provider;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getRowCount() {
    return rows.size();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnCount() {
    return this.provider.getColumnNames().length;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValueAt(int rowIndex, int colIndex) {
    if (rowIndex < 0 || colIndex < 0) {
      // FIXME e os limites superiores?
      return null;
    }
    return provider.getCellValue(rows.get(rowIndex), colIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAt(Object newValue, int rowIndex, int colIndex) {
    if (rowIndex < 0 || colIndex < 0) {
      // FIXME e os limites superiores?
      return;
    }
    if (!(provider instanceof ModifiableObjectTableProvider<?>)) {
      return;
    }
    T row = rows.get(rowIndex);
    if (row != null) {
      ((ModifiableObjectTableProvider<T>) provider).setValueAt(row, newValue,
        colIndex);
      fireTableCellUpdated(rowIndex, colIndex);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isCellEditable(int rowIndex, int columnIndex) {
    if (!(provider instanceof ModifiableObjectTableProvider<?>)) {
      return false;
    }
    T row = rows.get(rowIndex);
    if (row != null) {
      return ((ModifiableObjectTableProvider<T>) provider).isCellEditable(
        rowIndex, columnIndex);
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getColumnName(int colIndex) {
    return this.provider.getColumnNames()[colIndex];
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Class<?> getColumnClass(int colIndex) {
    Class<?>[] columnClasses = this.provider.getColumnClasses();
    if (columnClasses == null) {
      return super.getColumnClass(colIndex);
    }
    return columnClasses[colIndex];
  }

  /**
   * Remove todos os elementos de objetos.
   */
  public void clear() {
    int size = rows.size();
    this.rows = new ArrayList<T>();
    if (size > 0) {
      fireTableRowsDeleted(0, size - 1);
    }
  }

  /**
   * Substitui a lista de objetos-linha.
   * 
   * @param newRows nova lista de objetos-linha.
   * 
   * @throws IllegalArgumentException se a lista de novas linhas for null
   */
  public void setRows(List<T> newRows) {
    if (newRows == null) {
      throw new IllegalArgumentException("newRows == null");
    }
    this.rows = newRows;
    fireTableChanged(new TableModelEvent(this));
  }

  /**
   * Adiciona um objeto-linha  lista. Se j existir um objeto com o mesmo
   * identificador, este  sobrescrito (no permite duplicatas).
   * 
   * @param row objeto-linha a ser adicionado.
   * 
   * @throws IllegalArgumentException se <code>row</code> for nulo.
   */
  public void add(T row) {
    add(row, true);
  }

  /**
   * Adiciona uma nova linha  lista, com a possibilidade de notificar ou no a
   * tabela.
   * 
   * @param row nova linha
   * @param notify <code>true</code> se a tabela deve ser notificada da insero
   */
  private void add(T row, boolean notify) {
    if (row == null) {
      throw new IllegalArgumentException("row == null");
    }
    int index = rows.indexOf(row);
    if (index != -1) {
      rows.set(index, row);
      if (notify) {
        fireTableRowsUpdated(index, index);
      }
    }
    else {
      rows.add(row);
      if (notify) {
        index = rows.size() - 1;
        fireTableRowsInserted(index, index);
      }
    }
  }

  /**
   * Adiciona uma coleo de objetos-linha  lista. Se j existir um objeto com
   * o mesmo identificador, este  sobrescrito (no permite duplicatas).
   * 
   * @param rows coleo de objetos-linha a serem adicionados.
   * 
   * @throws IllegalArgumentException se <code>rows</code> for nulo.
   */
  public void addAll(Collection<T> rows) {
    if (rows == null) {
      throw new IllegalArgumentException("rows == null");
    }
    for (T row : rows) {
      /*
       * inserimos a linha sem notificar a tabela, pois faremos uma nica
       * notificao aps o loop
       */
      add(row, false);
    }
    fireTableChanged(new TableModelEvent(this));
  }

  /**
   * Remove um objeto-linha da lista.
   * 
   * @param row objeto-linha a ser removido.
   * 
   * @return true se o objeto-linha for encontrado e removido com sucesso, false
   *         caso contrrio.
   * 
   * @throws IllegalArgumentException se <code>row</code> for nulo.
   */
  public boolean remove(T row) {
    if (row == null) {
      throw new IllegalArgumentException("row == null");
    }
    int index = rows.indexOf(row);
    if (index == -1) {
      return false;
    }
    rows.remove(index);
    fireTableRowsDeleted(index, index);
    return true;
  }

  /**
   * Remove um objeto-linha da lista pelo seu ndice do modelo.
   * 
   * @param rowIndex ndice do objeto-linha a ser removido.
   * 
   * @return o objeto-linha removido.
   * 
   * @throws IndexOutOfBoundsException se o ndice for negativo ou se for maior
   *         do que o tamanho da lista de objeto-linha.
   */
  public T remove(int rowIndex) {
    T row = rows.remove(rowIndex);
    fireTableRowsDeleted(rowIndex, rowIndex);
    return row;
  }

  /**
   * Remove os objetos-linha especificados pelos seus ndices.
   * 
   * @see AbstractTableModel#fireTableRowsDeleted(int, int)
   * @param rowIndexes ndices dos objetos-linha a serem removidos.
   * 
   * @return coleo de objetos removidos.
   * 
   * @throws IllegalArgumentException se <code>rowIndexes</code> for nulo.
   * @throws IndexOutOfBoundsException se algum ndice for negativo ou se for
   *         maior do que o tamanho da lista de objeto-linha.
   * 
   */
  public Collection<T> removeAll(int[] rowIndexes) {
    if (rowIndexes == null) {
      throw new IllegalArgumentException("rowIndexes == null");
    }
    List<T> removedRows = new ArrayList<T>();
    if (rowIndexes.length > 0) {
      /*
       * devemos sinalizar mudanas apenas no intervalo que compreende as linhas
       * removidas (inclusive). Para isso, precisamos achar o menor e o maior
       * ndices dentre os reportados.
       */
      int start = Integer.MAX_VALUE, end = 0;
      for (int rowIndex : rowIndexes) {
        removedRows.add(rows.get(rowIndex));
        start = Math.min(start, rowIndex);
        end = Math.max(end, rowIndex);
      }
      rows.removeAll(removedRows);
      fireTableRowsDeleted(start, end);
    }
    return removedRows;
  }

  /**
   * Remove todos os objetos-linha.
   * 
   * @return coleo de objetos removidos.
   */
  public Collection<T> removeAll() {
    Collection<T> allRows = getRows();
    clear();
    return allRows;
  }

  /**
   * Modifica os dados de um objeto-linha da lista.
   * 
   * @param row objeto-linha a ser modificado.
   * 
   * @return true caso o objeto tenha sido modificado com sucesso, false caso
   *         este no tenha sido encontrado.
   * 
   * @throws IllegalArgumentException se a linha for null
   */
  public boolean modify(T row) {
    if (row == null) {
      throw new IllegalArgumentException("row == null");
    }
    int index = rows.indexOf(row);
    if (index == -1) {
      return false;
    }
    rows.set(index, row);
    fireTableRowsUpdated(index, index);
    return true;
  }

  /**
   * Obtm a lista de objetos-linha atualizada.
   * 
   * @return lista de objetos-linha.
   */
  public List<T> getRows() {
    return rows;
  }

  /**
   * Obtm um objeto-linha especfico.
   * 
   * @param rowIndex ndice do objeto-linha
   * 
   * @return o objeto-linha solicitado, ou null
   * 
   * @throws IndexOutOfBoundsException se o ndice for invlido
   */
  public T getRow(int rowIndex) {
    return rows.get(rowIndex);
  }

  /**
   * Troca duas linhas no modelo.
   * 
   * @param index1 - ndice da primeira linha
   * @param index2 - ndice da segunda linha
   */
  public void swapRows(int index1, int index2) {
    Collections.swap(getRows(), index1, index2);
    fireTableDataChanged();
  }
}
