package tecgraf.javautils.gui.crud.gui.table;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SortOrder;
import javax.swing.SwingConstants;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.crud.IRegistrationDescriptor;
import tecgraf.javautils.gui.crud.RegistrationImages;
import tecgraf.javautils.gui.crud.gui.actions.RegistrationObjectAction;
import tecgraf.javautils.gui.crud.gui.actions.RegistrationSelObjectDeleteAction;
import tecgraf.javautils.gui.crud.gui.actions.RegistrationSelObjectEditAction;
import tecgraf.javautils.gui.crud.gui.actions.RegistrationAddAction;
import tecgraf.javautils.gui.crud.gui.main.RegistrationMainPanel;
import tecgraf.javautils.gui.crud.gui.main.RegistrationMainPanel.Mode;
import tecgraf.javautils.gui.crud.gui.main.RegistrationModeListener;
import tecgraf.javautils.gui.table.SortableTable;

/**
 * Painel que representa os objetos que podem ser selecionados para
 * consulta/edio.
 * 
 * @author Tecgraf
 * @param <M> tipo do objeto do modelo.
 * @param <I> tipo do objeto da interface.
 */
public class RegistrationTablePanel<M, I> extends JPanel {

  /**
   * A tabela
   */
  final private SortableTable table = new SortableTable();

  /**
   * Scrollpane da tabela principal.
   */
  final private JScrollPane scrollPane;

  /**
   * Descritor
   */
  final private IRegistrationDescriptor<M, I> descriptor;

  /**
   * Boto: others
   */
  final private JButton othersButton = new JButton();

  /**
   * Ao de edio
   */
  final private RegistrationSelObjectEditAction<M, I> editAction;

  /**
   * Ao de adio.
   */
  final private RegistrationAddAction<M, I> addAction;

  /**
   * Ao de deleo
   */
  final private RegistrationSelObjectDeleteAction<M, I> delAction;

  /**
   * Painel mestre.
   */
  final private RegistrationMainPanel<M, I> mainPanel;

  /**
   * Inicializao da tabela com sua montagem em grupos, ajuste de adapters e
   * tamanhos de colunas.
   */
  final private void updateTableColumns() {
    final TableColumnModel columnModel = table.getColumnModel();
    final int numColumns = columnModel.getColumnCount();

    for (int c = 0; c < numColumns; c++) {
      final TableColumn column = columnModel.getColumn(c);
      column.setPreferredWidth(descriptor.getColumnWidth(c));
    }
  }

  /**
   * Retorna o arquivo selecionado.
   * 
   * @return o arquivo.
   */
  final public M getSelectedObject() {
    final int row = table.getSelectedRow();
    if (row < 0) {
      return null;
    }
    @SuppressWarnings("unchecked")
    M object = (M) table.getValueAt(row, 0);
    return object;
  }

  /**
   * Inicializao da tabela (relativo ao modelo).
   */
  private void initTableAttributes() {
    final RegistrationTableModel<M, I> model =
      new RegistrationTableModel<M, I>(descriptor);
    table.setModel(model);

    final RegistrationTableRenderer<M, I> renderer =
      new RegistrationTableRenderer<M, I>(descriptor);
    table.setDefaultRenderer(Object.class, renderer);
    table.sort(0, SortOrder.ASCENDING);
    table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    if (model.getRowCount() > 0) {
      table.setRowSelectionInterval(0, 0);
    }
    table.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
  }

  /**
   * Inicializao da tabela (relativo ao modelo).
   */
  private void updateTableAttributes() {
    final RegistrationTableModel<M, I> model =
      new RegistrationTableModel<M, I>(descriptor);
    table.setModel(model);
    table.sort(0, SortOrder.ASCENDING);
    if (model.getRowCount() > 0) {
      table.setRowSelectionInterval(0, 0);
    }
  }

  /**
   * Montagem de painel de botes
   * 
   * @param addButton boto add
   * @param delButton boto del
   * @param editButton boto edit
   * @param othersButton boto others
   * 
   * @return painel.
   */
  private JPanel buildButtonsPanel(JButton addButton, JButton delButton,
    JButton editButton, JButton othersButton) {
    final JPanel panel = new JPanel();
    panel.setLayout(new GridBagLayout());
    panel.add(addButton, new GBC(0, 0).center().none().insets(2));
    //    panel.add(editButton, new GBC(1, 0).center().none().insets(2));
    panel.add(delButton, new GBC(2, 0).center().none().insets(2));
    panel.add(new JPanel(), new GBC(3, 0).horizontal().insets(2));
    panel.add(othersButton, new GBC(4, 0).center().none().insets(2));

    // A decidir: se os botes devero ter textos ou no.
    addButton.setText(null);
    editButton.setText(null);
    delButton.setText(null);

    GUIUtils.matchPreferredSizes(addButton, delButton, editButton);

    return panel;
  }

  /**
   * Construtor
   * 
   * @param mainPanel painel
   * @param descriptor descritor
   */
  public RegistrationTablePanel(final RegistrationMainPanel<M, I> mainPanel,
    final IRegistrationDescriptor<M, I> descriptor) {
    this.mainPanel = mainPanel;
    this.descriptor = descriptor;
    this.scrollPane = new JScrollPane(table);

    this.editAction = new RegistrationSelObjectEditAction<M, I>(mainPanel);
    this.addAction = new RegistrationAddAction<M, I>(mainPanel);
    this.delAction = new RegistrationSelObjectDeleteAction<M, I>(mainPanel);
    final JButton delButton = new JButton(delAction);
    final JButton addButton = new JButton(addAction);
    final JButton editButton = new JButton(editAction);
    initOthersButton();

    setLayout(new BorderLayout());
    final JPanel buttonsPanel =
      buildButtonsPanel(addButton, delButton, editButton, othersButton);
    add(BorderLayout.CENTER, scrollPane);
    add(BorderLayout.SOUTH, buttonsPanel);

    initTableAttributes();

    setAddModeListener(mainPanel);
    setEditModeListener(mainPanel);
    setViewModeListener(mainPanel);

    setSelectionListener(mainPanel);

    updateTableColumns();
    updateTableAttributes();
    final RegistrationTableMouseListener<M, I> lst =
      new RegistrationTableMouseListener<M, I>(this, descriptor);
    table.addMouseListener(lst);

    updateSelectionButtons();

    setMinimumSize(new Dimension(300, 300));
  }

  /**
   * Atualiza estado de botes que dependem de seleo sobre um objeto (se o
   * mesmo existir).
   */
  private void updateSelectionButtons() {
    final int numObjects = descriptor.getNumObjects();
    final M selectedObject = getSelectedObject();
    final boolean flag =
      (numObjects > 0 && selectedObject != null && table.isEnabled());

    editAction.setEnabled(flag);
    delAction.setEnabled(flag);
    othersButton.setEnabled(flag);

    if (flag) {
      final List<RegistrationObjectAction<M, I>> othersMenuActions =
        descriptor.getOthersMenuActions(mainPanel, selectedObject);
      othersButton.setEnabled(othersMenuActions != null
        && !othersMenuActions.isEmpty());
    }
  }

  /**
   * Ajusta o listener de seleo.
   * 
   * @param mainPanel painel
   */
  private void setSelectionListener(final RegistrationMainPanel<M, I> mainPanel) {
    table.getSelectionModel().addListSelectionListener(
      new ListSelectionListener() {
        @Override
        public void valueChanged(ListSelectionEvent e) {
          updateSelectionButtons();
          mainPanel.setMode(Mode.VIEW);
        }
      });
  }

  /**
   * Ajusta listener: {@link Mode#ADD}
   * 
   * @param mainPanel painel
   */
  private void setAddModeListener(final RegistrationMainPanel<M, I> mainPanel) {
    final RegistrationTablePanel<M, I> self = RegistrationTablePanel.this;
    mainPanel.addModeListener(new RegistrationModeListener() {
      @Override
      public void modeChanged(Mode oldMode, Mode newMode) {
        if (newMode != Mode.ADD) {
          return;
        }
        self.setActive(false);
      }
    });
  }

  /**
   * Ajusta listener: {@link Mode#EDIT}
   * 
   * @param mainPanel painel
   */
  private void setEditModeListener(final RegistrationMainPanel<M, I> mainPanel) {
    final RegistrationTablePanel<M, I> self = RegistrationTablePanel.this;
    mainPanel.addModeListener(new RegistrationModeListener() {
      @Override
      public void modeChanged(Mode oldMode, Mode newMode) {
        if (newMode != Mode.EDIT) {
          return;
        }
        self.setActive(false);
      }
    });
  }

  /**
   * Ajusta listener: {@link Mode#VIEW}
   * 
   * @param mainPanel painel
   */
  private void setViewModeListener(final RegistrationMainPanel<M, I> mainPanel) {
    final RegistrationTablePanel<M, I> self = RegistrationTablePanel.this;
    mainPanel.addModeListener(new RegistrationModeListener() {
      @Override
      public void modeChanged(Mode oldMode, Mode newMode) {
        if (newMode != Mode.VIEW) {
          return;
        }
        self.setActive(true);
      }
    });
  }

  /**
   * Ajuste interno de inatividade.
   * 
   * @param flag indicativo
   */
  private void setActive(final boolean flag) {
    addAction.setEnabled(flag);
    table.setEnabled(flag);
    updateSelectionButtons();
  }

  /**
   * Inicializao do boto: others.
   */
  private void initOthersButton() {
    othersButton.setText(mainPanel.getString("button.others"));
    othersButton.setIcon(RegistrationImages.ICON_DOWN_4);
    othersButton.setHorizontalTextPosition(SwingConstants.RIGHT);
    othersButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        final M selectedObject = getSelectedObject();
        if (selectedObject == null) {
          return;
        }
        final RegistrationMainPanel<M, I> mainPanel = getMainPanel();
        final List<RegistrationObjectAction<M, I>> actions =
          descriptor.getOthersMenuActions(mainPanel, selectedObject);
        if (actions == null || actions.size() == 0) {
          return;
        }
        final JPopupMenu menu = new JPopupMenu();
        for (AbstractAction action : actions) {
          menu.add(action);
        }
        final int y = othersButton.getHeight();
        menu.show(othersButton, 0, y);
      }
    });
  }

  /**
   * Consulta modelo da tabela.
   * 
   * @return modelo.
   */
  private AbstractTableModel getTableModel() {
    return (AbstractTableModel) table.getModel();
  }

  /**
   * 
   */
  public void updateTable() {
    final AbstractTableModel model = getTableModel();
    model.fireTableDataChanged();
    if (model.getRowCount() > 0) {
      table.setRowSelectionInterval(0, 0);
    }
  }

  /**
   * Consulta painel principal.
   * 
   * @return painel
   */
  RegistrationMainPanel<M, I> getMainPanel() {
    return mainPanel;
  }

  /**
   * Consulta  ao: edio
   * 
   * @return ao
   */
  AbstractAction getEditAction() {
    return editAction;
  }

  /**
   * Consulta  ao: edio
   * 
   * @return ao
   */
  AbstractAction getDeleteAction() {
    return delAction;
  }

  /**
   * Consulta o descritor.
   * 
   * @return descritor
   */
  IRegistrationDescriptor<M, I> getDescriptor() {
    return descriptor;
  }

}
