package tecgraf.javautils.gui.print;

import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.print.PageFormat;
import java.awt.print.Paper;
import java.util.ArrayList;
import java.util.Collection;

/**
 * <p>
 * Lista de itens a serem impressos. Este componente permite dividir a pgina em
 * diversos nveis de linhas e colunas.
 * </p>
 * <p>
 * Os componentes de primeiro nvel (nvel zero) desta classe inseridos em uma
 * pgina permitiro dividir a linha em colunas, isto , cada item da lista ser
 * impresso lado a lado. Se, por sua vez, qualquer uma das colunas for outra
 * instncia deste componente (nvel 1), os itens dentro dela sero impressos na
 * vertical, um abaixo do outro. Se qualquer item dentro desta ltima lista for
 * tambm uma instncia deste componente (nvel 2), os itens dentro dele sero
 * impressos na horizontal, e assim por diante, com uma inverso de eixos a cada
 * subnvel.
 * </p>
 * 
 * @author Tecgraf
 */
public class PrintableReportItemArrayList extends
  ArrayList<PrintableReportItem> implements PrintableReportItem {

  /**
   * Nvel de profundidade de aninhamento das listas, iniciando em zero (lista
   * mais externa).
   */
  private int level;

  /**
   * Incrementa o nvel de profundidade de aninhamento das listas que forem
   * includas como filhas desta lista.
   * 
   * @param child elemento sendo includo como filho desta lista.
   */
  private void setChildrenLevel(PrintableReportItem child) {
    if (PrintableReportItemArrayList.class.isInstance(child)) {
      PrintableReportItemArrayList.class.cast(child).level = this.level + 1;
    }
    if (PrintableTable.class.isInstance(child)) {
      // Ver issue JUTILS-59
      PrintableTable.class.cast(child).setAdjustWidth(false);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void add(int index, PrintableReportItem element) {
    setChildrenLevel(element);
    super.add(index, element);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean add(PrintableReportItem e) {
    setChildrenLevel(e);
    return super.add(e);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean addAll(Collection<? extends PrintableReportItem> c) {
    for (PrintableReportItem item : c) {
      setChildrenLevel(item);
    }
    return super.addAll(c);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean addAll(int index, Collection<? extends PrintableReportItem> c) {
    for (PrintableReportItem item : c) {
      setChildrenLevel(item);
    }
    return super.addAll(index, c);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PrintableReportItem set(int index, PrintableReportItem element) {
    if (PrintableReportItemArrayList.class.isInstance(element)) {
      PrintableReportItemArrayList.class.cast(element).level = this.level + 1;
    }
    return super.set(index, element);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initPrinting(PrintConfiguration configuration) {
    // Cada item da lista pode determinar suas prprias configuraes para
    // impresso; no h uma configurao nica da lista.
  }

  /**
   * Imprime a lista de itens.
   * 
   * @param g2 componente grfico.
   * @param format formato da pgina a ser impressa.
   * @param pageIndex ndice da pgina a ser impressa.
   * 
   * @return true se todos os itens couberem na pgina (este mtodo sempre
   *         retorna true).
   */
  @Override
  public boolean print(Graphics2D g2, PageFormat format, int pageIndex) {
    Paper paper = format.getPaper();
    double xBackup = paper.getImageableX();
    double yBackup = paper.getImageableY();
    double widthBackup = paper.getImageableWidth();
    double heightBackup = paper.getImageableHeight();
    Paint paintColorBackup = g2.getPaint();

    for (PrintableReportItem item : this) {
      item.print(g2, format, pageIndex);
      g2.setPaint(paintColorBackup);
      adjustImageableArea(format, item.getWidth(), item.getHeight());
    }
    // Aps a impresso, a rea de impresso deve ser restaurada.
    paper.setImageableArea(xBackup, yBackup, widthBackup, heightBackup);
    format.setPaper(paper);

    // No importa saber se os itens couberam ou no na pgina, pois no vo
    // continuar na prxima linha. Cabe ao desenvolvedor verificar se os itens
    // foram impressos ou no.
    return true;
  }

  /**
   * <p>
   * Este mtodo reduz o tamanho e desloca a origem da rea de desenho dentro da
   * pgina, de acordo com a orientao do papel (PORTRAIT ou LANDSCAPE, a
   * orientao REVERSE_LANDSCAPE no est sendo tratada aqui, assim como no
   * est no mtodo
   * {@link DefaultPrintableReport#adjustPrintable(PageFormat, float, boolean)}
   * e o nvel de profundidade da lista de componentes (as listas podem ser
   * aninhadas umas dentro das outras, com uma inverso de eixos para cada
   * sub-nvel).
   * </p>
   * <p>
   * Para os nveis pares (0, 2, 4,...), a rea de desenho  reduzida em
   * largura, o que permite mltiplas colunas em uma mesma linha. Para os nveis
   * mpares (1, 3, 5, ...), a rea  reduzida em altura, o que permite vrias
   * linhas dentro de uma mesma coluna.
   * 
   * @param format formato da pgina.
   * @param printableWidth largura do componente a ser impresso.
   * @param printableHeight altura do componente a ser impresso.
   */
  private void adjustImageableArea(PageFormat format, float printableWidth,
    float printableHeight) {
    Paper paper = format.getPaper();
    double x = paper.getImageableX();
    double y = paper.getImageableY();
    double width = paper.getImageableWidth();
    double height = paper.getImageableHeight();

    if (level % 2 == 0) {
      // Nveis pares, iniciando em zero
      if (format.getOrientation() == PageFormat.PORTRAIT) {
        x += printableWidth;
        width -= printableWidth;
      }
      if (format.getOrientation() == PageFormat.LANDSCAPE) {
        height -= printableWidth;
      }
    }
    else {
      // Nveis mpares, iniciando em um
      if (format.getOrientation() == PageFormat.LANDSCAPE) {
        x += printableHeight;
        width -= printableHeight;
      }
      if (format.getOrientation() == PageFormat.PORTRAIT) {
        y += printableHeight;
        height -= printableWidth;
      }
    }
    paper.setImageableArea(x, y, width, height);
    format.setPaper(paper);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean simulatePrint(Graphics2D g2, PageFormat format, int pageIndex) {
    // No importa saber se os itens couberam ou no na pgina, pois no vo
    // continuar na prxima linha. Cabe ao desenvolvedor verificar se os itens
    // foram impressos ou no.
    return true;
  }

  /**
   * Obtm a altura da lista a ser impressa, o que depende do nvel de
   * profundidade de aninhamento atual. Se for um nvel par (iniciando em zero),
   * a lista corre na horizontal, ento a altura ser determinada pelo
   * componente mais alto. Se o nvel for mpar, a lista corre na vertical,
   * portanto a altura da lista ser a soma das alturas dos componentes a serem
   * impressos.
   * 
   * @return altura da lista a ser impressa.
   */
  @Override
  public float getHeight() {
    float totalHeight = 0;
    for (PrintableReportItem item : this) {
      if (level % 2 == 0) {
        if (totalHeight < item.getHeight()) {
          totalHeight = item.getHeight();
        }
      }
      else {
        totalHeight += item.getHeight();
      }
    }
    return totalHeight;
  }

  /**
   * Obtm a largura da lista a ser impressa, o que depende do nvel de
   * profundidade de aninhamento atual. Se for um nvel par (iniciando em zero),
   * a lista corre na horizontal, ento a largura ser a soma das alturas dos
   * componentes a serem impressos.Se o nvel for mpar, a lista corre na
   * vertical, portanto a largura da lista determinada pelo componente mais
   * largo.
   * 
   * @return largura da lista a ser impressa.
   */
  @Override
  public float getWidth() {
    float totalWidth = 0;
    for (PrintableReportItem item : this) {
      if (level % 2 == 0) {
        totalWidth += item.getWidth();
      }
      else {
        if (totalWidth < item.getWidth()) {
          totalWidth = item.getWidth();
        }
      }
    }
    return totalWidth;
  }

}
