package tecgraf.javautils.gui.field;

import java.util.Date;
import java.util.Locale;

import javax.swing.text.Document;

import tecgraf.javautils.core.lng.LNG;

/**
 * Campo para Data. A mscara do campo  criada de acordo com o {@link Locale}
 * usado no construtor. O construtor default obtm o {@link Locale} do
 * {@link LNG}. Os <CODE>locales</CODE> permitidos so: pt_BR, es_AR e en_US.
 * Caso o {@link Locale} usado no construtor no seja um dos permitidos, o pt_BR
 * ser usado. O componente possui mtodos para validar e obter os valores da
 * data assim como setar a data atual.
 */
public class DateField extends AbstractRegexField {

  /**
   * Constri um DateField com validao visual desabilitada. O {@link Locale}
   * usado ser obtido do {@link LNG}.
   * 
   */
  public DateField() {
    this(false);
  }

  /**
   * Constri um DateField. O {@link Locale} usado ser obtido do {@link LNG}
   * 
   * @param showValidation exibir efeito visual de validao
   */
  public DateField(boolean showValidation) {
    this(LNG.getLocale(), showValidation);
  }

  /**
   * Constri um DateField com validao visual desabilitada para um determinado
   * {@link Locale}.
   * 
   * @param locale {@link Locale}
   */
  public DateField(Locale locale) {
    this(locale, false);
  }

  /**
   * Constri um DateField para um determinado {@link Locale}.
   * 
   * @param locale {@link Locale}
   * @param showValidation exibir efeito visual de validao
   */
  public DateField(Locale locale, boolean showValidation) {
    super(showValidation);
    if (locale == null) {
      locale = LNG.getLocale();
    }
    setDocument(new DateDocument(locale));    
    super.setLocale(locale);
    setToolTipText(getMask());
  }

  @Override
  public void setLocale(Locale locale) {
    super.setLocale(locale);
    getDateDocument().setLocale(locale);
  }

  @Override
  public boolean isValidValue() {
    return getDateDocument().isValid();
  }

  /**
   * Verifica se uma data est no formato vlido.
   * 
   * @param date String da data
   * @param locale Locale para a data
   * 
   * @return boolean
   * 
   * @deprecated Utilizar o mtodo DateDocument.validateDateFormat
   */
  @Deprecated
  public static boolean validateDate(String date, Locale locale) {
    return DateDocument.validateDateFormat(date, locale);
  }

  /**
   * Verifica se uma data est no formato vlido.
   * 
   * @param date String da data
   * @param locale Locale para a data
   * 
   * @return boolean
   * 
   * @deprecated Utilizar o mtodo DateDocument.validateDateFormat
   */
  @Deprecated
  public static boolean validateDateFormat(String date, Locale locale) {
    return DateDocument.validateDateFormat(date, locale);
  }

  /**
   * Obtm um {@link Date} referente a data ou <CODE>NULL</CODE> caso a data
   * esteja errada.
   * 
   * @param date Data em String
   * @param locale Locale em que se encontra a data
   * 
   * @return array de interger com a data
   * 
   * @deprecated Utilizar o mtodo DateDocument.getDate
   */
  @Deprecated
  public static Date getDate(String date, Locale locale) {
    return DateDocument.getDate(date, locale);
  }

  /**
   * Obtm o dia ou NULL caso a data no esteja vlida.
   * 
   * @return Integer
   */
  public Integer getDay() {
    return getDateDocument().getDay();
  }

  /**
   * Obtm o ms ou NULL caso a data no esteja vlida.
   * 
   * @return Integer
   */
  public Integer getMonth() {
    return getDateDocument().getMonth();
  }

  /**
   * Obtm o ano ou NULL caso a data no esteja vlida.
   * 
   * @return Integer
   */
  public Integer getYear() {
    return getDateDocument().getYear();
  }

  /**
   * Obtm a data em milisegundos ou NULL caso data seja invlida.
   * 
   * @return Long
   */
  public Long getDate() {
    return getDateDocument().getDate();
  }

  /**
   * Ativa o auto complemento.
   * 
   * @param complete
   */
  public void setAutoComplete(boolean complete) {
    if (complete) {
      getDateDocument().rebuildCompleText();
    }
    else {
      getDateDocument().setCompleteText(null);
    }
  }

  /**
   * Seta o campo com uma data.
   * 
   * @param day Dia
   * @param month Ms
   * @param year Ano
   */
  public void setDate(int day, int month, int year) {
    getDateDocument().setDate(day, month, year);
  }

  /**
   * Obtm uma {@link String} de data formatada
   * 
   * @param locale {@link Locale} usado para formatar a data
   * @param date Data em milisegundos
   * @return {@link String}
   * 
   * @deprecated Utilizar o mtodo DateDocument.toString
   */
  @Deprecated
  public static String toString(Locale locale, long date) {
    return DateDocument.toString(locale, date);
  }

  /**
   * Seta o campo com uma data em milisegundos
   * 
   * @param lngDate
   */
  public void setDate(long lngDate) {
    getDateDocument().setDate(lngDate);
  }

  /**
   * Seta o campo com a data atual
   */
  public void setToCurrentDate() {
    getDateDocument().setToCurrentDate();
  }

  /**
   * Obtm a traduo {@link LNG} para a mscara usada no campo
   * 
   * @return {@link String}
   */
  public String getMask() {
    String lng = "tecgraf.javautils.gui.field.datefield.mask.";
    lng += getDateDocument().getFormat().name().toLowerCase();
    return LNG.get(lng);
  }

  /**
   * Obtm o texto do campo.
   * 
   * @return {@link String}
   */
  public String toString() {
    return getText();
  }

  /**
   * Obtm o modelo
   * 
   * @return {@link DateDocument}
   */
  private DateDocument getDateDocument() {
    return (DateDocument) getDocument();
  }

  @Override
  public void setDocument(Document doc) {
    if (doc != null && doc instanceof DateDocument) {
      super.setDocument(doc);
    }
  }

  /**
   * Adiciona um listener de data
   * 
   * @param listener {@link DateListener}
   */
  public void addDateListener(DateListener listener) {
    getDateDocument().addDateListener(listener);
  }

  /**
   * Remove um listener de data
   * 
   * @param listener {@link DateListener}
   */
  public void removeDateListener(DateListener listener) {
    getDateDocument().removeDateListener(listener);
  }

}
