
package tecgraf.javautils.gui.wizard;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 *
 * Representa um grupo de passos. A descrio definida no grupo de passos, 
 * usada para exibio na barra lateral do Wizard.
 *
 * @author Tecgraf/PUC-Rio
 */
public final class StepGroup {
  /**
   * A descrio do grupo.
   */
  private String description;

  /**
   * Um conjunto com todos os passos pertencentes ao grupo.
   */
  private Set<Step> stepSet;

  /**
   * Um mapa ligando a classe do passo ao passo.
   */
  private Map<Class<? extends Step>, Step> stepMap;

  /**
   * Cria um grupo de passos.
   *
   * @param description A descrio do grupo.
   *
   * @throws IllegalArgumentException Caso a descrio seja nula.
   */
  public StepGroup(String description) {
    if (description == null) {
      throw new IllegalArgumentException("A descrio no pode ser nula.");
    }
    this.stepSet = new HashSet<Step>();
    this.stepMap = new HashMap<Class<? extends Step>, Step>();
    this.description = description;
  }

  /**
   * Obtm a descrio do grupo.
   *
   * @return A descrio do grupo.
   */
  public String getDescription() {
    return this.description;
  }

  /**
   * Adiciona um passo ao grupo.
   *
   * @param step O passo a ser adicionado.
   */
  public void add(Step step) {
    final Class<? extends Step> clazz = step.getClass();
    this.stepMap.put(clazz, step);
    this.stepSet.add(step);
  }

  /**
   * Obtm um passo a partir da sua classe.
   *
   * @param stepClass A classe do passo.
   *
   * @return O passo ou null, caso no exista passo cadastrado para a classe
   *         indicada.
   */
  public Step get(Class<? extends Step> stepClass) {
    return this.stepMap.get(stepClass);
  }

  /**
   * Obtm um conjunto com todos os passos pertencentes ao grupo.
   *
   * @return Um conjunto com todos os passos pertencentes ao grupo.
   */
  public Set<? extends Step> getAll() {
    return Collections.unmodifiableSet(this.stepSet);
  }

  /**
   * Verifica se o grupo contm o passo especificado.
   *
   * @param step O passo a ser verificado.
   *
   * @return true, caso o passo esteja presente no grupo, ou false, caso
   *         contrrio.
   */
  public boolean contains(Step step) {
    return this.stepSet.contains(step);
  }
}
