package tecgraf.javautils.gui.imagepanel;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;

import javax.swing.JPanel;

/**
 * Painel que exibe a imagem com escala para ser exibida.
 *
 * @author Tecgraf/PUC-Rio
 */
public class ImagePanel extends JPanel {

  /**
   * Colro a ser feita no repaint quando no houver imagem.
   */
  private static final Color NO_COLOR = new JPanel().getBackground();

  /**
   * Tipos de alinhamentos horiziontais.
   *
   * @author Tecgraf/PUC-Rio
   */
  public enum HorAlign {
    /** Centro */
    CENTER,
    /** Oeste */
    WEST,
    /** Leste */
    EAST,
  }

  /**
   * Tipos de alinhamentos verticais.
   *
   * @author Tecgraf/PUC-Rio
   */
  public enum VerAlign {
    /** Centro */
    CENTER,

    /** Sul */
    SOUTH,

    /** Norte */
    NORTH,
  }

  /**
   * Imagem.
   */
  private Image image;

  /**
   * Largura preferencial do elemento.
   */
  private int preferredWidth = 0;

  /**
   * Alinhamento horizontal
   */
  private HorAlign horAlign = HorAlign.CENTER;

  /**
   * Alinhamento vertical
   */
  private VerAlign verAlign = VerAlign.CENTER;

  /**
   * Ajuste da imagem.
   *
   * @param image imagem.
   * @return o painel.
   */
  public ImagePanel setImage(Image image) {
    this.image = image;
    preferredWidth = (image == null ? 0 : 200);
    repaint();
    return this;
  }

  /**
   * Comsulta a imagem corrente.
   *
   * @return a imagem
   */
  public Image getImage() {
    return this.image;
  }

  /**
   * Consulta alinhamento.
   *
   * @return alinhamento.
   */
  public HorAlign getHorAlign() {
    return horAlign;
  }

  /**
   * Ajuste de alinhamento.
   *
   * @param horAlign alinhamento
   * @return o painel.
   */
  public ImagePanel setHorAlign(HorAlign horAlign) {
    this.horAlign = horAlign;
    repaint();
    return this;
  }

  /**
   * Consulta alinhamento.
   *
   * @return alinhamento.
   */
  public VerAlign getVerAlign() {
    return verAlign;
  }

  /**
   * Ajuste de alinhamento.
   *
   * @param verAlign alinhamento
   * @return o painel.
   */
  public ImagePanel setVerAlign(VerAlign verAlign) {
    this.verAlign = verAlign;
    repaint();
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Dimension getPreferredSize() {
    if (image == null) {
      return super.getPreferredSize();
    }
    else {
      int w = Math.round(preferredWidth);
      return getDimensionOnWidth(w);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Dimension getMinimumSize() {
    if (image == null) {
      return super.getMinimumSize();
    }
    else {
      int w = (int) Math.round(preferredWidth * 0.7);
      return getDimensionOnWidth(w);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Dimension getMaximumSize() {
    if (image == null) {
      return super.getMaximumSize();
    }
    else {
      int w = (int) Math.round(preferredWidth * 1.3);
      return getDimensionOnWidth(w);
    }
  }

  /**
   * Monta um objeto de dimenso com base em um alargura predefinida.
   *
   * @param w largura
   * @return dimenso
   */
  private Dimension getDimensionOnWidth(int w) {
    if (image == null) {
      return new Dimension(w, w);
    }
    double ratio = ((double) image.getWidth(null)) / image.getHeight(null);
    int h = (int) Math.round(w / ratio);
    return new Dimension(w, h);
  }

  /**
   * Monta um fator de escala.
   *
   * @param iMasterSize master
   * @param iTargetSize destino.
   * @return fator
   */
  public double getScaleFactor(int iMasterSize, int iTargetSize) {
    double dScale = 1;
    if (iMasterSize > iTargetSize) {
      dScale = (double) iTargetSize / (double) iMasterSize;
    }
    else {
      dScale = (double) iTargetSize / (double) iMasterSize;
    }
    return dScale;
  }

  /**
   * Monta a escala para fazer fit.
   *
   * @param original escala original
   * @param toFit escal a ser enquadrada.
   * @return a escala
   */
  public double getScaleFactorToFit(Dimension original, Dimension toFit) {
    double dScale = 1d;
    if (original != null && toFit != null) {
      double dScaleWidth = getScaleFactor(original.width, toFit.width);
      double dScaleHeight = getScaleFactor(original.height, toFit.height);
      dScale = Math.min(dScaleHeight, dScaleWidth);
    }
    return dScale;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void paintComponent(Graphics g) {
    if (image == null) {
      g.setColor(NO_COLOR);
      g.fillRect(0, 0, getWidth(), getHeight());
      return;
    }

    Color bgColor = getBackground();
    if (bgColor == null) {
      bgColor = NO_COLOR;
    }
    g.setColor(bgColor);
    g.fillRect(0, 0, getWidth(), getHeight());

    final int imgW = image.getWidth(null);
    final int imgH = image.getHeight(null);
    final Dimension original = new Dimension(imgW, imgH);

    double scaleFactor = Math.min(1d, getScaleFactorToFit(original, getSize()));
    int scaleWidth = (int) Math.round(imgW * scaleFactor);
    int scaleHeight = (int) Math.round(imgH * scaleFactor);

    Image scaled = image.getScaledInstance(scaleWidth, scaleHeight,
      Image.SCALE_SMOOTH);

    int width = getWidth() - 1;
    int height = getHeight() - 1;

    final double ha = getHorFactor();
    final double va = getVerFactor();

    int x = (int) Math.round((width - scaled.getWidth(this)) * ha);
    int y = (int) Math.round((height - scaled.getHeight(this)) * va);

    g.drawImage(scaled, x, y, this);
  }

  /**
   * Consulta o fator horizontal.
   *
   * @return fator
   */
  private double getHorFactor() {
    switch (horAlign) {
      case WEST:
        return 0.0;
      case CENTER:
        return 0.5;
      case EAST:
        return 1.0;
      default:
        throw new IllegalStateException("Invalid hor alignment");
    }
  }

  /**
   * Consulta o fator vertical.
   *
   * @return fator
   */
  private double getVerFactor() {
    switch (verAlign) {
      case NORTH:
        return 0.0;
      case CENTER:
        return 0.5;
      case SOUTH:
        return 1.0;
      default:
        throw new IllegalStateException("Invalid ver alignment");
    }
  }

}