/**
 * $Id: RespectEnabledStateCellRenderer.java 86326 2008-12-07 15:03:41Z costa $
 */

package tecgraf.javautils.gui.table;

import java.awt.Component;

import javax.swing.JTable;
import javax.swing.table.TableCellRenderer;

/**
 * Renderizador de clulas da tabela que honra o estado da tabela
 * (habilitado/desabilitado), garantindo que todas as clulas estejam no mesmo
 * estado. Isto faz com que tabelas desabilitadas realmente paream
 * desabilitadas.
 * <p>
 * Este renderizador recebe o renderizador original da clula e delega para o
 * mesmo o request para obter o componente. Feito isto, apenas garante que o
 * status do componente retornado  o mesmo da tabela.
 * <p>
 * Para us-lo, redefina o mtodo
 * <code>getCellRenderer(int row, int column)</code> da sua tabela para
 * 
 * <pre>
 * return new RespectEnabledStateCellRenderer(super.getCellRenderer(row, column));
 * </pre>
 * 
 * Obtido de: http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4795987
 * 
 * @author Tecgraf
 */
public class RespectEnabledStateCellRenderer implements TableCellRenderer {
  /**
   * Renderizador original, para o qual os requests sero repassados.
   */
  protected final TableCellRenderer parentRenderer;
  /**
   * Flag que controla o look & feel das clulas quando a tabela est
   * desabilitada. Se for igual a <code>true</code>, as clulas tero aparncia
   * de desabilitadas; seno, a tabela ter o comportamento default (manter sua
   * aparncia mas o usurio no poder interagir com as clulas).
   */
  private final boolean useDisabledLook;

  /**
   * Construtor. Clulas sero desabilitadas quando a tabela estiver
   * desabilitada.
   * 
   * @see #RespectEnabledStateCellRenderer(TableCellRenderer, boolean)
   * 
   * @param parentRenderer - renderizador original da clula
   */
  public RespectEnabledStateCellRenderer(TableCellRenderer parentRenderer) {
    this(parentRenderer, true);
  }

  /**
   * Construtor.
   * 
   * @param parentRenderer - renderizador original da clula
   * @param useDisabledLook - se igual a <code>true</code>, as clulas sero
   *        desabilitadas quando a tabela estiver desabilitada
   */
  public RespectEnabledStateCellRenderer(TableCellRenderer parentRenderer,
    boolean useDisabledLook) {
    if (!validateRenderer(parentRenderer)) {
      throw new IllegalArgumentException("Componente no suportado");
    }
    this.parentRenderer = parentRenderer;
    this.useDisabledLook = useDisabledLook;
  }

  /**
   * Valida o renderizador original da clula. A implementao default sempre
   * retorna <code>true</code>.
   * 
   * @param renderer - renderizador original da clula
   * @return true se o renderizador  vlido
   */
  protected boolean validateRenderer(TableCellRenderer renderer) {
    // implementao default no impe restries
    return true;
  }

  /**
   * Converte o valor antes da consulta ao renderizador original da clula. A
   * implementao default retorna o prprio valor, mas subclasses podem
   * redefinir este comportamento se necessrio.
   * 
   * @param value - valor original da clula
   * @param row - linha da clula
   * @param column - coluna da clula
   * @return o valor convertido
   */
  protected Object convertValue(Object value, int row, int column) {
    // implementao default no aplica converses
    return value;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getTableCellRendererComponent(JTable table, Object value,
    boolean isSelected, boolean hasFocus, int row, int column) {
    Component component =
      parentRenderer.getTableCellRendererComponent(table, convertValue(value,
        row, column), isSelected, hasFocus, row, column);
    if (useDisabledLook) {
      boolean tableIsEnabled = table.isEnabled();
      if (tableIsEnabled != component.isEnabled()) {
        component.setEnabled(tableIsEnabled);
      }
    }
    return component;
  }
}