/*
 * $Id: MenuButton.java 114628 2011-01-17 19:11:37Z costa $
 */

package tecgraf.javautils.gui;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JToggleButton;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;

/**
 * Boto com menu popup associado. O popup pode ser exibido  direita ou abaixo
 * do boto. Botes que possuem apenas o label (i.e. no tm imagem associada)
 * so exibidos com uma seta indicando a direo do menu. As margens laterais
 * so reduzidas para tornar o visual mais compacto.
 * <p>
 * Os itens do menu so adicionados da mesma forma que a um menu "comum" (
 * {@link #add(Action)}, {@link #add(JMenuItem)}, {@link #addSeparator()}).
 * 
 * @see MenuButtonSample
 */
public class MenuButton extends JToggleButton {

  /**
   * Borda horizontal.
   */
  private static final int HORIZONTAL_GAP = 5;

  /**
   * Borda vertical.
   */
  private static final int VERTICAL_GAP = 2;

  /**
   * <code>PopupPosition</code> representa a posio do menu popup em relao ao
   * boto. O boto principal possui uma imagem associada de acordo com esta
   * posio.
   */
  public enum PopupPosition {
    /**
     * menu popup ser exibido acima do boto
     */
    TOP(GUIResources.BUTTON_ARROW_UP_ICON),
    /**
     * menu popup ser exibido  esquerda do boto
     */
    LEFT(GUIResources.BUTTON_ARROW_LEFT_ICON),
    /**
     * menu popup ser exibido abaixo do boto
     */
    BOTTOM(GUIResources.BUTTON_ARROW_DOWN_ICON),
    /**
     * menu popup ser exibido do lado direito do boto
     */
    RIGHT(GUIResources.BUTTON_ARROW_RIGHT_ICON);

    /**
     * Imagem associada  direo do popup.
     */
    Image arrowImage;
    /**
     * Altura da seta.
     */
    int arrowHeight;
    /**
     * Largura da seta.
     */
    int arrowWidth;

    /**
     * Construtor.
     * 
     * @param icon cone associado  direo do popup
     */
    private PopupPosition(ImageIcon icon) {
      arrowImage = icon.getImage();
      arrowWidth = arrowImage.getWidth(null);
      arrowHeight = arrowImage.getHeight(null);
    }
  }

  /** Posio do menu popup em relao ao boto */
  private PopupPosition popupPos;

  /** Menu popup */
  private JPopupMenu popup;

  /**
   * Distncia da margem esquerda do cone da seta para a margem direita do
   * boto.
   */
  private int arrowXoffset;

  /**
   * Flag que indica se o popup deve ser alinhado  direita do boto.
   */
  private boolean alignRight;

  /**
   * Construtor para criar um boto apenas com texto (sem cone). A seta "v" ou
   * {@literal ">"}  acrescentada automaticamente.
   * <p>
   * O popup  alinhado  esquerda do boto.
   * 
   * @param text rtulo que ser exibido no boto
   * @param position posio do menu popup em relao ao boto principal
   */
  public MenuButton(String text, PopupPosition position) {
    this(text, position, false);
  }

  /**
   * Construtor para criar um boto apenas com texto (sem cone). A seta "v" ou
   * {@literal ">"}  acrescentada automaticamente.
   * 
   * @param text rtulo que ser exibido no boto
   * @param position posio do menu popup em relao ao boto principal
   * @param alignRight <code>true</code> se o popup deve ser alinhado  direita
   *        do boto (ignorado para alinhamento <code>LEFT</code> ou
   *        <code>RIGHT</code>).
   */
  public MenuButton(String text, PopupPosition position, boolean alignRight) {
    super(text);
    this.alignRight = alignRight;
    init(position);
  }

  /**
   * Construtor para criar um boto sem texto, apenas com uma imagem.
   * <p>
   * <b>ATENO:</b> a seta "v" ou {@literal ">"} no  acrescentada automaticamente (se
   * for necessrio, a imagem deve incluir a seta)
   * <p>
   * O popup  alinhado  esquerda do boto.
   * 
   * @param icon cone
   * @param position posio do menu popup em relao ao boto principal
   */
  public MenuButton(Icon icon, PopupPosition position) {
    this(icon, position, false);
  }

  /**
   * Construtor para criar um boto sem texto, apenas com uma imagem.
   * <p>
   * <b>ATENO:</b> a seta "v" ou {@literal ">"} no  acrescentada automaticamente (se
   * for necessrio, a imagem deve incluir a seta)
   * 
   * @param icon cone
   * @param position posio do menu popup em relao ao boto principal
   * @param alignRight <code>true</code> se o popup deve ser alinhado  direita
   *        do boto (ignorado para alinhamento <code>LEFT</code> ou
   *        <code>RIGHT</code>).
   */
  public MenuButton(Icon icon, PopupPosition position, boolean alignRight) {
    super(icon);
    this.alignRight = alignRight;
    init(position);
  }

  /**
   * Inicializaes. Define a posio, cria o popup etc.
   * 
   * @param position posio do popup com relao ao boto
   */
  private void init(PopupPosition position) {
    popupPos = position;
    popup = new PopupMenu(this);
    setHorizontalTextPosition(AbstractButton.LEADING);
    addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        /*
         * passamos a posio como (0, 0) pois ela ser redefinida mais adiante
         */
        popup.show(MenuButton.this, 0, 0);
      }
    });
    // 5 da borda + 2 de ajuste fino
    arrowXoffset = popupPos.arrowWidth + HORIZONTAL_GAP + 2;
    if (popupPos == PopupPosition.LEFT) {
      int leftMargin = getIconTextGap() + arrowXoffset;
      setMargin(new Insets(VERTICAL_GAP, leftMargin, VERTICAL_GAP,
        HORIZONTAL_GAP));
    }
    else {
      int rightMargin = getIconTextGap() + arrowXoffset;
      setMargin(new Insets(VERTICAL_GAP, HORIZONTAL_GAP, VERTICAL_GAP,
        rightMargin));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void paintComponent(Graphics g) {
    super.paintComponent(g);
    int arrowXpos;
    if (popupPos == PopupPosition.LEFT) {
      arrowXpos = HORIZONTAL_GAP + 2;
    }
    else {
      arrowXpos = getWidth() - arrowXoffset;
    }
    int arrowYpos = (getHeight() / 2) - (popupPos.arrowHeight / 2);
    g.drawImage(popupPos.arrowImage, arrowXpos, arrowYpos, null);
  }

  /**
   * Insere no fim do menuButton o item de menu que foi especificado.
   * 
   * @param menuItem o item a ser adicionado ao menu
   * @return o item de menu adicionado
   */
  public JMenuItem add(JMenuItem menuItem) {
    return popup.add(menuItem);
  }

  /**
   * Insere no fim do menuButton um novo item de menu, com a ao especificada.
   * 
   * @param action a ao a ser adicionada ao menu
   * @return o item de menu adicionado
   */
  public JMenuItem add(Action action) {
    return popup.add(action);
  }

  /**
   * Insere no fim do menuButton um novo separador.
   * 
   * @return o prprio boto, para encadeamento
   */
  public MenuButton addSeparator() {
    popup.addSeparator();
    return this;
  }

  /**
   * Indica se o popup deve ser alinhado  direita do boto.
   * 
   * @param alignRight <code>true</code> se o popup deve ser alinhado  direita
   *        do boto
   * @return o prprio boto, para encadeamento
   */
  public MenuButton setAlignRight(boolean alignRight) {
    this.alignRight = alignRight;
    return this;
  }

  /**
   * Menu popup usado.
   */
  private static class PopupMenu extends JPopupMenu {

    /**
     * Construtor.
     * 
     * @param button boto que servir de referncia para o menu
     */
    public PopupMenu(final MenuButton button) {
      addPopupMenuListener(new PopupMenuListener() {

        @Override
        public void popupMenuCanceled(PopupMenuEvent e) {
          // vazio
        }

        @Override
        public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
          button.setSelected(false);
        }

        @Override
        public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
          /*
           * redefinimos a posio do popup em funo do seu tipo (TOP, LEFT
           * etc.)
           */
          Point buttonPos = button.getLocationOnScreen();
          Dimension size = getPreferredSize();
          int x = (int) buttonPos.getX();
          int y = (int) buttonPos.getY();
          int bw = button.getWidth();
          int bh = button.getHeight();
          switch (button.popupPos) {
            case TOP:
              if (button.alignRight) {
                x += bw - size.width;
              }
              y -= size.height;
              break;

            case LEFT:
              x -= size.width;
              break;

            case BOTTOM:
              if (button.alignRight) {
                x += bw - size.width;
              }
              y += bh;
              break;

            case RIGHT:
              x += bw;
              break;

            default:
              break;
          }
          setLocation(x, y);
        }
      });
    }
  }
}
