/**
 * $Id$
 */
package tecgraf.javautils.gui.panel;

import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.KeyEventDispatcher;
import java.awt.KeyboardFocusManager;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.Timer;
import java.util.TimerTask;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.border.Border;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIResources;
import tecgraf.javautils.gui.SwingThreadDispatcher;

/**
 * Classe que desenha um painel com um campo de texto e boto para que seja
 * utilizada uma filtragem a partir do que o usurio digitar no campo.
 * 
 * @see FilteredView
 * @author Tecgraf
 */
public class TextFieldFilterPanel extends JPanel {

  /**
   * Tempo em que aps o ltimo caracter digitado, o campo de texto deve ficar
   * ocioso para que comece a filtragem.
   */
  private static final long TEXTFIELD_COUNT_DOWN = 1;

  /** Caixa de texto para o filtro por nome do arquivo. */
  protected JTextField filterTextField;

  /** Boto para limpar a caixa de texto contendo o critrio de filtro. */
  protected JButton clearFieldButton;

  /** Classe de visualizao que dever tratar a filtragem */
  private FilteredView fView;

  /** Dispatcher para eventos de teclado que atua como filtro de input. */
  private KeyEventDispatcher filterInputDispatcher;

  /** Sinaliza se a borda para o painel de filtragem ser exibida. */
  private boolean showBorder;

  /** Label associado a caixa de filtragem ou a borda de ttulo. */
  private String textLabel;

  /**
   * Construtor que usa imagem padro para boto de limpar
   * 
   * @param fView classe de visualizao que dever tratar a filtragem
   */
  public TextFieldFilterPanel(FilteredView fView) {
    this(fView, GUIResources.BUTTON_CLEAR_ICON);
  }

  /**
   * Construtor que permite definir outra imagem para o boto de limpar
   * 
   * @param fView classe de visualizao que dever tratar a filtragem
   * @param clearButtonIcon imagem do boto de limpar
   */
  public TextFieldFilterPanel(FilteredView fView, ImageIcon clearButtonIcon) {
    this(fView, clearButtonIcon, true, LNG.get("TreeFilterPanel.border.title"));
    filterTextField.setToolTipText(LNG.get("TreeFilterPanel.filterTextField.tooltip"));
  }

  /**
   * Construtor que permite definir outra imagem para o boto de limpar
   * 
   * @param fView classe de visualizao que dever tratar a filtragem
   * @param clearButtonIcon imagem do boto de limpar
   * @param showBorder sinaliza se borda para painel ser exibida
   * @param textLabel texto para o label da caixa de filtragem ou para ttulo da
   *        borda de todo painel
   */
  public TextFieldFilterPanel(FilteredView fView, ImageIcon clearButtonIcon, boolean showBorder, String textLabel) {
    super(new GridBagLayout());
    this.fView = fView;
    this.showBorder = showBorder;
    this.textLabel = textLabel;
    this.filterTextField = new JTextField(10);
    createfilterInputDipatcher(filterTextField);

    filterTextField.getDocument().addDocumentListener(new DocumentListener() {
      @Override
      public void changedUpdate(DocumentEvent e) {
        update();
      }

      @Override
      public void removeUpdate(DocumentEvent e) {
        update();
      }

      @Override
      public void insertUpdate(DocumentEvent e) {
        update();
      }

      public void update() {
        clearFieldButton.setEnabled(filterTextField.getText().length() > 0);
      }
    });
    filterTextField.addKeyListener(new KeyAdapter() {
      Timer timer = new Timer();

      @Override
      public void keyReleased(KeyEvent e) {
        timer.cancel();
        timer = new Timer();
        timer.schedule(new TimerTask() {
          @Override
          public void run() {
            SwingThreadDispatcher.invokeLater(new Runnable() {
              @Override
              public void run() {
                filter();
              }
            });
          }
        }, TEXTFIELD_COUNT_DOWN * 1000);
      }
    });

    clearFieldButton = new JButton(clearButtonIcon);
    clearFieldButton.setMargin(new Insets(0, 0, 0, 0));
    clearFieldButton.setEnabled(false);
    clearFieldButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        filterTextField.setText("");
        clearFieldButton.setEnabled(false);
        filterTextField.requestFocusInWindow();
        filter();
      }
    });

    JPanel entryPanel = createEntryPanel();
    this.add(entryPanel, new GBC(0, 0).both());

  }

  /**
   * Cria o painel para entrada de dados da filtragem. Utiliza componentes de
   * texto e boto previamente criados.
   * 
   * @return painel para entrada de dados da filtragem
   */
  private JPanel createEntryPanel() {
    JPanel entryPanel = new JPanel(new GridBagLayout());
    Insets margins = new Insets(5, 5, 5, 5);
    int gridY = 0;

    // se houver texto, ser usado na borda ou no label, conforme escolhido no construtor
    if (textLabel != null) {
      if (showBorder) {
        Border border = BorderFactory.createTitledBorder(textLabel);
        entryPanel.setBorder(border);
      }
      else {
        margins = new Insets(5, 5, 5, 0);
        entryPanel.add(new JLabel(textLabel), new GBC(gridY++, 0).insets(5, 0, 5, 0).west());
      }
    }

    entryPanel.add(filterTextField, new GBC(gridY++, 0).insets(margins).west().weightx(1).horizontal());
    entryPanel.add(clearFieldButton, new GBC(gridY++, 0).insets(margins).west());
    return entryPanel;
  }

  /**
   * Filtra a rvore de projetos, considerando o filtro para nome digitado pelo
   * usurio na caixa de texto e a seleo de tipo de arquivo.
   */
  private void filter() {
    fView.filter(filterTextField.getText());
  }

  /**
   * Adiciona o dispatcher de filtro de input ao KeyboardFocusManager.
   */
  private void addFilterInputDispatcher() {
    KeyboardFocusManager.getCurrentKeyboardFocusManager().addKeyEventDispatcher(filterInputDispatcher);
  }

  /**
   * Remove o dispatcher de filtro de input do KeyboardFocusManager.
   */
  private void removeFilterInputDispatcher() {
    KeyboardFocusManager.getCurrentKeyboardFocusManager().removeKeyEventDispatcher(filterInputDispatcher);
  }

  /**
   * Cria um dispatcher para eventos de teclado que atue como um filtro de
   * entrada para o componente especificado. O usurio no conseguir digitar
   * nada alm dos caracteres permitidos.
   * 
   * @param cmp componente cuja entrada ser filtrada.
   */
  private void createfilterInputDipatcher(final Component cmp) {
    filterInputDispatcher = new KeyEventDispatcher() {
      /**
       * Desconsidera os eventos de teclado que forem dirigidos para um
       * determinado componente e no forem considerados "vlidos".
       * 
       * @param e informaes sobre o evento ocorrido.
       * 
       * @return true se o evento puder ser repassado, false se ele tiver que
       *         ser descartado.
       */
      @Override
      public boolean dispatchKeyEvent(KeyEvent e) {
        boolean discardEvent = false;
        if (e.getID() == KeyEvent.KEY_TYPED) {
          if (cmp.isFocusOwner() && !isValid(e.getKeyChar())) {
            discardEvent = true;
          }
        }
        return discardEvent;
      }

      /**
       * Verifica se o caracter especificado  vlido. So considerados
       * caracteres vlidos dgitos, letras, ".", "_", "?", "*" e "$".
       * 
       * @param keyChar caracter a ser verificado.
       * 
       * @return true se o caracter for vlido.
       */
      private boolean isValid(char keyChar) {
        if (Character.isLetterOrDigit(keyChar)) {
          return true;
        }
        else if ("?*$._".indexOf(keyChar) >= 0) {
          return true;
        }
        return false;
      }
    };
  }

  /**
   * <p>
   * Esconde ou exibe o painel, de acordo com o parmetro especificado.
   * </p>
   * <p>
   * Quando exibe, transfere o foco para a caixa de texto. Quando esconde,
   * remove o filtro.
   * </p>
   */
  @Override
  public void setVisible(boolean visible) {
    if (visible == isVisible()) {
      return;
    }
    super.setVisible(visible);
    if (visible) {
      filterTextField.requestFocusInWindow();
      addFilterInputDispatcher();
    }
    else {
      /*
       * como estamos escondendo o painel, devemos desativar o filtro
       * (retornando-o para ALL_FILES caso esteja com outro valor)
       */
      reset();
      removeFilterInputDispatcher();
    }
  }

  /**
   * Pe o filtro no estado inicial, no qual aceita todos os arquivos.
   */
  public void reset() {
    filterTextField.setText(null);
  }
}
