/*
 * $Id: ContainerSelection.java 150399 2014-02-26 19:08:39Z oikawa $
 */

package tecgraf.javautils.gui.selector;

import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIResources;
import tecgraf.javautils.gui.panel.FilteredView;
import tecgraf.javautils.gui.panel.TextFieldFilterPanel;
import tecgraf.javautils.gui.table.ObjectTableProvider;

/**
 * A classe <code>ContainerSelection</code> modela um componente que permite
 * escolher entre um conjunto de tens disponveis, usando um segundo conjunto
 * que guarda os tens escolhidos. O componente oferece botes que permitem
 * movimentar os tens entre os conjuntos de disponveis e escolhidos.
 * 
 * @param <T> .
 * 
 * @see ContainerSelectionSample
 * 
 * @author Julia
 */
public class ContainerSelection<T> implements FilteredView {
  /** Prefixo para as chaves de i18n. */
  private static final String LNG_KEY_PREFIX = ContainerSelection.class.getName() + ".";

  /** lista padro de tens selecionados */
  private Collection<T> defaultSelectedItems;

  /** O boto que permite restaurar a lista padro de tens selecionados. */
  private JButton restoreDefaultsButton;

  /** Indicador se o boto restaurar ser usado. */
  private boolean hasRestoreDefaultsButton;

  /**
   * O boto que permite remover todos os elemento do conjunto de disponveis e
   * adicion-los no conjunto de elementos escolhidos.
   */
  private JButton addAllButton;

  /**
   * O boto que permite remover um elemento do conjunto de disponveis e
   * adicion-lo no conjunto de elementos escolhidos.
   */
  private JButton addButton;

  /** O painel que representa esse componente. */
  private JPanel panel;

  /** O painel que contm os botes. */
  private JPanel buttonPanel;

  /** A posio vertical do ltimo boto inserido no painel de botes. */
  int gridy = 0;

  /**
   * O boto que permite remover todos os elemento do conjunto de escolhidos e
   * adicion-los no conjunto de elementos disponveis.
   */
  private JButton removeAllButton;

  /**
   * O boto que permite remover um elemento do conjunto de escolhidos e
   * adicion-lo no conjunto de elementos disponveis.
   */
  private JButton removeButton;

  /** O componente que representa o conjunto de elementos escolhidos. */
  private ItemContainer<T> selectedContainer;

  /** O componente que representa o conjunto de elementos disponveis. */
  private ItemContainer<T> availableContainer;

  /** Ttulo para o cojunto de elementos escolhidos. */
  private String selectedText;

  /** Ttulo para o conjunto de elementos disponveis. */
  private String availableText;

  /**
   * Indica se existe ttulo para o conjunto de elementos disponneis OU para o
   * conjunto dos elementos escolhidos. Rtulos so opcionais, podem existir de
   * forma independente, mantendo o alinhamento com seu container associado.
   */
  private boolean hasContainerTitle;

  /**
   * Indica se o boto de adicionar todos os elementos deve ser exibido ou no.
   * Default: true
   */
  private boolean hasAddAllButton = true;

  /**
   * Indica se o boto de remover todos os elementos selecionados deve ser
   * exibido ou no. Default: true
   */
  private boolean hasRemoveAllButton = true;

  /**
   * Indica se o texto dos botes deve ser exibido ou no. Default: true
   */
  private boolean showButtonsText = true;

  /** Posio onde os botes devem ficar no painel, NORTH, SOUTH ou CENTER */
  private int buttonPanelAnchor = SwingConstants.CENTER;

  /** Lista contendo o listeners de seleo registrados. */
  private List<ContainerSelectionListener> containerSelectionListeners;

  /**
   * Painel com o filtro a ser visualizado apenas se o filtro tiver que ser
   * mostrado
   */
  private TextFieldFilterPanel filterPanel;

  /**
   * Enumerao usada para especificar qual dos containers deve ser ordenvel.
   */
  public enum SortableContainer {
    /**
     * Apenas o container da esquerda deve ser ordenvel.
     */
    LEFT_ONLY,
    /**
     * Apenas o container da direita deve ser ordenvel.
     */
    RIGHT_ONLY,
    /**
     * Ambos os containers devem ser ordenveis.
     */
    BOTH,
    /**
     * Nenhum container deve ser ordenvel.
     */
    NONE
  };

  /**
   * Dispara um evento de modificao do container de seleo.
   */
  protected void fireContainerSelectionEvent() {
    for (ContainerSelectionListener csl : containerSelectionListeners) {
      csl.containerChanged();
    }
  }

  /**
   * Adiciona um listener de modificao do container de seleo. Esse listener
   *  notificado sempre que algum elemento  colocado ou removido do container
   * com os elementos selecionados.
   * 
   * @param l O listener de alterao do contedo do container de seleo.
   */
  public void addContainerSelectionListener(ContainerSelectionListener l) {
    containerSelectionListeners.add(l);
  }

  /**
   * Remove um listener de modificao do container de seleo. Esse listener 
   * notificado sempre que algum elemento  colocado ou removido do container
   * com os elementos selecionados.
   * 
   * @param l O listener de alterao do contedo do container de seleo.
   */
  public void removeContainerSelectionListener(ContainerSelectionListener l) {
    containerSelectionListeners.remove(l);
  }

  /**
   * Limpa os tens selecionados e os disponveis.
   */
  public void clear() {
    this.clearAvailableItems();
    this.clearSelectedItems();
  }

  // Mtodos para AvailableItems
  /**
   * Limpa a lista de tens disponveis.
   */
  public void clearAvailableItems() {
    availableContainer.removeAll();
  }

  /**
   * Obtm o container dos itens disponveis.
   * 
   * @return ItemContainer O container.
   */
  public ItemContainer<T> getAvailableContainer() {
    return availableContainer;
  }

  /**
   * Obtm uma lista no-modificvel dos elementos disponveis.
   * 
   * @return A lista com os elementos disponveis.
   * 
   * @see Collections#unmodifiableList(List)
   */
  public List<T> getAvailableItems() {
    return Collections.unmodifiableList(availableContainer.getAll());
  }

  /**
   * Obtm o primeiro item disponivel.
   * 
   * @return O primeiro item selecionado ou <code>null</code> se no existirem
   *         itens.
   */
  public T getFirstAvailableItem() {
    if (!hasAvailableItems()) {
      return null;
    }
    return getAvailableItems().get(0);
  }

  /**
   * Verifica se existem tens disponveis.
   * 
   * @return true se houver tens disponveis.
   */
  public boolean hasAvailableItems() {
    return !getAvailableItems().isEmpty();
  }

  /**
   * Preenche o conjunto dos elementos disponveis para seleo.
   * 
   * @param elements Uma coleo com os elementos passveis de seleo.
   */
  protected void loadAvailableItems(Collection<T> elements) {
    availableContainer.load(elements);
  }

  // Mtodos para SelectedItems
  /**
   * Limpa a lista de tens selecionados.
   */
  public void clearSelectedItems() {
    selectedContainer.removeAll();
  }

  /**
   * Obtm o primeiro item selecionado.
   * 
   * @return O primeiro item selecionado ou <code>null</code> se no existirem
   *         itens.
   */
  public T getFirstSelectedItem() {
    if (!hasSelectedItems()) {
      return null;
    }
    return getSelectedItems().get(0);
  }

  /**
   * Obtm o container dos itens selecionados.
   * 
   * @return ItemContainer O container.
   */
  public ItemContainer<T> getSelectedContainer() {
    return selectedContainer;
  }

  /**
   * Obtm uma lista no-modificvel dos elementos selecionados.
   * 
   * @return A lista dos elementos selecionados.
   * 
   * @see Collections#unmodifiableList(List)
   */
  public List<T> getSelectedItems() {
    return Collections.unmodifiableList(selectedContainer.getAll());
  }

  /**
   * Verifica se existem tens selecionados.
   * 
   * @return true se houver tens selecionados.
   */
  public boolean hasSelectedItems() {
    return !getSelectedItems().isEmpty();
  }

  /**
   * Preenche o conjunto dos elementos selecionados.
   * 
   * @param elements Uma coleo com os elementos selecionados.
   */
  private void loadSelectedItems(Collection<T> elements) {
    this.selectedContainer.load(elements);
  }

  // --
  /**
   * Devolve o painel que modela esse componente de seleo.
   * 
   * @return O painel para ser usado como componente de seleo em uma janela ou
   *         dilogo.
   */
  public JPanel getPanel() {
    return this.panel;
  }

  /**
   * Verifica se existem elementos no container.
   * 
   * @return <code>true</code> se existem ou <code>false</code> caso contrrio.
   */
  public boolean hasItems() {
    return hasSelectedItems() || hasAvailableItems();
  }

  /**
   * <p>
   * Carrega os tens no container.
   * </p>
   * <p>
   * Se o mesmo tem estiver na lista de selecionados e na lista de disponveis,
   * ele aparecer apenas na lista de selecionados. Desta forma, os tens
   * disponveis podem ser todos os tens utilizados.
   * </p>
   * 
   * @param availableItems Os tens disponveis.
   * @param selectedItems Os tens selecionados.
   */
  public void loadItems(Collection<T> availableItems, Collection<T> selectedItems) {
    if (hasRestoreDefaultsButton) {
      // Copia a lista inicial de selecionados para torn-la a lista padro de
      // selecionados.
      Collection<T> defaultLocalList = new LinkedList<T>();
      defaultLocalList.addAll(selectedItems);
      this.loadItems(availableItems, selectedItems, defaultLocalList);
    }
    else {
      this.loadItems(availableItems, selectedItems, null);
    }
  }

  /**
   * <p>
   * Carrega os tens no container e recebe os tens que ficaro selecionados
   * quando se optar por restaura o estado padro. O boto recuperar ficar
   * desabilitado se ele estiver visvel e a lista padro de itens selecionados
   * for nula.
   * </p>
   * <p>
   * Se o mesmo tem estiver na lista de selecionados e na lista de disponveis,
   * ele aparecer apenas na lista de selecionados. Desta forma, os tens
   * disponveis podem ser todos os tens utilizados.
   * </p>
   * 
   * @param availableItems Os tens disponveis.
   * @param selectedItems Os tens selecionados.
   * @param defaultSelectedItems Os tens selecionados pelo boto restaurar.
   */
  public void loadItems(Collection<T> availableItems, Collection<T> selectedItems, Collection<T> defaultSelectedItems) {
    if (this.restoreDefaultsButton != null) {
      // se a defaultSelectedItems for nula o boto de restaurao
      // ficar desabilitado
      this.restoreDefaultsButton.setEnabled(defaultSelectedItems != null);
    }
    this.defaultSelectedItems = defaultSelectedItems;
    Collection<T> availableCollection = new LinkedList<T>();
    availableCollection.addAll(availableItems);
    availableCollection.removeAll(selectedItems);
    this.clear();
    this.loadAvailableItems(availableCollection);
    this.loadSelectedItems(selectedItems);
    fireContainerSelectionEvent();
  }

  /**
   * Carrega os tens disponveis e se houver tens selecionados, os remove.
   * 
   * @param availableItems tens disponveis.
   */
  public void loadItems(Collection<T> availableItems) {
    loadItems(availableItems, new ArrayList<T>());
  }

  /**
   * Habilita os componentes da interface.
   * 
   * @param enabled true para habilitar.
   */
  public void setEnabled(boolean enabled) {
    if (this.restoreDefaultsButton != null) {
      // se a defaultSelectedItems for nula o boto de restaurao
      // ficar desabilitado
      this.restoreDefaultsButton.setEnabled(enabled && this.defaultSelectedItems != null);
    }
    this.availableContainer.setEnabled(enabled);
    this.selectedContainer.setEnabled(enabled);
    if (enabled) {
      if (this.availableContainer.hasSelectedItems()) {
        this.addButton.setEnabled(true);
      }
      if (!availableContainer.getAll().isEmpty()) {
        this.addAllButton.setEnabled(true);
      }
      if (this.selectedContainer.hasSelectedItems()) {
        this.removeButton.setEnabled(true);
      }
      if (!selectedContainer.getAll().isEmpty()) {
        this.removeAllButton.setEnabled(true);
      }
    }
    else {
      this.addButton.setEnabled(false);
      this.addAllButton.setEnabled(false);
      this.removeButton.setEnabled(false);
      this.removeAllButton.setEnabled(false);
    }
  }

  /**
   * Transfere todos os elementos de um conjunto para o outro.
   * 
   * @param source O conjunto de origem.
   * @param target O conjunto destino.
   */
  protected void moveAllRows(ItemContainer<T> source, ItemContainer<T> target) {
    Collection<T> elements = source.removeAll();
    target.load(elements);
    fireContainerSelectionEvent();
  }

  /**
   * Transfere os elementos selecionados de um conjunto para outro.
   * 
   * @param source O conjunto de origem.
   * @param target O conjunto destino.
   */
  protected void moveRows(ItemContainer<T> source, ItemContainer<T> target) {
    Collection<T> elements = source.removeSelectedElements();
    target.load(elements);
    fireContainerSelectionEvent();
  }

  /**
   * Monta o painel, arrumando os componentes de interface.
   */
  private void buildPane() {
    int height;
    if (hasRestoreDefaultsButton) {
      height = 5;
    }
    else {
      height = 4;
    }
    // Monta panel com as listas/tabelas e botes
    panel = new JPanel(new GridBagLayout());
    GBC constraints;

    // Insere o titulo dos elementos disponveis
    int gridY = 0;
    if (hasContainerTitle) {
      constraints = new GBC(0, gridY++).insets(10, 2, 2, 2).west().none();
      JLabel availableLabel = new JLabel(availableText);
      panel.add(availableLabel, constraints);
    }

    // Monta painel de filtro
    filterPanel = makeTextFieldFilterPanel();
    constraints = new GBC(0, gridY).insets(2, 2, 2, 2).horizontal();
    panel.add(filterPanel, constraints);
    filterPanel.setVisible(false);

    // Monta container esquerdo
    constraints = new GBC(0, gridY + 1).insets(2, 2, 2, 2).width(1).both().height(height);
    panel.add(availableContainer.getView(), constraints);

    // Monta painel com os botes
    JPanel buttonPanel = makeButtonPanel();
    // Insere painel de botes no painel principal
    constraints = new GBC(1, gridY + 1).insets(2, 2, 2, 2).both().height(height).weightx(0.0);
    panel.add(buttonPanel, constraints);

    // Insere o ttulo dos elementos selecionados
    gridY = 0;
    if (hasContainerTitle) {
      constraints = new GBC(2, gridY++).width(1).height(1).west().none().insets(10, 2, 2, 2);
      JLabel selectedLabel = new JLabel(selectedText);
      panel.add(selectedLabel, constraints);
    }
    // Monta container direito
    constraints = new GBC(2, gridY + 1).insets(2, 2, 2, 2).both().east().height(height);
    panel.add(selectedContainer.getView(), constraints);
  }

  /**
   * Monta o painel para filtrar os elementos selecionados. Este mtodo pode ser
   * sobrescrito para se criar um painel de filtragem diferente do padro.
   * 
   * @return painel para filtragem padro para os elementos selecionados
   */
  protected TextFieldFilterPanel makeTextFieldFilterPanel() {
    return new TextFieldFilterPanel(this);
  }

  /**
   * Cria o painel central de Botes
   * 
   * @return painel contendo os botes de adicionar, remover,...
   */
  private JPanel makeButtonPanel() {
    buttonPanel = new JPanel(new GridBagLayout());
    GBC constraints;

    constraints = new GBC(0, gridy++).insets(2, 2, 2, 2).horizontal().center();
    buttonPanel.add(addButton, constraints);

    constraints = new GBC(0, gridy++).insets(2, 2, 2, 2).horizontal().center();
    buttonPanel.add(removeButton, constraints);

    if (hasAddAllButton) {
      constraints = new GBC(0, gridy++).insets(10, 2, 2, 2).horizontal().center();
      buttonPanel.add(addAllButton, constraints);
    }
    if (hasRemoveAllButton) {
      if (hasAddAllButton) {
        constraints = new GBC(0, gridy++).insets(2, 2, 2, 2).horizontal().center();
      }
      else {
        constraints = new GBC(0, gridy++).insets(10, 2, 2, 2).horizontal().center();
      }
      buttonPanel.add(removeAllButton, constraints);
    }

    // Adiciona o boto "restaurar padro"
    if (hasRestoreDefaultsButton) {
      constraints = new GBC(0, gridy++).insets(10, 2, 2, 2).horizontal().center();
      restoreDefaultsButton = makeRestoreDefaultsButton();
      buttonPanel.add(restoreDefaultsButton, constraints);
    }
    JPanel returnPanel = new JPanel(new BorderLayout());
    switch (buttonPanelAnchor) {
      case SwingConstants.NORTH:
        returnPanel.add(buttonPanel, BorderLayout.NORTH);
        break;
      case SwingConstants.SOUTH:
        returnPanel.add(buttonPanel, BorderLayout.SOUTH);
        break;
      case SwingConstants.CENTER:
        returnPanel.add(buttonPanel, BorderLayout.CENTER);
        break;
      default:
        returnPanel.add(buttonPanel, BorderLayout.CENTER);
    }
    return returnPanel;
  }

  /**
   * Cria o boto de adio de todos os elementos do conjunto de disponveis
   * adicionando todos os elementos do conjunto de escolhidos. Esse boto
   * somente est habilitado se houver algum elemento no conjunto de
   * disponveis.
   * 
   * @return O boto criado.
   */
  private JButton makeAddAllButton() {
    final String text = showButtonsText ? LNG.get(LNG_KEY_PREFIX + "addAll") : "";
    addAllButton = createAddAllButton(text);
    addAllButton.setHorizontalTextPosition(SwingConstants.LEADING);
    addAllButton.setToolTipText(LNG.get(LNG_KEY_PREFIX + "addAll.tooltip"));
    addAllButton.setEnabled(false);
    addAllButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        moveAllRows(availableContainer, selectedContainer);
      }
    });
    return addAllButton;
  }

  /**
   * Cria o boto adicionar todos os elementos. Deve ser sobrecrito para alterar
   * o cone padro do componente.
   * 
   * @param text texto para o boto, j considera o valor da flag
   *        'showButtonsText'
   * 
   * @return boto adicionar todos os elementos
   */
  protected JButton createAddAllButton(String text) {
    return new JButton(text, GUIResources.SELECTOR_ADDALL_ICON);
  }

  /**
   * Cria o boto de adio de elementos no conjunto de escolhidos, retirando os
   * elementos do conjunto de disponveis. Esse boto somente est habilitado se
   * houver algum elemento selecionado no conjunto de disponveis.
   * 
   * @return O boto criado.
   */
  private JButton makeAddButton() {
    final String text = showButtonsText ? LNG.get(LNG_KEY_PREFIX + "add") : "";
    addButton = createAddButton(text);
    addButton.setHorizontalTextPosition(SwingConstants.LEADING);
    addButton.setToolTipText(LNG.get(LNG_KEY_PREFIX + "add.tooltip"));
    addButton.setEnabled(false); // s ativa se tiver algum item selecionado
    addButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        moveRows(availableContainer, selectedContainer);
      }
    });
    return addButton;
  }

  /**
   * Cria o boto adicionar elemento. Deve ser sobrecrito para alterar o cone
   * padro do componente.
   * 
   * @param text texto para o boto, j considera o valor da flag
   *        'showButtonsText'
   * 
   * @return boto adicionar elemento
   */
  protected JButton createAddButton(String text) {
    return new JButton(text, GUIResources.SELECTOR_ADD_ICON);
  }

  /**
   * Cria o componente que representa um conjunto de elementos.
   * 
   * @param columnFormatter O formatador responsvel pela formatao dos
   *        elementos desse conjunto.
   * @param singleSelectionButton O boto default associado a remoo de
   *        elementos selecionados desse conjunto.
   * @param allSelectedButton O boto associado a remoo de todos os elementos
   *        desse conjunto.
   * @param enabled Indica se o container est habilitado ou no.
   * @param enableSort - indica se o componente deve ser ordenado
   * @param adjustColumnsWidth - Flag que indica se a tabela deve ser ajustada
   *        para ocupar, pelo menos, o tamanho mnimo para que no seja
   *        necessrio redimensionar suas colunas para que seja possvel ler
   *        todo o contedo destas.
   * 
   * @return O componente que modela um conjunto de elementos.
   */
  private ItemContainer<T> makeContainer(ObjectTableProvider columnFormatter, JButton singleSelectionButton,
    JButton allSelectedButton, boolean enabled, boolean enableSort, boolean adjustColumnsWidth) {
    ItemContainer<T> container = new ItemContainer<T>(columnFormatter, singleSelectionButton, allSelectedButton,
      enabled, enableSort, adjustColumnsWidth);
    return container;
  }

  /**
   * Cria o boto de remoo de todos os elementos no conjunto de escolhidos
   * adicionando todos os elementos do conjunto de disponveis. Esse boto
   * somente est habilitado se houver algum elemento no conjunto de escolhidos.
   * 
   * @return O boto criado.
   */
  private JButton makeRemoveAllButton() {
    final String text = showButtonsText ? LNG.get(LNG_KEY_PREFIX + "removeAll") : "";
    removeAllButton = createRemoveAllButton(text);
    removeAllButton.setToolTipText(LNG.get(LNG_KEY_PREFIX + "removeAll.tooltip"));
    removeAllButton.setEnabled(false);
    removeAllButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        moveAllRows(selectedContainer, availableContainer);
      }
    });
    return removeAllButton;
  }

  /**
   * Cria o boto remover todos os elementos. Deve ser sobrecrito para alterar o
   * cone padro do componente.
   * 
   * @param text texto para o boto, j considera o valor da flag
   *        'showButtonsText'
   * 
   * @return boto remover todos os elementos
   */
  protected JButton createRemoveAllButton(String text) {
    return new JButton(text, GUIResources.SELECTOR_REMOVEALL_ICON);
  }

  /**
   * Cria o boto de remoo de elementos do conjunto de escolhidos, adicionado
   * esses elementos do conjunto de disponveis. Esse boto somente est
   * habilitado se houver algum elemento selecionado no conjunto de escolhidos.
   * 
   * @return O boto criado.
   */
  private JButton makeRemoveButton() {
    final String text = showButtonsText ? LNG.get(LNG_KEY_PREFIX + "remove") : "";
    removeButton = createRemoveButton(text);
    removeButton.setToolTipText(LNG.get(LNG_KEY_PREFIX + "remove.tooltip"));
    removeButton.setEnabled(false);
    // s ativa se tiver algum item selecionado
    removeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        moveRows(selectedContainer, availableContainer);
      }
    });
    return removeButton;
  }

  /**
   * Cria o boto remover elemento. Deve ser sobrecrito para alterar o cone
   * padro do componente.
   * 
   * @param text texto para o boto, j considera o valor da flag
   *        'showButtonsText'
   * 
   * @return boto remover elemento
   */
  protected JButton createRemoveButton(String text) {
    return new JButton(text, GUIResources.SELECTOR_REMOVE_ICON);
  }

  /**
   * Cria o boto de restaurao da lista default.
   * 
   * @return O boto criado.
   */
  private JButton makeRestoreDefaultsButton() {
    final String text = showButtonsText ? LNG.get(LNG_KEY_PREFIX + "restoreDefaults") : "";
    restoreDefaultsButton = createRestoreDefaultButton(text);
    restoreDefaultsButton.setToolTipText(LNG.get(LNG_KEY_PREFIX + "restoreDefaults.tooltip"));
    restoreDefaultsButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        restoreDefaults();
      }
    });
    return restoreDefaultsButton;
  }

  /**
   * Cria o boto restaurar default. Deve ser sobrecrito para alterar o cone
   * padro do componente.
   * 
   * @param text texto para o boto, j considera o valor da flag
   *        'showButtonsText'
   * 
   * @return boto restaurar default
   */
  protected JButton createRestoreDefaultButton(String text) {
    return new JButton(text, GUIResources.SELECTOR_RESTORE_ICON);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos. Esse componente
   * ter o boto restaurar e estar habilitado para o uso.
   * 
   * @param formatter O formatador responsvel pela formatao do conjunto de
   *        elementos disponveis e do conjunto de elementos escolhidos.
   */
  public ContainerSelection(ObjectTableProvider formatter) {
    this(formatter, formatter, true);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos. Esse componente
   * ter o boto restaurar e estar habilitado para o uso.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter) {
    this(firstFormatter, secondFormatter, true);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos. Esse componente
   * ter o boto restaurar.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   * @param enabled Indica se os containers esto habilitados ou no.
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter, boolean enabled) {
    this(firstFormatter, secondFormatter, enabled, true);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   * @param enabled Indica se os containers esto habilitados ou no.
   * @param hasRestoreDefaultsButton Indica se o boto restaurar ser usado.
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter, boolean enabled,
    boolean hasRestoreDefaultsButton) {
    this(firstFormatter, secondFormatter, enabled, hasRestoreDefaultsButton, null, null, true, true);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   * @param enabled Indica se os containers esto habilitados ou no.
   * @param hasRestoreDefaultsButton Indica se o boto restaurar ser usado.
   * @param hasAddAllButton Indica se o boto de adcionar todos ser usado.
   * @param hasRemoveAllButton Indica se o boto de adcionar todos ser usado.
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter, boolean enabled,
    boolean hasRestoreDefaultsButton, boolean hasAddAllButton, boolean hasRemoveAllButton) {
    this(firstFormatter, secondFormatter, enabled, hasRestoreDefaultsButton, null, null, hasAddAllButton,
      hasRemoveAllButton);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   * @param enabled Indica se os containers esto habilitados ou no.
   * @param hasRestoreDefaultsButton Indica se o boto restaurar ser usado.
   * @param hasAddAllButton Indica se o boto de adcionar todos ser usado.
   * @param hasRemoveAllButton Indica se o boto de adcionar todos ser usado.
   * @param buttonPanelAnchor Posio onde o painel de botes ficar localizado,
   *        SwingConstants: NORTH, SOUTH, CENTER
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter, boolean enabled,
    boolean hasRestoreDefaultsButton, boolean hasAddAllButton, boolean hasRemoveAllButton, int buttonPanelAnchor) {
    this(firstFormatter, secondFormatter, enabled, hasRestoreDefaultsButton, null, null, hasAddAllButton,
      hasRemoveAllButton, buttonPanelAnchor);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   * @param enabled Indica se os containers esto habilitados ou no.
   * @param hasRestoreDefaultsButton Indica se o boto restaurar ser usado.
   * @param hasAddAllButton Indica se o boto de adcionar todos ser usado.
   * @param hasRemoveAllButton Indica se o boto de adcionar todos ser usado.
   * @param buttonPanelAnchor Posio onde o painel de butes ficar localizado,
   *        SwingConstants: NORTH, SOUTH, CENTER
   * @param enableSort indica se o container dever se ordenar ou no
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter, boolean enabled,
    boolean hasRestoreDefaultsButton, boolean hasAddAllButton, boolean hasRemoveAllButton, int buttonPanelAnchor,
    boolean enableSort) {
    this(firstFormatter, secondFormatter, enabled, hasRestoreDefaultsButton, null, null, hasAddAllButton,
      hasRemoveAllButton, buttonPanelAnchor, enableSort);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   * @param enabled Indica se os containers esto habilitados ou no.
   * @param hasRestoreDefaultsButton Indica se o boto restaurar ser usado.
   * @param availableText ttulo para a tabela de elementos disponveis.
   * @param selectedText ttulo para a tabela de elementos escolhidos.
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter, boolean enabled,
    boolean hasRestoreDefaultsButton, String availableText, String selectedText) {
    this(firstFormatter, secondFormatter, enabled, hasRestoreDefaultsButton, availableText, selectedText, true, true);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   * @param enabled Indica se os containers esto habilitados ou no.
   * @param hasRestoreDefaultsButton Indica se o boto restaurar ser usado.
   * @param availableText ttulo para a tabela de elementos disponveis.
   * @param selectedText ttulo para a tabela de elementos escolhidos.
   * @param hasAddAllButton Indica se o boto de adcionar todos ser usado.
   * @param hasRemoveAllButton Indica se o boto de adcionar todos ser usado.
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter, boolean enabled,
    boolean hasRestoreDefaultsButton, String availableText, String selectedText, boolean hasAddAllButton,
    boolean hasRemoveAllButton) {
    this(firstFormatter, secondFormatter, enabled, hasRestoreDefaultsButton, availableText, selectedText,
      hasAddAllButton, hasRemoveAllButton, SwingConstants.CENTER);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   * @param enabled Indica se os containers esto habilitados ou no.
   * @param hasRestoreDefaultsButton Indica se o boto restaurar ser usado.
   * @param availableText ttulo para a tabela de elementos disponveis.
   * @param selectedText ttulo para a tabela de elementos escolhidos.
   * @param hasAddAllButton Indica se o boto de adcionar todos ser usado.
   * @param hasRemoveAllButton Indica se o boto de adcionar todos ser usado.
   * @param buttonPanelAnchor Posio onde o painel de Botes deve ficar. Os
   *        valores podem ser: SwingConstants. NORTH, SOUTH, CENTER
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter, boolean enabled,
    boolean hasRestoreDefaultsButton, String availableText, String selectedText, boolean hasAddAllButton,
    boolean hasRemoveAllButton, int buttonPanelAnchor) {
    this(firstFormatter, secondFormatter, enabled, hasRestoreDefaultsButton, availableText, selectedText,
      hasAddAllButton, hasRemoveAllButton, buttonPanelAnchor, true);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   * @param enabled Indica se os containers esto habilitados ou no.
   * @param hasRestoreDefaultsButton Indica se o boto restaurar ser usado.
   * @param availableText ttulo para a tabela de elementos disponveis.
   * @param selectedText ttulo para a tabela de elementos escolhidos.
   * @param hasAddAllButton Indica se o boto de adcionar todos ser usado.
   * @param hasRemoveAllButton Indica se o boto de adcionar todos ser usado.
   * @param buttonPanelAnchor Posio onde o painel de Botes deve ficar. Os
   *        valores podem ser: SwingConstants. NORTH, SOUTH, CENTER
   * @param enableSort indica de a ordenao do container est habilitada ou nao
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter, boolean enabled,
    boolean hasRestoreDefaultsButton, String availableText, String selectedText, boolean hasAddAllButton,
    boolean hasRemoveAllButton, int buttonPanelAnchor, boolean enableSort) {
    this(firstFormatter, secondFormatter, enabled, hasRestoreDefaultsButton, availableText, selectedText,
      hasAddAllButton, hasRemoveAllButton, buttonPanelAnchor, true, true);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   * @param sortableContainer indica qual dos containers deve ser ordenvel
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter,
    SortableContainer sortableContainer) {
    this(firstFormatter, secondFormatter, true, true, null, null, true, true, SwingConstants.CENTER, sortableContainer,
      true);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   * @param enabled Indica se os containers esto habilitados ou no.
   * @param hasRestoreDefaultsButton Indica se o boto restaurar ser usado.
   * @param availableText ttulo para a tabela de elementos disponveis.
   * @param selectedText ttulo para a tabela de elementos escolhidos.
   * @param hasAddAllButton Indica se o boto de adcionar todos ser usado.
   * @param hasRemoveAllButton Indica se o boto de adcionar todos ser usado.
   * @param buttonPanelAnchor Posio onde o painel de Botes deve ficar. Os
   *        valores podem ser: SwingConstants. NORTH, SOUTH, CENTER
   * @param sortableContainer indica qual dos containers deve ser ordenvel
   * @param showButtonsText indica se o texto dos botes deve ser exibido
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter, boolean enabled,
    boolean hasRestoreDefaultsButton, String availableText, String selectedText, boolean hasAddAllButton,
    boolean hasRemoveAllButton, int buttonPanelAnchor, SortableContainer sortableContainer, boolean showButtonsText) {
    this(firstFormatter, secondFormatter, enabled, hasRestoreDefaultsButton, availableText, selectedText,
      hasAddAllButton, hasRemoveAllButton, buttonPanelAnchor, sortableContainer, showButtonsText, true);
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   * @param enabled Indica se os containers esto habilitados ou no.
   * @param hasRestoreDefaultsButton Indica se o boto restaurar ser usado.
   * @param availableText ttulo para a tabela de elementos disponveis.
   * @param selectedText ttulo para a tabela de elementos escolhidos.
   * @param hasAddAllButton Indica se o boto de adcionar todos ser usado.
   * @param hasRemoveAllButton Indica se o boto de adcionar todos ser usado.
   * @param buttonPanelAnchor Posio onde o painel de Botes deve ficar. Os
   *        valores podem ser: SwingConstants. NORTH, SOUTH, CENTER
   * @param sortableContainer indica qual dos containers deve ser ordenvel
   * @param showButtonsText indica se o texto dos botes deve ser exibido
   * @param adjustColumnsWidth - Flag que indica se a tabela deve ser ajustada
   *        para ocupar, pelo menos, o tamanho mnimo para que no seja
   *        necessrio redimensionar suas colunas para que seja possvel ler
   *        todo o contedo destas.
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter, boolean enabled,
    boolean hasRestoreDefaultsButton, String availableText, String selectedText, boolean hasAddAllButton,
    boolean hasRemoveAllButton, int buttonPanelAnchor, SortableContainer sortableContainer, boolean showButtonsText,
    boolean adjustColumnsWidth) {
    this.hasRestoreDefaultsButton = hasRestoreDefaultsButton;
    this.hasAddAllButton = hasAddAllButton;
    this.hasRemoveAllButton = hasRemoveAllButton;
    this.hasContainerTitle = (availableText != null || selectedText != null);
    this.buttonPanelAnchor = buttonPanelAnchor;
    this.availableText = availableText;
    this.selectedText = selectedText;
    this.showButtonsText = showButtonsText;
    this.addButton = makeAddButton();
    this.removeButton = makeRemoveButton();
    this.addAllButton = makeAddAllButton();
    this.removeAllButton = makeRemoveAllButton();
    boolean sortFirst = sortableContainer == SortableContainer.BOTH || sortableContainer == SortableContainer.LEFT_ONLY;
    boolean sortSecond = sortableContainer == SortableContainer.BOTH
      || sortableContainer == SortableContainer.RIGHT_ONLY;
    /* Cria o primeiro container (da esquerda) */
    this.availableContainer = makeContainer(firstFormatter, this.addButton, this.addAllButton, enabled, sortFirst,
      adjustColumnsWidth);
    /* Cria o segundo container (da direita) */
    this.selectedContainer = makeContainer(secondFormatter, this.removeButton, this.removeAllButton, enabled,
      sortSecond, adjustColumnsWidth);
    containerSelectionListeners = new LinkedList<ContainerSelectionListener>();
    /* Monta o painel com os componentes de interface */
    buildPane();
  }

  /**
   * Constri um componente para seleo de elementos que usa dois conjuntos: um
   * de elementos disponveis e outro de elementos escolhidos.
   * 
   * @param firstFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos disponveis.
   * @param secondFormatter O formatador responsvel pela formatao do conjunto
   *        de elementos escolhidos.
   * @param enabled Indica se os containers esto habilitados ou no.
   * @param hasRestoreDefaultsButton Indica se o boto restaurar ser usado.
   * @param availableText ttulo para a tabela de elementos disponveis.
   * @param selectedText ttulo para a tabela de elementos escolhidos.
   * @param hasAddAllButton Indica se o boto de adcionar todos ser usado.
   * @param hasRemoveAllButton Indica se o boto de adcionar todos ser usado.
   * @param buttonPanelAnchor Posio onde o painel de Botes deve ficar. Os
   *        valores podem ser: SwingConstants. NORTH, SOUTH, CENTER
   * @param enableSort indica se ambos os containers devem ser ordenveis
   * @param showButtonsText indica se o texto dos botes deve ser exibido
   */
  public ContainerSelection(ObjectTableProvider firstFormatter, ObjectTableProvider secondFormatter, boolean enabled,
    boolean hasRestoreDefaultsButton, String availableText, String selectedText, boolean hasAddAllButton,
    boolean hasRemoveAllButton, int buttonPanelAnchor, boolean enableSort, boolean showButtonsText) {
    this(firstFormatter, secondFormatter, enabled, hasRestoreDefaultsButton, availableText, selectedText,
      hasAddAllButton, hasRemoveAllButton, buttonPanelAnchor, SortableContainer.BOTH, showButtonsText);
  }

  /**
   * Recarrega lista padro de seleo.
   */
  public void restoreDefaults() {
    ArrayList<T> list = new ArrayList<T>();
    list.addAll(getAvailableItems());
    list.addAll(getSelectedItems());
    this.loadItems(list, defaultSelectedItems);
  }

  /**
   * Restaura os itens dos componentes para o seu estado inicial.
   */
  public void reset() {
    moveAllRows(selectedContainer, availableContainer);
  }

  /**
   * Ajusta a largura de todas as colunas das tabelas de exibio dos elementos
   * disponveis e escolhidos, buscando a menor largura que seja suficiente para
   * exibio do contedo.
   */
  public void adjustTableColumns() {
    this.availableContainer.getTable().adjustColumnWidth();
    this.selectedContainer.getTable().adjustColumnWidth();
  }

  /**
   * Habilita/desabilita tooltip
   * 
   * @param enabled
   */
  public final void setToolTipEnabled(boolean enabled) {
    this.availableContainer.getTable().setToolTipEnabled(enabled);
    this.selectedContainer.getTable().setToolTipEnabled(enabled);
  }

  /**
   * Adiciona um boto ao painel de botes do seletor. O valor de
   * <code>button</code> no pode ser null.
   * 
   * @param button o boto a ser adicionado ao seletor.
   */
  public void addButton(JButton button) {
    if (button == null) {
      throw new NullPointerException();
    }
    GBC constraints = new GBC(0, gridy++).insets(10, 2, 2, 2).horizontal().center();
    buttonPanel.add(button, constraints);
  }

  /**
   * Atribui o valor para mostrar ou no o filtro de campo de texto para o
   * container de itens disponveis
   * 
   * @param showFilter
   */
  public void setShowFilter(boolean showFilter) {
    filterPanel.setVisible(showFilter);
  }

  /**
   * Trata o filtro: obtm a string digitada pelo usurio no campo de texto do
   * painel de filtro e repassa pro itemContainer de valores disponveis
   */
  @Override
  public void filter(String toFilter) {
    availableContainer.filter(toFilter);
  }

}
