package tecgraf.javautils.gui.table;

import javax.swing.event.EventListenerList;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreePath;

/**
 * Modelo abstrato para a tabela que contm uma rvore.
 *
 *
 * @author Tecgraf/PUC-Rio
 */
public abstract class AbstractTreeTableModel implements TreeTableModel {
  /** Representa a raiz da rvore. */
  protected Object root;

  /** Os listeners deste modelo. */
  protected EventListenerList listenerList = new EventListenerList();

  /** Construtor */
  public AbstractTreeTableModel() {
    this(null);
  }

  /**
   * Construtor
   *
   * @param root raiz da rvore.
   */
  public AbstractTreeTableModel(Object root) {
    this.root = root;
  }

  /**
   * Define a raiz da rvore.
   *
   * @param root a nova raiz da rvore.
   */
  public void setRoot(Object root) {
    this.root = root;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getRoot() {
    return root;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isLeaf(Object node) {
    return getChildCount(node) == 0;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void valueForPathChanged(TreePath path, Object newValue) {
  }

  /**
   * {@inheritDoc}
   */
  // This is not called in the JTree's default mode: use a naive implementation.
  @Override
  public int getIndexOfChild(Object parent, Object child) {
    for (int i = 0; i < getChildCount(parent); i++) {
      if (getChild(parent, i).equals(child)) {
        return i;
      }
    }
    return -1;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addTreeModelListener(TreeModelListener l) {
    listenerList.add(TreeModelListener.class, l);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeTreeModelListener(TreeModelListener l) {
    listenerList.remove(TreeModelListener.class, l);
  }

  /**
   * Notifica todos os listeners que foram registrados para receber notificaes
   * desse tipo de evento. A isntncia do evento  criada usando os parmetros
   * para o mtodo.
   *
   * @param source a origem do evento.
   * @param path o caminho do n pai alterado.
   * @param childIndices os ndices dos filhos alterados em ordem crescente.
   * @param children a relao de filhos alterados.
   *
   * @see EventListenerList
   */
  protected void fireTreeNodesChanged(Object source, Object[] path,
    int[] childIndices, Object[] children) {
    Object[] listeners = listenerList.getListenerList();
    TreeModelEvent e = null;
    for (int i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == TreeModelListener.class) {
        if (e == null) {
          e = new TreeModelEvent(source, path, childIndices, children);
        }
        ((TreeModelListener) listeners[i + 1]).treeNodesChanged(e);
      }
    }
  }

  /**
   * Notifica todos os listeners que foram registrados para receber notificaes
   * desse tipo de evento. A isntncia do evento  criada usando os parmetros
   * para o mtodo.
   *
   * @param source a origem do evento.
   * @param path o caminho do n pai alterado.
   * @param childIndices os ndices dos filhos alterados em ordem crescente.
   * @param children a relao de filhos alterados.
   *
   * @see EventListenerList
   */
  protected void fireTreeNodesInserted(Object source, Object[] path,
    int[] childIndices, Object[] children) {
    Object[] listeners = listenerList.getListenerList();
    TreeModelEvent e = null;
    for (int i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == TreeModelListener.class) {
        if (e == null) {
          e = new TreeModelEvent(source, path, childIndices, children);
        }
        ((TreeModelListener) listeners[i + 1]).treeNodesInserted(e);
      }
    }
  }

  /**
   * Notifica todos os listeners que foram registrados para receber notificaes
   * desse tipo de evento. A isntncia do evento  criada usando os parmetros
   * para o mtodo.
   *
   * @param source a origem do evento.
   * @param path o caminho do n pai alterado.
   * @param childIndices os ndices dos filhos alterados em ordem crescente.
   * @param children a relao de filhos alterados.
   *
   * @see EventListenerList
   */
  protected void fireTreeNodesRemoved(Object source, Object[] path,
    int[] childIndices, Object[] children) {
    Object[] listeners = listenerList.getListenerList();
    TreeModelEvent e = null;
    for (int i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == TreeModelListener.class) {
        if (e == null) {
          e = new TreeModelEvent(source, path, childIndices, children);
        }
        ((TreeModelListener) listeners[i + 1]).treeNodesRemoved(e);
      }
    }
  }

  /**
   * Notifica todos os listeners que foram registrados para receber notificaes
   * desse tipo de evento. A isntncia do evento  criada usando os parmetros
   * para o mtodo.
   *
   * @param source a origem do evento.
   * @param path o caminho do n pai alterado.
   * @param childIndices os ndices dos filhos alterados em ordem crescente.
   * @param children a relao de filhos alterados.
   *
   * @see EventListenerList
   */
  protected void fireTreeStructureChanged(Object source, Object[] path,
    int[] childIndices, Object[] children) {
    Object[] listeners = listenerList.getListenerList();
    TreeModelEvent e = null;
    for (int i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == TreeModelListener.class) {
        if (e == null) {
          e = new TreeModelEvent(source, path, childIndices, children);
        }
        ((TreeModelListener) listeners[i + 1]).treeStructureChanged(e);
      }
    }
  }

  /**
   * {@inheritDoc}
   *
   * Por padro, assume que a primeira coluna contm a rvore.
   */
  @Override
  public Class<?> getColumnClass(int column) {
    if (column == 0) {
      return TreeTableModel.class;
    }
    return Object.class;
  }

  /**
   * {@inheritDoc} O padro  definir somente a coluna da rvore como editvel.
   */
  @Override
  public boolean isCellEditable(Object node, int column) {
    return getColumnClass(column) == TreeTableModel.class;
  }

  /**
   * {@inheritDoc} O padro no faz nada.
   */
  @Override
  public void setValueAt(Object aValue, Object node, int column) {
  }

  // Devero ser implementados pelas sub-classes:
  //public Object getChild(Object parent, int index)
  //public int getChildCount(Object parent)
  //public int getColumnCount()
  //public String getColumnName(Object node, int column)
  //public Object getValueAt(Object node, int column)

}
