package tecgraf.javautils.parsers.actions;

import java.util.LinkedList;
import java.util.List;

import tecgraf.javautils.parsers.Session;
import tecgraf.javautils.parsers.exception.AutomatonException;
import tecgraf.javautils.parsers.symbols.Symbol;

/**
 * Representa uma composio de aes de transio. Todas as aes da composio
 * sero executadas, respeitando a ordem em que foram inseridas na composio.
 */
public final class CompositeTransitionAction implements TransitionAction {
  /**
   * Contm as aes que fazem parte da composio.
   */
  private final List<TransitionAction> actions;

  /**
   * Adiciona uma ao  composio.
   * 
   * @param action A ao a ser adicionada.
   * 
   * @return true, se a ao foi adicionadas, ou false, caso contrrio.
   * 
   * @throws IllegalArgumentException Caso a ao recebida seja nula.
   */
  public boolean addAction(final TransitionAction action) {
    if (action == null) {
      throw new IllegalArgumentException(
        "No  permitido incluir aes nulas (null).");
    }
    return this.actions.add(action);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final Symbol<?> symbol, final Session session)
    throws AutomatonException {
    for (final TransitionAction action : this.actions) {
      action.execute(symbol, session);
    }
  }

  /**
   * Cria uma composio de aes de transio.
   */
  public CompositeTransitionAction() {
    this.actions = new LinkedList<TransitionAction>();
  }

  /**
   * Cria uma composio de aes de transio.
   * 
   * @param actions As aes (No aceita {@code null}).
   */
  public CompositeTransitionAction(final TransitionAction... actions) {
    this();
    for (final TransitionAction action : actions) {
      addAction(action);
    }
  }
}
