package tecgraf.javautils.parsers;

import java.util.List;

import tecgraf.javautils.parsers.exception.AutomatonException;
import tecgraf.javautils.parsers.exception.InvalidStateException;
import tecgraf.javautils.parsers.iterators.SymbolIterator;

/**
 * Representa um autmato finito (mquina de estados finitos).
 */
public class FiniteAutomaton {

  /**
   * Representa o estado inicial do autmato.
   */
  private final State initialState;

  /**
   * Constri um autmato finito.
   * 
   * @param initialState O estado inicial do autmato.
   */
  protected FiniteAutomaton(State initialState) {
    this.initialState = initialState;
  }

  /**
   * Obtm o estado inicial do autmato.
   * 
   * @return O estado inicial.
   */
  public final State getInitialState() {
    return this.initialState;
  }

  /**
   * Inicia a execuo do autmato para os smbolos recebidos.
   * 
   * @param symbolIterator Um iterador com os smbolos que representam a entrada
   *        do autmato.
   * 
   * @return Uma lista com os tokens gerados pelo autmato.
   * 
   * @throws InvalidStateException Caso no existam mais smbolos e o autmato
   *         est num estado no-final.
   * @throws AutomatonException Caso ocorra algum erro durante a execuo do
   *         autmato.
   * @throws IllegalArgumentException Caso seja passado um iterador nulo.
   */
  public final List<Token> start(SymbolIterator symbolIterator)
    throws InvalidStateException, AutomatonException {
    if (symbolIterator == null) {
      throw new IllegalArgumentException(
        "No  possvel iterar sobre um iterador nulo (null).");
    }
    Session session = new Session();
    State state = this.initialState;
    state.enter(session);
    try {
      while (symbolIterator.hasNext()) {
        state.exit(session);
        state = state.getNext(symbolIterator.getNext(), session);
        state.enter(session);
      }
      state.exit(session);
      if (!state.isFinal()) {
        throw new InvalidStateException(session, state);
      }
    }
    finally {
      session.close();
    }
    return session.getTokenList();
  }
}
