package tecgraf.javautils.parsers;

import java.text.MessageFormat;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import tecgraf.javautils.parsers.symbols.Symbol;

/**
 * Representa uma sesso de execuo de um autmato finito (
 * {@link tecgraf.javautils.parsers.FiniteAutomaton}).
 */
public final class Session {
  /**
   * Representa o token corrente da sesso. Nele so adicionados os smbolos
   * pelo mtodo {@link #appendSymbol(Symbol)}. Quando o mtodo
   * {@link #generateToken()}  chamado, um novo token  criado.
   */
  private Token currentToken;

  /**
   * Representa uma lista com todos os tokens que j foram gerados na sesso
   * atual.
   */
  private final List<Token> tokenList;

  /**
   * Indica que a sesso j est fechada.
   */
  private boolean closed;

  /**
   * Cria uma sesso.
   */
  Session() {
    this.currentToken = new Token();
    this.tokenList = new LinkedList<>();
    this.closed = false;
  }

  /**
   * Adiciona um smbolo ao token corrente da sesso.
   * 
   * @param symbol O smbolo a ser inserido.
   * 
   * @throws IllegalArgumentException Caso o smbolo seja nulo.
   * @throws IllegalStateException Caso a sesso esteja fechada.
   */
  public void appendSymbol(Symbol<?> symbol) {
    if (symbol == null) {
      throw new IllegalArgumentException("O smbolo no pode ser nulo (null).");
    }
    if (this.isClosed()) {
      throw new IllegalStateException(
        "No  possvel inserir smbolos em uma sesso fechada.");
    }
    this.currentToken.addSymbol(symbol);
  }

  /**
   * Insere o token atual na lista de tokens gerados.
   * 
   * @throws IllegalStateException Caso a sesso esteja fechada.
   */
  public void generateToken() {
    if (this.isClosed()) {
      throw new IllegalStateException(
        "No  possvel gerar tokens em uma sesso fechada.");
    }
    this.tokenList.add(this.currentToken);
    this.currentToken = new Token();
  }

  /**
   * Fecha a sesso.
   */
  void close() {
    if (!this.isClosed()) {
      if (this.currentToken.getSize() > 0) {
        this.generateToken();
      }
      this.closed = true;
    }
  }

  /**
   * Verifica se a sesso est fechada.
   * 
   * @return true, se a sesso estiver fechada, ou false, caso contrrio.
   */
  private boolean isClosed() {
    return this.closed;
  }

  /**
   * <p>
   * Obtm a lista de tokens gerados sob a forma de uma {@link List} que no
   * permite modificaes.
   * </p>
   * 
   * <p>
   * Para maiores detalhes, consulte:
   * {@link Collections#unmodifiableList(java.util.List)}.
   * </p>
   * 
   * @return A lista de tokens gerados.
   */
  List<Token> getTokenList() {
    return Collections.unmodifiableList(this.tokenList);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return MessageFormat.format("Buffer: {0} - Tokens: {1}", this.currentToken, this.tokenList);
  }
}
