package tecgraf.javautils.parsers;

import java.util.Map;
import java.util.TreeMap;

import tecgraf.javautils.parsers.actions.StateAction;
import tecgraf.javautils.parsers.actions.TransitionAction;
import tecgraf.javautils.parsers.exception.AutomatonException;
import tecgraf.javautils.parsers.exception.NoTransitionException;
import tecgraf.javautils.parsers.symbols.Symbol;

/**
 * Representa um estado de um autmato finito (
 * {@link tecgraf.javautils.parsers.FiniteAutomaton}).
 */
public class State {
  /**
   * Faz o mapeamento entre os smbolos e as transies a serem executadas. As
   * chaves do mapeamento so do tipo {@link Symbol} e os valores so do tipo
   * {@link Transition}.
   */
  private final Map<Symbol<?>, Transition> transitionMap;

  /**
   * Define a transio default de um estado. Essa transio  executada quando
   * um smbolo no definido em {@link #transitionMap}  recebido pelo mtodo
   * {@link #getNext(Symbol, Session)}.
   */
  private Transition defaultTransition;

  /**
   * Define se o estado  final.
   */
  private final boolean isFinal;

  /**
   * Define a ao que ser executada ao entrar no estado.
   */
  private StateAction enterAction;

  /**
   * Define a ao que ser executada ao sair do estado.
   */
  private StateAction exitAction;

  /**
   * Constri um estado.
   * 
   * @param isFinal Define se o estado  final ou no.
   */
  public State(boolean isFinal) {
    this.isFinal = isFinal;
    this.transitionMap = new TreeMap<>();
  }

  /**
   * Constri um estado, com sua respectiva ao de entrada.
   * 
   * @param isFinal Define se o estado  final ou no.
   * @param enterAction A ao que ser executada ao se entrar no estado.
   * 
   * @throws IllegalArgumentException Caso seja recebida uma ao de entrada
   *         nula.
   */
  public State(boolean isFinal, StateAction enterAction) {
    this(isFinal);
    if (enterAction == null) {
      throw new IllegalArgumentException(
        "No  permitida uma ao de entrada nula (null).");
    }
    this.enterAction = enterAction;
  }

  /**
   * Constri um estado, com suas respectivas aes de entrada e sada.
   * 
   * @param isFinal Define se o estado  final ou no.
   * @param enterAction A ao que ser executada ao se entrar no estado.
   * @param exitAction A ao que ser executada ao sair do estado.
   * 
   * @throws IllegalArgumentException Caso seja recebida uma ao de sada nula.
   */
  public State(boolean isFinal, StateAction enterAction, StateAction exitAction) {
    this(isFinal, enterAction);
    if (exitAction == null) {
      throw new IllegalArgumentException(
        "No  permitida uma ao de sada nula (null).");
    }
    this.exitAction = exitAction;
  }

  /**
   *  executado pelo autmato quando este estado passa a ser o estado atual (ao
   * entrar no estado).
   * 
   * @param session A sesso de execuo do autmato.
   * 
   * @throws AutomatonException Caso ocorra algum erro ao executar a ao (
   *         {@link StateAction}) de entrada ({@link #enterAction}).
   */
  final void enter(Session session) throws AutomatonException {
    if (this.enterAction != null) {
      this.enterAction.execute(session);
    }
  }

  /**
   *  executado pelo autmato quando este estado deixa de ser o estado atual
   * (ao sair do estado).
   * 
   * @param session A sesso de execuo do autmato.
   * 
   * @throws AutomatonException Caso ocorra algum erro ao executar a ao (
   *         {@link StateAction}) de sada ({@link #exitAction}).
   */
  final void exit(Session session) throws AutomatonException {
    if (this.exitAction != null) {
      this.exitAction.execute(session);
    }
  }

  /**
   * Obtm o prximo estado do autmato finito ({@link FiniteAutomaton}).
   * 
   * @param symbol O smbolo de entrada.
   * @param session A sesso de execuo.
   * 
   * @return O prximo estado do autmato.
   * 
   * @throws NoTransitionException Caso no esteja definida nenhuma transio
   *         para o smbolo de entrada.
   * @throws AutomatonException Caso ocorra algum problema durante a transio
   *         para o prximo estado.
   */
  final State getNext(Symbol<?> symbol, Session session)
    throws NoTransitionException, AutomatonException {
    Transition transition = this.transitionMap.get(symbol);
    if (transition == null) {
      transition = this.defaultTransition;
    }
    if (transition == null) {
      throw new NoTransitionException(session, symbol, this);
    }
    return transition.execute(symbol, session);
  }

  /**
   * <p>
   * Adiciona uma transio que ser executada quando o smbolo for recebido
   * pelo mtodo {@link #getNext(Symbol, Session)}.
   * </p>
   * 
   * <p>
   * Obs: Caso seja necessrio executar vrios aes para um determinado
   * smbolo, utilize um objeto da classe
   * {@link tecgraf.javautils.parsers.actions.CompositeTransitionAction}.
   * </p>
   * 
   * @param symbol O smbolo que representa a transio.
   * @param function A funo a ser executada no momento da transio.
   * @param target O estado-destino da transio.
   * 
   * @throws IllegalArgumentException Caso algum dos parmetros recebidos seja
   *         nulo.
   */
  public final void addTransition(Symbol<?> symbol, TransitionAction function,
    State target) {
    if (symbol == null) {
      throw new IllegalArgumentException("O smbolo no pode ser nulo (null).");
    }
    this.addTransition(new Symbol[] { symbol }, function, target);
  }

  /**
   * <p>
   * Adiciona uma transio que ser executada quando algum dos smbolos for
   * recebido pelo mtodo {@link #getNext(Symbol, Session)}.
   * </p>
   * 
   * <p>
   * Obs: Caso seja necessrio executar vrios aes para um determinado
   * smbolo, utilize um objeto da classe
   * {@link tecgraf.javautils.parsers.actions.CompositeTransitionAction}.
   * </p>
   * 
   * @param symbols Os smbolos que representam a transio.
   * @param function A funo a ser executada no momento da transio.
   * @param target O estado-destino da transio.
   * 
   * @throws IllegalArgumentException Caso algum dos parmetros recebidos seja
   *         nulo.
   */
  public final void addTransition(Symbol<?>[] symbols,
    TransitionAction function, State target) {
    if (symbols == null) {
      throw new IllegalArgumentException(
        "O array de smbolos no pode ser nulo (null).");
    }
    if (function == null) {
      throw new IllegalArgumentException("A funo no pode ser nula (null).");
    }
    if (target == null) {
      throw new IllegalArgumentException(
        "O estado-destino no pode ser nulo (null).");
    }
    Transition transition = new Transition(target, function);
    for (Symbol<?> symbol : symbols) {
      this.transitionMap.put(symbol, transition);
    }
  }

  /**
   * Define a transio default do autmato.  utilizada quando no existe
   * transio especfica para um determinado smbolo.
   * 
   * @param function A funo a ser executada no momento da transio.
   * @param target O estado-destino da transio.
   * 
   * @throws IllegalArgumentException Caso algum dos parmetros seja nulo
   *         (null).
   */
  public final void setDefaultTransition(TransitionAction function, State target) {
    if (function == null) {
      throw new IllegalArgumentException("A funo no pode ser nula (null).");
    }
    if (target == null) {
      throw new IllegalArgumentException(
        "O estado-destino no pode ser nulo (null).");
    }
    this.defaultTransition = new Transition(target, function);
  }

  /**
   * Verifica se o estado  final ou no.
   * 
   * @return true, caso o estado seja final, ou false, caso contrrio.
   */
  public final boolean isFinal() {
    return this.isFinal;
  }

  /**
   * Define a ao que ser executada ao se entrar no estado.
   * 
   * @param enterAction A ao a ser executada.
   * 
   * @throws IllegalArgumentException Caso a ao de entrada seja nula.
   */
  public final void setEnterAction(StateAction enterAction) {
    if (enterAction == null) {
      throw new IllegalArgumentException(
        "A ao de entrada no pode ser nula (null).");
    }
    this.enterAction = enterAction;
  }

  /**
   * Define a ao que ser executada ao sair do estado.
   * 
   * @param exitAction A ao a ser executada.
   * 
   * @throws IllegalArgumentException Caso a ao de sada seja nula.
   */
  public final void setExitAction(StateAction exitAction) {
    if (exitAction == null) {
      throw new IllegalArgumentException(
        "A ao de sada no pode ser nula (null).");
    }
    this.exitAction = exitAction;
  }
}
