package tecgraf.javautils.parsers;

import tecgraf.javautils.parsers.actions.TransitionAction;
import tecgraf.javautils.parsers.exception.AutomatonException;
import tecgraf.javautils.parsers.symbols.Symbol;

/**
 * Representa uma transio entre estados (
 * {@link tecgraf.javautils.parsers.State} de um autmato finito (
 * {@link tecgraf.javautils.parsers.FiniteAutomaton}).
 */
public final class Transition {
  /** Representa a ao que ser executada no momento da transio (opcional). */
  private TransitionAction action;

  /** Representa o estado-destino da transio. */
  private State target;

  /**
   * Cria uma transio que ter o estado recebido como estado-destino.
   * 
   * @param target O estado-destino.
   * 
   * @throws IllegalArgumentException Caso o estado recebido seja nulo (null).
   */
  public Transition(State target) {
    if (target == null) {
      throw new IllegalArgumentException(
        "No  possvel executar uma transio para um estado nulo (null).");
    }
    this.target = target;
  }

  /**
   * Cria uma transio que ter o estado recebido como estado-destino e
   * executar a ao recebida quando a transio for executada.
   * 
   * @param target O estado-destino.
   * @param action A ao a ser executada no momento da transio.
   * 
   * @throws IllegalArgumentException Caso a ao de transio seja nula (null).
   */
  public Transition(State target, TransitionAction action) {
    this(target);
    if (action == null) {
      throw new IllegalArgumentException(
        "No  possvel executar uma transio que tenha uma ao nula (null).");
    }
    this.action = action;
  }

  /**
   * Executa a ao de transio (se houver) e retorna o estado-destino.
   * 
   * @param symbol O smbolo que est sendo utilizado atualmente pelo autmato
   *        finito.
   * @param session A sesso de execuo do autmato.
   * 
   * @return O estado-destino da transio.
   * 
   * @throws AutomatonException Caso ocorra algum problema executando a ao de
   *         transio.
   */
  State execute(Symbol<?> symbol, Session session) throws AutomatonException {
    if (this.action != null) {
      this.action.execute(symbol, session);
    }
    return this.target;
  }
}
