package tecgraf.javautils.pdfviewer.dialog;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.util.Locale;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JPanel;
import javax.swing.KeyStroke;

import tecgraf.javautils.pdfviewer.core.PDFCorePanel.ZoomFitPolicy;
import tecgraf.javautils.pdfviewer.core.PDFDocument;
import tecgraf.javautils.pdfviewer.viewer.PDFViewerPanel;

/**
 * Dilogo de exibio de PDFs
 * 
 * @author Tecgraf
 */
public class PDFDialog extends JDialog {

  /**
   * Painel interno
   */
  private PDFViewerPanel pdfViewerPanel;

  /**
   * Construtor
   * 
   * @param owner
   * @param title ttulo.
   * @param locale tratador do i18n.
   */
  public PDFDialog(final Window owner, final String title, final Locale locale) {
    super(owner, title, ModalityType.DOCUMENT_MODAL);
    pdfViewerPanel = new PDFViewerPanel(locale);
    setDefaultCloseOperation(DISPOSE_ON_CLOSE);
    addEscListener();
    final JPanel buttonsPanel = buildButtonsPanel();
    setLayout(new BorderLayout());
    add(buttonsPanel, BorderLayout.SOUTH);
    add(pdfViewerPanel, BorderLayout.CENTER);

  }

  /**
   * Exibe documento.
   * 
   * @param document documento a ser exibido
   */
  public void loadDocument(PDFDocument document) {
    pdfViewerPanel.loadDocument(document);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setVisible(boolean visible) {
    if (visible) {
      setDefaultDimension();
      setDefaultMode();
      setLocationRelativeTo(getOwner());
    }
    super.setVisible(visible);
  }

  /**
   * Ajuste de modo default.
   */
  private void setDefaultMode() {
    pdfViewerPanel.setZoomFitPolicy(ZoomFitPolicy.WHOLE_PAGE);
  }

  /**
   * Ajuste do tamnaho padro da janela.
   */
  private void setDefaultDimension() {
    setMinimumSize(new Dimension(300, 300));

    final Toolkit toolkit = Toolkit.getDefaultToolkit();
    final Dimension screenSize = toolkit.getScreenSize();
    final Dimension docSize = pdfViewerPanel.getDocumentPageSize();
    final int width;
    final int height;
    if (docSize.height == 0 || docSize.width == 0) {
      width = 700;
      height = 500;
    }
    else {
      final double docRatio = ((double) docSize.width) / docSize.height;
      final double factor = 0.80;
      if (docRatio > 1) {
        width = (int) Math.round(screenSize.width * factor * 0.5);
        height = (int) Math.round(width / docRatio);
      }
      else {
        height = (int) Math.round(screenSize.height * factor);
        width = (int) Math.round(height * docRatio);
      }
    }
    setSize((new Dimension(width, height)));
  }

  /**
   * Mtodo de internacionalizao.
   * 
   * @param tag tag
   * @return texto
   */
  private String getString(final String tag) {
    final String prefix = this.getClass().getSimpleName();
    final String realtag = prefix + "." + tag;
    final String title = pdfViewerPanel.getString(realtag);
    return title;
  }

  /**
   * Monta painel de botes.
   * 
   * @return painel
   */
  private JPanel buildButtonsPanel() {
    final AbstractAction closeAction = buildCloseAction();
    final JButton closeButton = new JButton(closeAction);
    final String title = getString("close.button");
    closeButton.setText(title);
    final JPanel buttonsPanel = new JPanel();
    buttonsPanel.setLayout(new FlowLayout(FlowLayout.CENTER));
    buttonsPanel.add(closeButton);
    return buttonsPanel;
  }

  /**
   * Ajusta a tecla esc para fechar o dilogo.
   */
  private void addEscListener() {
    final AbstractAction closeAction = buildCloseAction();
    final int mode = JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT;
    final InputMap inputMap = rootPane.getInputMap(mode);
    final ActionMap actionMap = rootPane.getActionMap();
    final int esc = KeyEvent.VK_ESCAPE;
    final KeyStroke cancelStroke = KeyStroke.getKeyStroke(esc, 0);
    final String actionMapKey = cancelStroke.toString();
    inputMap.put(cancelStroke, actionMapKey);
    actionMap.put(actionMapKey, closeAction);
  }

  /**
   * Monta ao de close.
   * 
   * @return ao
   */
  private AbstractAction buildCloseAction() {
    final AbstractAction closeAction = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        PDFDialog.this.setVisible(false);
        PDFDialog.this.dispose();
      }
    };
    return closeAction;
  }
}
