package tecgraf.javautils.sparkserver.standard;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.eclipse.jetty.websocket.api.Session;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketClose;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketConnect;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketMessage;
import org.eclipse.jetty.websocket.api.annotations.WebSocket;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import tecgraf.javautils.sparkserver.core.JuIWebSocketClass;

@WebSocket
abstract public class JuWebSocketClass implements JuIWebSocketClass {

  private static final Map<Class<? extends JuWebSocketClass>, List<Session>> sessionsMap = new HashMap<>();

  abstract public void connected(Session session);

  abstract public void closed(Session session, int statusCode, String reason);

  abstract public void messageReceived(Session session, String message);

  protected void broadcastString(String value) {
    getAllSessions().parallelStream().forEach(s -> {
      sendString(s, value);
    });
  }

  protected void sendString(Session session, String message) {
    final Logger logger = getLogger();
    try {
      session.getRemote().sendString(message);
      logger.info("Sent: " + session + " -- " + message);

    }
    catch (IOException e) {
      logger.error("Send failure: " + message, e);
    }
  }

  @OnWebSocketConnect
  public final void _connected(Session session) {
    this.addSession(session);
    this.connected(session);
    getLogger().info("Connected: " + session);
  }

  @OnWebSocketClose
  public final void _closed(Session session, int statusCode, String reason) {
    this.removeSession(session);
    this.closed(session, statusCode, reason);
    final Logger logger = getLogger();
    logger.info("Closed: " + session + " -- " + statusCode + " -- " + reason);
  }

  @OnWebSocketMessage
  public final void _messageReceived(Session session, String message) {
    this.messageReceived(session, message);
    final Logger logger = getLogger();
    logger.info("Received: " + session + " -- " + message);
  }

  @Override
  public Logger getLogger() {
    return LoggerFactory.getLogger(getClass());
  }

  private List<Session> getAllSessions() {
    final Class<? extends JuWebSocketClass> clazz = getClass();
    final List<Session> sessions = sessionsMap.get(clazz);
    if (sessions == null) {
      sessionsMap.put(clazz, new ArrayList<>());
    }
    return sessionsMap.get(clazz);
  }

  synchronized private void addSession(Session session) {
    final List<Session> sessions = getAllSessions();
    sessions.add(session);
  }

  synchronized private void removeSession(Session session) {
    final List<Session> sessions = getAllSessions();
    sessions.remove(session);
  }

}
