package tecgraf.javautils.sparkserver.utils;

import java.util.HashSet;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.swagger.annotations.Api;
import io.swagger.jaxrs.Reader;
import io.swagger.jaxrs.config.BeanConfig;
import io.swagger.models.Swagger;
import org.reflections.Reflections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import tecgraf.javautils.sparkserver.core.JuIController;
import tecgraf.javautils.sparkserver.core.JuIEndpoint;
import tecgraf.javautils.sparkserver.standard.JuServer;

public class JuSwaggerParser {

  private final static Logger logger = LoggerFactory.getLogger(JuSwaggerParser.class);

  public static String getSwaggerJsonString(JuServer server) throws JsonProcessingException {
    final JuSwaggerReader reader = new JuSwaggerReader(server);
    Swagger swagger = reader.read();
    String json = swaggerToJson(swagger);
    return json;
  }

  public static String getSwaggerJson(JuIController controller) throws JsonProcessingException {
    final String title = controller.getClass().getSimpleName().toLowerCase();
    Swagger swagger = getSwagger(controller);
    String json = swaggerToJson(swagger);
    return json;
  }

  public static Swagger getSwagger(JuIController controller) {
    final Class<? extends JuIController> controllerClass = controller.getClass();
    final String className = controllerClass.getSimpleName();
    logger.info("Search swagger for controller class: " + className);

    final Set<Class<?>> loadClasses = new HashSet<>();
    controller.getEndpoints().forEach(ep -> {
      final Class<?> endpointClass = ep.getClass();
      loadClasses.add(endpointClass);
      logger.info("Search swagger for endpoint class: " + endpointClass);
    });
    BeanConfig beanConfig = new BeanConfig();
    beanConfig.setTitle(className);
    beanConfig.setScan(true);
    beanConfig.setPrettyPrint(true);
    beanConfig.scanAndRead();
    beanConfig.setResourcePackage(controllerClass.getPackageName());

    Swagger swagger = beanConfig.getSwagger();
    final Reader reader = new Reader(swagger);
    return reader.read(loadClasses);
  }

  public static String swaggerToJson(Swagger swagger) throws JsonProcessingException {
    ObjectMapper objectMapper = new ObjectMapper();
    objectMapper.setSerializationInclusion(JsonInclude.Include.NON_EMPTY);
    String json = objectMapper.writeValueAsString(swagger);
    return json;
  }

}

