package tecgraf.javautils.sparkserver.standard;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import tecgraf.javautils.sparkserver.core.JuIController;
import tecgraf.javautils.sparkserver.core.JuIEndpoint;
import tecgraf.javautils.sparkserver.core.JuIFunction;
import tecgraf.javautils.sparkserver.core.JuIWebSocket;

public class JuController implements JuIController {

  private final List<JuIEndpoint> endpoints = new ArrayList<>();
  private final List<JuIWebSocket> websockets = new ArrayList<>();

  private String pathPrefix;
  private String description;
  private String name;

  protected JuController() {
  }

  @Override
  public String getPathPrefix() {
    return pathPrefix;
  }

  public void setPathPrefix(String pathPrefix) {
    if (pathPrefix != null) {
      if (pathPrefix.indexOf("/") != -1) {
        throw new IllegalArgumentException("path-prefix cannot contain '/'");
      }
    }
    this.pathPrefix = pathPrefix;
  }

  @Override
  public Stream<JuIEndpoint> getEndpoints() {
    return endpoints.stream();
  }

  @Override
  public Stream<JuIWebSocket> getWebSockets() {
    return websockets.stream();
  }

  @Override
  public String getName() {
    return (name == null ? this.getClass().getSimpleName() : name.trim());
  }

  @Override
  public String getDescription() {
    return (description == null ? "" : description.trim());
  }

  @Override
  public JuIController setName(String name) {
    this.name = name;
    return this;
  }

  @Override
  public JuIController setDescription(String description) {
    this.description = description;
    return this;
  }

  protected <T> JuIEndpoint addEndpoint(JuIEndpoint endpoint) {
    this.endpoints.add(endpoint);
    return endpoint;
  }

  protected <T> JuIEndpoint addEndpoint(JuVerb verb, String path, JuRoute<T> route) {
    final JuEndpoint ep = new JuEndpoint(verb, path, route);
    return addEndpoint(ep);
  }

  protected <T> JuIEndpoint addEndpoint(JuVerb verb, String path, JuIFunction<T> function, Class<T> responseClass) {
    final JuIEndpoint ep = new JuEndpoint(verb, path, function, responseClass);
    this.endpoints.add(ep);
    return addEndpoint(ep);
  }

  protected <T> JuIWebSocket addWebSocket(String path, Class<? extends JuWebSocketClass> clazz) {
    final String realPath = getRealPath(path);
    final JuWebSocket websocket = new JuWebSocket(realPath, clazz);
    this.websockets.add(websocket);
    return websocket;
  }

  protected Logger getLogger() {
    return LoggerFactory.getLogger(this.getClass());
  }

  private String getRealPath(String path) {
    final String prefix = getPathPrefix();
    final boolean noPath = path == null || path.trim().isEmpty();
    final boolean noPrefix = prefix == null || prefix.trim().isEmpty();
    if (noPath && noPrefix) {
      return "/";
    }
    if (noPath) {
      return prefix;
    }
    if (noPrefix) {
      return path;
    }
    return prefix + "/" + path;
  }
}
