package tecgraf.javautils.version;

import java.io.Serializable;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Utilit�rio para tratamento vers�es de elementos de programa��o: classes,
 * sistemas, arquivos e/ou qualquer outra entidade.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class VersionNumber implements Comparable<VersionNumber>, Serializable {

  /** Maior valor suportado para itens (partes) de vers�o: major, minor ou patch */
  public static final int MAX_PART = 99;

  /** Menor vers�o poss�vel de existir. */
  public static final VersionNumber ZERO_VERSION = new VersionNumber(0, 0, 0);

  /** Maior vers�o poss�vel de existir. */
  public static final VersionNumber LIMIT_VERSION = new VersionNumber(MAX_PART,
    MAX_PART, MAX_PART);

  /** Formatador para exibi��o da vers�o como identificador */
  private static final String TAG_FORMATTER = "v%01d_%02d_%02d";

  /** Formatador para exibi��o da vers�o como identificador. */
  private static final String BRANCH_FORMATTER = "v%01d_%02d";

  /** Formatador para exibi��o da vers�o como texto. */
  private static final String TXT_FORMATTER = "%d.%d.%d";

  /** Regex para n�mero (parte de vers�o) */
  private static final String DRX = "\\d+{1,2}";

  /** Regex para grupo de n�mero (parte de vers�o) */
  private static final String GDRX = "(" + DRX + ")";

  /** Regex para busca de vers�o em string com identificador tag */
  private static final String TAG_REGEX = String.format("v%s_%s_%s", GDRX,
    GDRX, GDRX);

  /** Regex para busca de vers�o em string com identificador branch */
  private static final String BRANCH_REGEX = String
    .format("v%s_%s", GDRX, GDRX);

  /** Regex para busca de vers�o em string com texto. */
  private static final String TXT_REGEX = String.format("%s\\.%s\\.%s", GDRX,
    GDRX, GDRX);

  /**
   * Pattern de busca de vers�o com base em tag (pr�-compilado para otimiza��o).
   */
  static final Pattern TAG_PATT = Pattern.compile(TAG_REGEX);

  /**
   * Pattern de busca de vers�o com base em branch (pr�-compilado para
   * otimiza��o).
   */
  static final Pattern BRANCH_PATT = Pattern.compile(BRANCH_REGEX);

  /**
   * Pattern de busca de vers�o com base em texto (pr�-compilado para
   * otimiza��o).
   */
  static final Pattern TXT_PATT = Pattern.compile(TXT_REGEX);

  /** Constante de invalidade das partes da vers�o. */
  private static final int INVALID = -9999;

  /** Indica que objeto est� congelado e n�o pode mais ser alterado. */
  private boolean isFrozen = false;

  /** Major (parte formadora da vers�o) */
  private int major;

  /** Minor (parte formadora da vers�o) */
  private int minor;

  /** Patch (parte formadora da vers�o) */
  private int patch;

  /**
   * Construtor
   * 
   * @param major major
   * @param minor minor
   * @param patch patch
   */
  public VersionNumber(int major, int minor, int patch) {
    this.major = major;
    this.minor = minor;
    this.patch = patch;
    check();
  }

  /**
   * Monta uma vers�o com base em texto.
   * 
   * @param text texto
   * @return versão ou {@code null} no caso de falha de formatação do texto
   */
  public static VersionNumber fromString(String text) {
    int[] idxs = new int[] { 1, 2, 3 };
    return fromStringAsPattern(TXT_PATT, text, idxs);
  }

  /**
   * Monta uma vers�o com base em texto de tag.
   * 
   * @param text texto
   * @return versão ou {@code null} no caso de falha de formatação do texto
   */
  public static VersionNumber fromStringAsBranch(String text) {
    int[] idxs = new int[] { 1, 2, 1 };
    VersionNumber version = fromStringAsPattern(BRANCH_PATT, text, idxs);
    if (version == null) {
      return null;
    }
    version.setPatch(0);
    return version;
  }

  /**
   * Monta uma vers�o com base em texto.
   * 
   * @param pattern padr�o de busca para string.
   * @param text texto
   * @param indexes �ndices.
   * 
   * @return vers�o
   */
  private static VersionNumber fromStringAsPattern(Pattern pattern,
    String text, int[] indexes) {
    if (text == null) {
      return null;
    }
    Matcher matcher = pattern.matcher(text);
    if (!matcher.matches()) {
      return null;
    }
    String strMajor = matcher.group(indexes[0]);
    String strMinor = matcher.group(indexes[1]);
    String strPatch = matcher.group(indexes[2]);
    if (strMajor == null || strMinor == null || strPatch == null) {
      return null;
    }
    try {
      int mj = Integer.parseInt(strMajor);
      int mn = Integer.parseInt(strMinor);
      int pt = Integer.parseInt(strPatch);
      VersionNumber version = new VersionNumber(mj, mn, pt);
      return version;
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Monta uma vers�o com base em texto de tag.
   * 
   * @param text texto
   * @return vers�o
   */
  public static VersionNumber fromStringAsTag(String text) {
    int[] idxs = new int[] { 1, 2, 3 };
    return fromStringAsPattern(TAG_PATT, text, idxs);
  }

  /**
   * Indica validade de uma parte de vers�o.
   * 
   * @param part parte
   * @return indicativo
   */
  public static boolean isValidPart(int part) {
    return part >= 0 && part <= VersionNumber.MAX_PART;
  }

  /**
   * M�todo para ajustar uma vers�o para a poss�vel pr�xima (eventualmente n�o
   * existente); devendo ser usado basicamente para forma��o de algoritmos de
   * l�gica de vers�o.
   */
  public void add() {
    patch += 1;
    if (patch <= MAX_PART) {
      return;
    }

    patch = 0;
    minor += 1;
    if (minor <= MAX_PART) {
      return;
    }

    patch = 0;
    minor = 0;
    major += 1;
    if (major > MAX_PART) {
      invalidate();
    }
  }

  /**
   * Check do objeto levantando exec��o caso os atributos internos estejam
   * inconsistentes (n�meros major, minor e patch).
   */
  public void check() {
    checkPart(major);
    checkPart(minor);
    checkPart(patch);
  }

  /** M�todo interno de levantamento de exce��o. */
  private void checkFrozen() {
    if (isFrozen) {
      throw new IllegalStateException("version data is frozen.");
    }
  }

  /**
   * M�todo interno de verifica��o.
   * 
   * @param part parte da vers�o
   */
  private void checkPart(int part) {
    if (part > MAX_PART || part < 0) {
      String err = String.format("bad part version number: %s", part);
      throw new IllegalStateException(err);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected VersionNumber clone() {
    return new VersionNumber(major, minor, patch);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(VersionNumber version) {
    if (version == null) {
      throw new IllegalArgumentException("null version for comparison");
    }
    if (major != version.major) {
      return major > version.major ? 1 : -1;
    }
    if (minor != version.minor) {
      return minor > version.minor ? 1 : -1;
    }
    if (patch != version.patch) {
      return patch > version.patch ? 1 : -1;
    }
    return 0;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    final VersionNumber other = (VersionNumber) obj;
    if (major != other.major) {
      return false;
    }
    if (minor != other.minor) {
      return false;
    }
    if (patch != other.patch) {
      return false;
    }
    return true;
  }

  /** Congela altera��o no dado. */
  public void freeze() {
    this.isFrozen = true;
  }

  /**
   * Major.
   * 
   * @return valor
   */
  public int getMajor() {
    return major;
  }

  /**
   * Minor.
   * 
   * @return valor
   */
  public int getMinor() {
    return minor;
  }

  /**
   * Patch.
   * 
   * @return o valor
   */
  public int getPatch() {
    return patch;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + major;
    result = prime * result + minor;
    result = prime * result + patch;
    return result;
  }

  /**
   * Ajusta para uma nova vers�o (com incremento de major); que <b>N�O</b> faz
   * altera��es dependentes entre as partes.
   * 
   * @param deltaMajor incremento major (podendo ser negativo)
   * @param deltaMinor incremento minor (podendo ser negativo)
   * @param deltaPatch incremento patch (podendo ser negativo)
   */
  public void increment(int deltaMajor, int deltaMinor, int deltaPatch) {
    checkFrozen();
    setMajor(getMajor() + deltaMajor);
    setMinor(getMinor() + deltaMinor);
    setPatch(getPatch() + deltaPatch);
  }

  /** Torna dado inv�lido. */
  public void invalidate() {
    checkFrozen();
    this.major = INVALID;
    this.minor = INVALID;
    this.patch = INVALID;
  }

  /**
   * Indica se dado � valido (ou foi inicializado).
   * 
   * @return indicativo
   */
  public boolean isValid() {
    int mj = getMajor();
    int mn = getMinor();
    int pt = getPatch();
    boolean okMajor = isValidPart(mj);
    boolean okMinor = isValidPart(mn);
    boolean okPatch = isValidPart(pt);
    return okMajor && okMinor && okPatch;
  }

  /**
   * Ajusta o valor de: marcador de vers�o (ver {@link #major}).
   * 
   * @param major o novo valor a ser ajustado
   */
  public void setMajor(int major) {
    checkFrozen();
    checkPart(major);
    this.major = major;
  }

  /**
   * Ajusta o valor de: marcador de vers�o (ver {@link #minor}).
   * 
   * @param minor o novo valor a ser ajustado
   */
  public void setMinor(int minor) {
    checkFrozen();
    checkPart(minor);
    this.minor = minor;
  }

  /**
   * Ajusta o valor de: marcador de vers�o (ver {@link #patch}).
   * 
   * @param patch o novo valor a ser ajustado
   */
  public void setPatch(int patch) {
    checkFrozen();
    checkPart(patch);
    this.patch = patch;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return String.format(TXT_FORMATTER, major, minor, patch);
  }

  /**
   * Retorna um identificador da vers�o.
   * 
   * @return texto que pode ser usado como identificador.
   */
  final public String toStringAsBranch() {
    final String text = String.format(BRANCH_FORMATTER, major, minor);
    return text;
  }

  /**
   * Retorna um identificador da vers�o.
   * 
   * @return texto que pode ser usado como identificador.
   */
  public String toStringAsTag() {
    return String.format(TAG_FORMATTER, major, minor, patch);
  }
}
