/*
 * $Id: XMLBasicHandler.java 2014 2006-02-10 16:24:43 +0000 (Fri, 10 Feb 2006)
 * costa $
 */
package tecgraf.javautils.xml;

import java.io.Reader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.helpers.DefaultHandler;

import tecgraf.javautils.xml.exception.XMLException;
import tecgraf.javautils.xml.exception.XMLInternalErrorException;

/**
 * Handler default. Esta classe  extendida pela aplicao, mas tipicamente
 * apenas os mtodos fatalError, error e warning precisam ser redefinidos.
 * 
 * @author Andre Oliveira da Costa
 */
public class XMLBasicHandler implements XMLHandlerInterface {

  /**
   * Elemento da pilha de elementos j processados mas aguardando coleta pelo
   * "pai". Cada elemento est associado ao seu respectivo nvel na hierarquia
   * XML para possibilitar identificao inequvoca (se usssemos apenas o nome
   * no permitiramos que tags fossem filhas delas mesmas).
   */
  private static class StackElement {
    /**
     * Elemento XML.
     */
    XMLElementInterface element;
    /**
     * Nvel na hierarquia XML associado ao elemento.
     */
    int level;

    /**
     * Construtor.
     * 
     * @param element elemento XML
     * @param level nvel do elemento na hierarquia do XML
     */
    StackElement(XMLElementInterface element, int level) {
      this.element = element;
      this.level = level;
    }

    /**
     * Verifica se o elemento corrente possui uma determinada tag e est em um
     * nvel especfico.
     * 
     * @param tagName nome da tag
     * @param level nvel
     * @return <code>true</code> se o elemento possui a mesma tag e est no
     *         mesmo nvel
     */
    boolean equals(String tagName, int level) {
      return element.getTag().equalsIgnoreCase(tagName) && this.level == level;
    }
  }

  /**
   *  medida que vo sendo processados os elementos vo sendo colocados nesta
   * pilha. Quando os elementos so criados (callback <code>startElement</code>)
   * os elementos so anexados  pilha; ela s esvazia quando um elemento 
   * fechado, pois todos os seus filhos so "coletados". O prprio elemento,
   * porm, permanece na pilha, para que possa ser coletado pelo seu pai.<br>
   * Isto significa que esta pilha tipicamente contm tambm elementos j
   * fechados (i.e. elementos para os quais j foi recebido a callback
   * <code>endElement</code>).
   */
  private Stack<StackElement> stack;

  /**
   * Esta pilha, ao contrrio de <code>stack</code>, armazena apenas os
   * elementos que ainda se encontram abertos; assim que um elemento  fechado
   * ele  removido da pilha.<br>
   * Ela  necessria para evitar que elementos j fechados recebam eventos de
   * caracteres destinados a outros elementos.<br>
   * Observar que esta pilha armazena referncias para os mesmos elementos da
   * pilha <code>stack</code> (o que  conveniente, pois queremos realmente
   * lidar com os mesmos elementos).
   */
  private Stack<XMLElementInterface> openedElements;

  /** Raiz */
  private XMLElementInterface root;

  /** Entrada */
  private InputSource inputSource;

  /** Fabrica */
  private XMLElementFactoryInterface xmlFactory;

  /** DTD */
  private String dtd;

  /** handler interno ao SAX */
  final private InternalHandler internalHandler;

  /**
   * Nvel corrente na hierarquia do XML. Usado para controlar a pilha com os
   * filhos de cada tag.
   */
  private int tagLevel;

  /**
   * Retorna o elemento-raiz do documento
   * 
   * @return raiz
   */
  @Override
  final public XMLElementInterface getRootElement() {
    return root;
  }

  /**
   * Retorna o handler interno (SAX) usado por este handler.
   * 
   * @return handler interno (SAX) usado por este handler
   */
  @Override
  final public DefaultHandler getSAXInternalHandler() {
    return internalHandler;
  }

  /**
   * Evento de leitura de caracteres (CDATA). Deve anexar os caracteres lidos ao
   * valor do elemento atualmente no topo da pilha de elementos abertos (
   * <code>openedElements</code>).
   * 
   * @param ch - buffer onde foram lidos os caracteres
   * @param start - posio no buffer do primeiro caracter lido
   * @param length - quantidade de caracteres lidos
   * @throws XMLException
   * 
   * @see DefaultHandler#characters(char[], int, int)
   */
  @Override
  final public void characters(final char[] ch, final int start,
    final int length) throws XMLException {
    XMLElementInterface element = openedElements.peek();
    element.newCharsEvent(ch, start, length);
  }

  /**
   * Evento de nova tag. Delega  factory a responsabilidade de criar o objeto
   * correto. Coleta os atributos do elemento e os armazena em um mapa no
   * prprio elemento. Se for o primeiro elemento a ser criado, identifica-o
   * como raiz da rvore.
   * 
   * @param uri - o identificador do namespace do elemento, ou a string vazia se
   *        o elemento no possui namespace especfico ou se o processamento de
   *        namespaces no est habilitado
   * @param localName - o nome local (sem prefixo), ou a string vazia se o
   *        processamento de namespaces no est habilitado
   * @param qName - o nome qualificado da tag
   * @param attributes - lista de atributos associados  tag
   * @throws XMLException
   * 
   * @see DefaultHandler#startElement(String, String, String, Attributes)
   */
  @Override
  final public void startElement(final String uri, final String localName,
    final String qName, final List<XMLAttribute> attributes)
    throws XMLException {
    XMLElementInterface element = null;
    element = xmlFactory.createXMLElementFromTag(qName);
    if (element == null) {
      throw new XMLException("no existe elemento associado  tag " + qName);
    }
    /*
     * como o elemento acabou de ser "aberto", deve constar das duas pilhas
     */
    stack.push(new StackElement(element, tagLevel));
    openedElements.push(element);
    /*
     * descemos +1 nvel na hierarquia do XML
     */
    tagLevel++;

    /*
     * "desempacotamos" os atributos SAX do elemento para facilitar uso
     * posterior
     */
    if (attributes != null && attributes.size() > 0) {
      element.newAttributeList(attributes);
    }

    /*
     * se o stack agora contm apenas um elemento, trata-se da raiz do documento
     */
    if (stack.size() == 1) {
      root = element;
    }

    /* sinaliza para o objeto que este acabou de ser criado */
    element.startTag();
  }

  /**
   * Evento de tag de fechamento de um elemento. Neste momento, temos que
   * desempilhar quaisquer elementos que existam na pilha e defini-los como
   * filhos do elemento em questo.  nesta hora tambm que o objeto da
   * aplicao correspondente ser criado pela aplicao (se for o caso).
   * <p>
   * Esta callback executa o mtodo <code>endTag</code> do elemento, passando
   * como parmetro uma lista com os filhos deste (a lista estar vazia caso o
   * elemento no tenha filhos).
   * 
   * @param uri - URL associada ao namespace do elemento. Caso o elemento no
   *        tenha namespace ou o uso de namespaces esteja desabilitado,  uma
   *        string vazia
   * @param localName - o nome local ou uma string vazia caso o uso de
   *        namespaces esteja desabilitado
   * @param qName - o nome do elemento (tag)
   * @throws XMLException
   */
  @Override
  final public void endElement(final String uri, final String localName,
    final String qName) throws XMLException {

    /*
     * como este elemento acaba de ser fechado, temos que retir-lo da pilha de
     * elementos abertos. O elemento pode ou no permanecer na pilha "stack".
     */
    openedElements.pop();
    /*
     * voltamos 1 nvel na hierarquia do XML
     */
    tagLevel--;

    /*
     * Todos os elementos criados desde a criao do elemento em questo devem
     * ser cadastrados como filhos deste
     */
    StackElement stackTop = stack.peek();
    if (stackTop == null) {
      throw new XMLInternalErrorException(qName,
        "pilha vazia ao desempilhar a tag");
    }
    ArrayList<XMLElementInterface> tempChildrenList =
      new ArrayList<XMLElementInterface>();
    /*
     * enquanto o elemento no topo da pilha no for o elemento corrente,
     * coletamos o topo da pilha e o acrescentamos  lista de filhos do elemento
     * corrente
     */
    while (!stackTop.equals(qName, tagLevel)) {
      stack.pop();
      /*
       * Temos que inserir no incio da lista para que os elementos fiquem na
       * ordem correta
       */
      tempChildrenList.add(0, stackTop.element);
      stackTop = stack.peek();
    }
    stackTop.element.convertValueFromXML();

    /*
     * Damos ao objeto XML a chance de criar sua contraparte na lgica, se for
     * este o caso
     */
    stackTop.element.endTag(tempChildrenList);
  }

  /**
   * Define o inputSource do handler a partir de um <code>Reader</code>.
   * 
   * @param reader <code>Reader</code> a ser usado para leitura do XML
   */
  @Override
  final public void setInputSource(final Reader reader) {
    inputSource = new InputSource(reader);
  }

  /**
   * Retorna o <code>InputSource</code> associado a este handler.
   * 
   * @return <code>InputSource</code> associado ao handler
   */
  @Override
  final public InputSource getInputSource() {
    return inputSource;
  }

  /**
   * Evento chamado quando uma <i>external entity</i> (p.ex. a URL associada ao
   * DTD)  resolvida.
   * <p>
   * Este mtodo deve ser sobrescrito pelas classes que precisam redefinir a
   * localizao do DTD.
   * 
   * @param publicId FIXME entender do que se trata... na prtica, raramente 
   *        usado
   * @param systemId para declaraes do tipo
   *        <code>&lt;!DOCTYPE xxx SYSTEM yyy&gt;</code>, o valor deste
   *        parmetro  yyy (a referncia ao DTD)
   * @return null para sinalizar que o tratamento default deve ser aplicado
   * @throws XMLException
   * 
   * @see DefaultHandler#resolveEntity(String, String)
   */
  @Override
  public InputSource resolveEntity(final String publicId, final String systemId)
    throws XMLException {
    /*
     * FIXME se esta callback por algum motivo puder ser chamada mais de uma vez
     * para um mesmo documento, redefiniremos o DTD. Verificar.
     */
    setDTD(systemId);
    return null;
  }

  /**
   * Define o DTD associado ao documento. Usado no processo de escrita do XML.
   * 
   * @param dtd
   */
  @Override
  final public void setDTD(final String dtd) {
    this.dtd = dtd;
  }

  /**
   * Retorna o DTD associado ao documento.
   * 
   * @return DTD associado ao documento.
   */
  @Override
  final public String getDTD() {
    return dtd;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public XMLElementFactoryInterface getXMLFactory() {
    return xmlFactory;
  }

  /**
   * Callback executada ao final do processamento do documento.
   * 
   * @throws XMLException
   */
  @Override
  public final void endDocument() throws XMLException {
    // FIXME por simetria, deveramos sinalizar tambm o evento startDocument?
    endDocument(root);
  }

  /**
   * Callback executada quando o elemento foi inteiramente lido. Recebe como
   * parmetro o elemento-raiz do XML. Aplicaes que precisem ser notificadas
   * quando a leitura do documento for concluda devem redefinir este mtodo.
   * 
   * @param rootElement raiz do XML
   */
  @Override
  public void endDocument(final XMLElementInterface rootElement) {
    /* vazio */
  }

  /**
   * Evento associado a um warning. Deve ser redefinido pelas subclasses.
   * 
   * @param e - exceo que gerou o warning
   * @throws XMLException
   */
  @Override
  public void warning(final XMLException e) throws XMLException {
    throw e;
  }

  /**
   * Evento associado a um erro fatal. Deve ser redefinido pelas subclasses.
   * 
   * @param e - exceo que gerou o erro
   * @throws XMLException
   */
  @Override
  public void fatalError(final XMLException e) throws XMLException {
    throw e;
  }

  /**
   * Evento associado a um erro. Deve ser redefinido pelas subclasses.
   * 
   * @param e - exceo que gerou o erro
   * @throws XMLException
   * 
   * @see org.xml.sax.ErrorHandler#error(org.xml.sax.SAXParseException)
   */
  @Override
  public void error(final XMLException e) throws XMLException {
    throw e;
  }

  /**
   * Cria um handler que usa uma {@link XMLElementFactoryInterface fbrica} para
   * criar objetos XML a partir das tags.
   * 
   * @param xmlFactory fabrica
   */
  public XMLBasicHandler(final XMLElementFactoryInterface xmlFactory) {
    internalHandler = new InternalHandler(this);
    this.xmlFactory = xmlFactory;
    stack = new Stack<StackElement>();
    openedElements = new Stack<XMLElementInterface>();
    root = null;
    tagLevel = 0;
  }

  /**
   * Cria um handler que usa um mapa para criar objetos XML a partir das tags.
   * 
   * @param tagToObjMap mapa relacionando tags a objetos XML
   */
  public XMLBasicHandler(
    Map<String, Class<? extends XMLElementInterface>> tagToObjMap) {
    this(new XMLBasicElementFactory(tagToObjMap, null));
  }

  /**
   * Cria um handler que mapeia todas as tags para um nico tipo de objeto XML.
   * 
   * @param cls classe do objeto XML que ser mapeado a todas as tags
   */
  public XMLBasicHandler(Class<? extends XMLElementInterface> cls) {
    this(new XMLBasicElementFactory(cls));
  }
}
