/* $Id: XMLElement.java 123004 2011-10-13 14:35:40Z costa $ */

package tecgraf.javautils.xml;

import java.io.IOException;
import java.io.Writer;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import tecgraf.javautils.xml.exception.XMLParseException;

/**
 * Classe-base para todos os elementos XML, cuja visibilidade est restrita ao
 * pacote; classes da aplicao que modelam elementos XML devem extender
 * <code>XMLElement</code> ou <code>XMLLeafElement</code>.
 * 
 * @author Andr Oliveira da Costa
 */
public abstract class XMLElement implements XMLElementInterface {

  /**
   * Estado do buffer interno.
   */
  private enum BufferState {
    /**
     * Fechado.
     */
    CLOSED,
    /**
     * Aberto pelo parser.
     */
    OPENED_BY_PARSER,
    /**
     * Aberto pelo usurio.
     */
    OPENED_BY_USER,
  }

  /** Lista de atributos. */
  private List<XMLAttribute> attributes;

  /**
   * Buffer temporrio para armazenamento do valor associado  tag enquanto esta
   * no  fechada.
   */
  private StringBuffer valueBuffer;

  /**
   * Valor do elemento XML (texto entre as tags de abertura e fechamento). A
   * interpretao do mesmo de acordo com a sua semntica cabe  aplicao.
   */
  private String value;

  /**
   * Flag que indica se o processamento do valor do elemento j foi concludo.
   */
  private BufferState bufferState;

  /** Tag */
  private String tag;

  /** Objeto de contexto da aplicao. */
  private Object contextObject;

  /** Objeto analogo na aplicao. */
  private Object appObject;

  /** Mapa de converso de entidades XML (lt = &lt;", gt = "&gt;" etc.) */
  private Map<String, String> xmlConversionTable;

  /** Expresso regular para converso de entidades XML para texto. */
  private String xmlToTextRegexp;

  /**
   * Evento SAX associado  leitura de caracteres associados ao valor do
   * elemento XML. Esta implementao anexa os caracteres ao valor do elemento.
   * 
   * @param charArray array de caracteres contendo os chars lidos
   * @param start incio dos dados no array
   * @param length quantidade de caracteres lidos
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#newCharsEvent(char[],int,int)
   * @see XMLBasicHandler#characters(char[], int, int)
   */
  @Override
  public void newCharsEvent(final char[] charArray, final int start,
    final int length) {
    if (valueBuffer == null) {
      initBuffer(BufferState.OPENED_BY_PARSER);
    }
    valueBuffer.append(charArray, start, length);
  }

  /**
   * Inicializa o buffer interno.
   * 
   * @param state estado inicial (criado pelo parser ou pelo usurio)
   */
  private void initBuffer(BufferState state) {
    bufferState = state;
    if (value != null) {
      valueBuffer = new StringBuffer(value);
    }
    else {
      valueBuffer = new StringBuffer();
    }
  }

  /**
   * Anexa parte de um array de caracteres ao um valor, e soma a estes uma
   * string. Usado na converso de caracteres de XML para texto.
   * 
   * @param value valor que ser alterado
   * @param charArray array a ser anexado
   * @param string string a ser anexada
   * @param refPos primeira posio do array a ser considerada
   * @param currPos posio corrente no array
   * 
   * @return nova posio corrente (aps o final da string anexada)
   */
  private static int appendStringToValue(final StringBuffer value,
    final char[] charArray, final String string, final int refPos,
    final int currPos) {
    int length = currPos - refPos;
    if (length > 0) {
      value.append(charArray, refPos, length);
    }
    value.append(string);
    return refPos + length + 1;
  }

  /**
   * Converte o valor associado ao elemento para texto (converte sequncias de
   * caracteres XML para seus correspondentes ASCII). No caso desta
   * implementao, converte tambm o buffer temporrio associado ao valor para
   * uma string. S a partir deste momento os mtodos que consultam e usam o
   * valor do elemento podem ser executados.
   * 
   * Executado pelo <i>handler</i> quando termina o processamento do elemento.
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#convertValueFromXML()
   * @see XMLBasicHandler#endElement(String, String, String)
   * @see #getStrValue()
   * @see #hasValue()
   * @see #isEmpty()
   */
  @Override
  final public void convertValueFromXML() {
    if (valueBuffer == null) {
      /*
       * ou o elemento no possua valor ou a converso j foi feita
       */
      consolidateValue();
      return;
    }
    final Pattern pattern = Pattern.compile(xmlToTextRegexp);
    final Matcher matcher = pattern.matcher(valueBuffer);
    final StringBuffer newValue = new StringBuffer(valueBuffer.length());
    while (matcher.find()) {
      final String capture = matcher.group();
      final String replacement = xmlConversionTable.get(capture);
      if (replacement == null) {
        // FIXME lanar exceo
        // sequncias no reconhecidas so anexadas sem passar por converso
        System.err.println("sequencia nao tratada: &" + capture + ';');
        newValue.append('&' + capture + ';');
      }
      else {
        matcher.appendReplacement(newValue, replacement);
      }
    }
    matcher.appendTail(newValue);
    valueBuffer = newValue;
    consolidateValue();
  }

  /**
   * Finaliza o processamento do valor. Converte o buffer temporrio na string
   * associada ao valor, zera o buffer e sinaliza que o valor est pronto para
   * ser consultado/utilizado.
   */
  private void consolidateValue() {
    value = valueBuffer == null ? null : valueBuffer.toString();
    valueBuffer = null;
    bufferState = BufferState.CLOSED;
  }

  /**
   * Converte texto ASCII para XML, fazendo escape dos caracteres &lt;, &gt; e
   * &amp;.
   * 
   * @param text texto.
   * @see tecgraf.javautils.xml.XMLElementInterface#convertTextToXML(java.lang.String)
   */
  @Override
  @Deprecated
  public String convertTextToXML(final String text) {
    return XMLElement.xmlEncode(text);
  }

  /**
   * Codifica texto para XML:
   * <ul>
   * <li>&amp; para &amp;amp;
   * <li>&gt; para &amp;gt;
   * <li>&lt; para &amp;lt;
   * </ul>
   * 
   * @param text - texto a ser convertido
   * @return nova string com o texto convertido
   */
  @Deprecated
  public static String xmlEncode(final String text) {
    final char[] charArray = text.toCharArray();
    final StringBuffer converted = new StringBuffer(text.length());
    int start = 0;
    for (int i = 0; i < charArray.length; i++) {
      final char c = charArray[i];
      switch (c) {
        case '<':
          start = appendStringToValue(converted, charArray, "&lt;", start, i);
          break;
        case '>':
          start = appendStringToValue(converted, charArray, "&gt;", start, i);
          break;
        case '&':
          start = appendStringToValue(converted, charArray, "&amp;", start, i);
          break;
      }
    }

    // se sobraram caracteres, anexa-os ao valor
    if (charArray.length > start) {
      converted.append(charArray, start, charArray.length - start);
    }
    return converted.toString();
  }

  /**
   * Adiciona um valor ao elemento (<i>append</i>).
   * 
   * @param text valor textual a ser acrescido. Se for <code>null</code> no 
   *        considerado
   * @param newLine flag indicativo de quebra de linha aps a acrscimo.
   * @see tecgraf.javautils.xml.XMLElementInterface#appendValue(java.lang.String,boolean)
   */
  @Override
  final public XMLElementInterface appendValue(final String text,
    final boolean newLine) {
    if (valueBuffer == null) {
      initBuffer(BufferState.OPENED_BY_USER);
    }
    if (text != null) {
      valueBuffer.append(text);
      if (newLine) {
        valueBuffer.append('\n');
      }
    }
    return this;
  }

  /**
   * Define o valor "vazio" para o elemento (na verdade, remove o valor)
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#resetValue()
   */
  @Override
  final public XMLElementInterface resetValue() {
    valueBuffer = null;
    value = null;
    bufferState = BufferState.CLOSED;
    return this;
  }

  /**
   * Define o valor do elemento.
   * 
   * @param value o valor a ser usado.
   * @see tecgraf.javautils.xml.XMLElementInterface#setValue(java.lang.String)
   */
  @Override
  final public XMLElementInterface setValue(final String value) {
    if (value == null) {
      resetValue();
      return this;
    }
    this.value = value;
    bufferState = BufferState.CLOSED;
    return this;
  }

  /**
   * Define um valor double para o elemento, convertendo-o para texto.
   * 
   * @param value o valor <u><i>double</i></u> a ser ajustado.
   * @see tecgraf.javautils.xml.XMLElementInterface#setValue(double)
   */
  @Override
  final public XMLElementInterface setValue(final double value) {
    return setValue(String.valueOf(value));
  }

  /**
   * Define a tag do elemento.
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#setTag(java.lang.String)
   */
  @Override
  final public XMLElementInterface setTag(final String tag) {
    this.tag = tag;
    return this;
  }

  /**
   * Retorna a tag associada ao elemento.
   * 
   * @return a tag do elemento.
   * @see tecgraf.javautils.xml.XMLElementInterface#getTag()
   */
  @Override
  public final String getTag() {
    return tag;
  }

  /**
   * Retorna o valor associado ao elemento, com a opo de ser feito escape dos
   * caracteres problemticos (&lt;, &gt;, &amp; etc.).
   * 
   * @param escapeChars se igual a true, os caracteres problemticos sero
   *        tratados
   * 
   * @return valor do elemento
   * @throws IllegalStateException se o elemento ainda est sendo processado
   */
  public String getStrValue(final boolean escapeChars) {
    if (escapeChars) {
      return XMLElement.xmlEncode(getStrValue());
    }
    return getStrValue();
  }

  /**
   * Retorna o valor do elemento como texto. Se no h valor associado, retorna
   * uma string vazia.
   * 
   * @return o valor do elemento (como texto) ou uma string vazia
   * @see tecgraf.javautils.xml.XMLElementInterface#getStrValue()
   * @throws IllegalStateException se o elemento ainda est sendo processado
   */
  @Override
  public String getStrValue() {
    checkBufferState();
    return value == null ? "" : value;
  }

  /**
   * Verifica se o buffer est em um estado em que pode ser consolidado.
   */
  private void checkBufferState() {
    switch (bufferState) {
      case OPENED_BY_PARSER:
        /*
         * o parser ainda est processando o elemento
         */
        throw new IllegalStateException("elemento ainda est sendo processado");

      case OPENED_BY_USER:
        /*
         * o buffer foi criado pelo usurio, assumimos que j foi completamente
         * preenchido
         */
        consolidateValue();
        break;

      default:
        // vazio
    }
  }

  /**
   * Retorna o valor do elemento como um <code>float</code>. Caso a converso
   * para texto no seja possvel, lana <code>XMLParseException</code>.
   * 
   * @return o valor como um <code>float</code>.
   * @see tecgraf.javautils.xml.XMLElementInterface#getFloatValue()
   * @throws IllegalStateException se o elemento ainda est sendo processado
   */
  @Override
  public final float getFloatValue() {
    try {
      return Float.valueOf(getStrValue().trim()).floatValue();
    }
    catch (NumberFormatException e) {
      throw new XMLParseException(tag);
    }
  }

  /**
   * Retorna o valor do elemento como um <code>double</code>. Caso a converso
   * para texto no seja possvel, lana <code>XMLParseException</code>.
   * 
   * @return o valor como um <code>double</code>.
   * @see tecgraf.javautils.xml.XMLElementInterface#getDoubleValue()
   * @throws IllegalStateException se o elemento ainda est sendo processado
   */
  @Override
  public final double getDoubleValue() {
    try {
      return Double.valueOf(getStrValue().trim()).doubleValue();
    }
    catch (NumberFormatException e) {
      throw new XMLParseException(tag);
    }
  }

  /**
   * Retorna o valor do elemento como um <code>int</code>. Caso a converso para
   * texto no seja possvel, lana <code>XMLParseException</code>.
   * 
   * @return o valor como um <code>int</code>.
   * @see tecgraf.javautils.xml.XMLElementInterface#getIntValue()
   * @throws IllegalStateException se o elemento ainda est sendo processado
   */
  @Override
  public final int getIntValue() {
    try {
      return Integer.valueOf(getStrValue().trim()).intValue();
    }
    catch (NumberFormatException e) {
      throw new XMLParseException(tag);
    }
  }

  /**
   * Retorna o valor do elemento como um <code>long</code>. Caso a converso
   * para texto no seja possvel, lana <code>XMLParseException</code>.
   * 
   * @return o valor como um <code>long</code>.
   * @see tecgraf.javautils.xml.XMLElementInterface#getLongValue()
   * @throws IllegalStateException se o elemento ainda est sendo processado
   */
  @Override
  public final long getLongValue() {
    try {
      return Long.valueOf(getStrValue().trim()).longValue();
    }
    catch (NumberFormatException e) {
      throw new XMLParseException(tag);
    }
  }

  /**
   * Define um novo atributo para o elemento. FIXME atualmente, permite
   * mltiplas ocorrncias de um atributo (ver observao em
   * getAttributeStrValue())
   * 
   * @param name o nome do atributo.
   * @param strValue o valor do atributo (string).
   * @see tecgraf.javautils.xml.XMLElementInterface#newAttribute(java.lang.String,java.lang.String)
   */
  @Override
  public final XMLElementInterface newAttribute(final String name,
    final String strValue) {
    attributes.add(new XMLAttribute(name, strValue));
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public XMLElementInterface newAttribute(String name, Enum<?> enumValue) {
    return newAttribute(name, enumValue.toString());
  }

  /**
   * Define um novo atributo com valor inteiro para o elemento.
   * 
   * @param name o nome do atributo.
   * @param intValue o valor do atributo (inteiro).
   * @see tecgraf.javautils.xml.XMLElementInterface#newAttribute(java.lang.String,int)
   */
  @Override
  public final XMLElementInterface newAttribute(String name, int intValue) {
    return newAttribute(name, String.valueOf(intValue));
  }

  /**
   * Define um novo atributo com valor inteiro (long) para o elemento.
   * 
   * @param name o nome do atributo.
   * @param longValue o valor do atributo (inteiro long).
   * @see tecgraf.javautils.xml.XMLElementInterface#newAttribute(java.lang.String,int)
   */
  @Override
  public final XMLElementInterface newAttribute(String name, long longValue) {
    return newAttribute(name, String.valueOf(longValue));
  }

  /**
   * Define um novo atributo com valor booleano para o elemento.
   * <code>true</code> ser gravado como "TRUE", e <code>false</code> como
   * "FALSE".
   * 
   * @param name o nome do atributo.
   * @param boolValue o valor do atributo (booleano).
   * @see tecgraf.javautils.xml.XMLElementInterface#newAttribute(java.lang.String,boolean)
   */
  @Override
  public XMLElementInterface newAttribute(String name, boolean boolValue) {
    return newAttribute(name, boolValue == true ? "TRUE" : "FALSE");
  }

  /**
   * Define um novo atributo com valor <code>double</code> para o elemento.
   * 
   * @param name o nome do atributo.
   * @param doubleValue o valor do atributo (<code>double</code>).
   * @see tecgraf.javautils.xml.XMLElementInterface#newAttribute(java.lang.String,double)
   */
  @Override
  public final XMLElementInterface newAttribute(String name, double doubleValue) {
    return newAttribute(name, String.valueOf(doubleValue));
  }

  /**
   * Redefine os atributos do elemento a partir de uma lista externa. A lista j
   * deve conter elementos do tipo <code>XMLAttribute</code>.
   * 
   * @param newAttrs lista de novos atributos.
   * @see tecgraf.javautils.xml.XMLElementInterface#newAttributeList(java.util.List)
   */
  @Override
  public final XMLElementInterface newAttributeList(
    final List<XMLAttribute> newAttrs) {
    attributes.clear();
    attributes.addAll(newAttrs);
    return this;
  }

  /**
   * Reinicializa a lista de atributos do elemento.
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#resetAttributeList()
   */
  @Override
  public final void resetAttributeList() {
    attributes.clear();
  }

  /**
   * Consulta da existencia de um atributo.
   * 
   * @param name nome do atributo.
   * @return flag indicativo.
   * @see tecgraf.javautils.xml.XMLElementInterface#hasAttribute(java.lang.String)
   */
  @Override
  public final boolean hasAttribute(final String name) {
    return (getAttributeStrValue(name) != null);
  }

  /**
   * Consulta da existencia de valor. Observar que qualquer caracter (incluindo
   * espaos em branco e terminadores de linha) configura a existncia de valor.
   * 
   * <ul>
   * <li>&lt;tag&gt;&lt;/tag&gt; - hasValue() == false
   * <li>&lt;tag&gt; &lt;/tag&gt; - hasValue() == true
   * <li>&lt;tag&gt;<br>
   * &lt;/tag&gt; - hasValue() == true
   * <li>&lt;tag&gt;qualquer coisa&lt;/tag&gt; - hasValue() == true
   * </ul>
   * 
   * @return flag indicativo.
   * 
   * @throws IllegalStateException se o elemento ainda est sendo processado
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#hasValue()
   * @see #isEmpty()
   */
  @Override
  public final boolean hasValue() {
    checkBufferState();
    return (value != null);
  }

  /**
   * Indica se o elemento no possui valor ({@link #hasValue()} == false) ou se
   * o valor  constitudo apenas por espaos ou terminadores de linha.
   * 
   * <ul>
   * <li>&lt;tag&gt;&lt;/tag&gt; - isEmpty() == true
   * <li>&lt;tag&gt; &lt;/tag&gt; - isEmpty() == true
   * <li>&lt;tag&gt;<br>
   * &lt;/tag&gt; - isEmpty() == true
   * <li>&lt;tag&gt;qualquer coisa diferente de espaos&lt;/tag&gt; - isEmpty()
   * == false
   * </ul>
   * 
   * @return true se o valor do elemento  uma string vazia ou  constitudo
   *         apenas por espaos ou terminadores de linha
   * @throws IllegalStateException se o elemento ainda est sendo processado
   * 
   * @see #hasValue()
   */
  @Override
  public final boolean isEmpty() {
    return getStrValue().trim().isEmpty();
  }

  /**
   * Consulta ao valor de um atributo. Esta implementao usa busca sequencial.
   * Se for o caso, implementar um mapeamento auxiliar (p.ex. mapa [nome -->
   * ndice]) para otimizar o acesso. Outra alternativa seria usar aqui mapas
   * [nome --> {valor, ndice}], e ordenar os valores pelo ndice (talvez em uma
   * estrutura auxiliar) quando for necessrio exibi-los.
   * 
   * @param name nome do atributo.
   * @return valor do atributo como <code>String</code>
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#getAttributeStrValue(java.lang.String)
   */
  @Override
  public String getAttributeStrValue(final String name) {
    for (Iterator<XMLAttribute> iterator = attributes.iterator(); iterator
      .hasNext();) {
      final XMLAttribute attr = iterator.next();
      if (name.equals(attr.getName())) {
        return attr.getValue();
      }
    }
    return null;
  }

  /**
   * Retorna o valor de um atributo como <code>boolean</code>.
   * 
   * @param name nome do atributo.
   * @return true se o atributo tem como valor a string "true"
   *         (case-insensitive), false caso contrrio
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#getAttributeBooleanValue(java.lang.String)
   */
  @Override
  public boolean getAttributeBooleanValue(final String name) {
    return Boolean.valueOf(getAttributeStrValue(name)).booleanValue();
  }

  /**
   * Retorna o valor de um atributo como <code>int</code>.
   * 
   * @param name nome do atributo.
   * @return inteiro associado ao valor. Se a converso no for possvel, lana
   *         <code>XMLParseException</code>
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#getAttributeIntValue(java.lang.String)
   */
  @Override
  public final int getAttributeIntValue(final String name) {
    try {
      return Integer.valueOf(getAttributeStrValue(name)).intValue();
    }
    catch (NumberFormatException e) {
      throw new XMLParseException(tag, name);
    }
  }

  /**
   * Retorna o valor de um atributo como <code>double</code>.
   * 
   * @param name nome do atributo.
   * @return <code>double</code> associado ao valor. Se a converso no for
   *         possvel, lana <code>XMLParseException</code>
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#getAttributeIntValue(java.lang.String)
   */
  @Override
  public double getAttributeDoubleValue(String name) {
    try {
      return Double.valueOf(getAttributeStrValue(name)).doubleValue();
    }
    catch (NumberFormatException e) {
      throw new XMLParseException(tag, name);
    }
  }

  /**
   * Retorna uma cpia <i>read-only</i> da lista de atributos.
   * 
   * @return cpia read-only da lista de atributos.
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#getAttributes()
   */
  @Override
  public List<? extends XMLAttribute> getAttributes() {
    return Collections.unmodifiableList(attributes);
  }

  /**
   * Escreve a tag de abertura do elemento com seus respectivos atributos, sem
   * anexar fim-de-linha.
   * 
   * @param stream stream de sada
   * @param ident identao
   * @throws IOException se houver algum erro de escrita
   */
  @Override
  public void writeStartTag(final Writer stream, final String ident)
    throws IOException {
    stream.write(ident + "<" + getTag());
    writeAttributes(stream);
    stream.write(">");
  }

  /**
   * Escreve a tag de abertura do elemento com seus respectivos atributos,
   * anexando um fim-de-linha ao final da mesma.
   * 
   * @param stream stream de sada
   * @param ident identao
   * @throws IOException se houver algum erro de escrita
   */
  @Override
  public void writeStartTagln(final Writer stream, final String ident)
    throws IOException {
    writeStartTag(stream, ident);
    stream.write('\n');
  }

  /**
   * Escreve os atributos do elemento na forma nome="valor", na ordem em que
   * estes foram armazenados.
   * 
   * @param stream de sada
   * @throws IOException se houver algum erro de escrita
   */
  @Override
  public void writeAttributes(final Writer stream) throws IOException {
    writeAttributes(stream, getAttributes());
  }

  /**
   * Escreve uma lista de atributos no stream corrente, na forma nome="valor".
   * 
   * @param writer stream de sada
   * @param attributes lista de <code>XMLAttribute</code>
   * @throws IOException se houver algum erro de escrita
   */
  public static void writeAttributes(final Writer writer,
    final List<? extends XMLAttribute> attributes) throws IOException {
    if (attributes != null && attributes.isEmpty() == false) {
      for (Iterator<? extends XMLAttribute> iterator = attributes.iterator(); iterator
        .hasNext();) {
        XMLAttribute attr = iterator.next();
        if (attr.getValue() != null) {
          writer.write(" " + attr.getName() + "=\"" + attr.getValue() + "\"");
        }
      }
    }
  }

  /**
   * Escreve o valor do elemento. Os caracteres &amp;, &lt; e &gt; so
   * devidamente codificador para HTML.
   * 
   * @param stream stream de sada
   * @throws IOException se houver algum erro de escrita
   */
  @Override
  public void writeValue(final Writer stream) throws IOException {
    stream.write(convertTextToXML(getStrValue()));
  }

  /**
   * Escreve a tag de fechamento do elemento.
   * 
   * @param stream stream de sada
   * @param ident identao
   * @throws IOException se houver algum erro de escrita
   */
  @Override
  public void writeEndTag(final Writer stream, final String ident)
    throws IOException {
    stream.write(ident + "</" + getTag() + ">\n");
  }

  /**
   * Escreve a tag de fechamento do elemento, sem identao. Usado no fechamento
   * de <i>data elements</i>.
   * 
   * @param stream stream de sada
   * @throws IOException se houver algum erro de escrita
   */
  @Override
  final public void writeEndTag(final Writer stream) throws IOException {
    writeEndTag(stream, "");
  }

  /**
   * Implementao default (vazia) para a callback de incio do processamento de
   * um elemento.
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#startTag()
   */
  @Override
  public void startTag() {
    /* vazio */
  }

  /**
   * Define o objeto da aplicao associado ao elemento. Usado tipicamente
   * durante o processo de escrita dos elementos, onde esses so criados a
   * partir de objetos da aplicao e precisam extrair destes os dados que sero
   * escritos no XML.
   * 
   * @param appObject objeto da aplicao associado ao elemento XML
   * @return o prprio elemento, para encadeamento de operaes
   */
  @Override
  final public XMLElementInterface setAppObject(final Object appObject) {
    this.appObject = appObject;
    return this;
  }

  /**
   * Retorna o objeto da aplicao associado ao elemento XML. Este objeto no
   * necessariamente precisa existir (i.e. o objeto pode ser null).
   * 
   * @return o objeto da aplicao associado ao elemento XML
   */
  @Override
  final public Object getAppObject() {
    return appObject;
  }

  /**
   * Associa um contexto da aplicao ao elemento XML.
   * 
   * @param contextObject objeto representativo de contexto da aplicacao.
   * @return o prprio elemento, para encadeamento de operaes
   */
  @Override
  final public XMLElementInterface setAppContextObject(
    final Object contextObject) {
    this.contextObject = contextObject;
    return this;
  }

  /**
   * Retorna o contexto da aplicao associado ao elemento XML.
   * 
   * @return o objeto representativo de contexto da aplicacao.
   */
  @Override
  final public Object getAppContextObject() {
    return contextObject;
  }

  /**
   * Inicializa a tabela de converso de XML para texto, usada na converso de
   * &amp;lt; para &lt;, p.ex.
   */
  protected XMLElement() {
    xmlConversionTable = new HashMap<String, String>();
    xmlConversionTable.put("lt", "<");
    xmlConversionTable.put("gt", ">");
    xmlConversionTable.put("amp", "&");

    /*
     * A rigor, a regexp abaixo poderia ser construda automaticamente, se isto
     * se mostrar interessante.
     */
    xmlToTextRegexp = "&(lt|gt|amp)&";
    attributes = new LinkedList<XMLAttribute>();
    bufferState = BufferState.CLOSED;
  }

  /**
   * Constri um elemento recebendo sua tag como parmetro.
   * 
   * @param tagName tag
   */
  protected XMLElement(String tagName) {
    this();
    setTag(tagName);
  }
}
