/*
 * $Id: XMLHandlerInterface.java 2018 2006-02-10 16:48:20 +0000 (Fri, 10 Feb
 * 2006) costa $
 */
package tecgraf.javautils.xml;

import java.io.Reader;
import java.util.List;

import org.xml.sax.InputSource;
import org.xml.sax.helpers.DefaultHandler;

import tecgraf.javautils.xml.exception.XMLException;

/**
 * Interface para os handlers de eventos usados pelo parser. Define o conjunto
 * de operaes que devem ser implementadas pelo handler definido pela
 * aplicao.
 * 
 * @author Andre Oliveira da Costa
 */
public interface XMLHandlerInterface {

  /**
   * Retorna o elemento-raiz do documento
   * 
   * @return raiz
   */
  public XMLElementInterface getRootElement();

  /**
   * Evento de leitura de caracteres (CDATA). Deve anexar os caracteres lidos ao
   * valor do objeto atualmente no topo da pilha.
   * 
   * @param ch array de caracteres contendo os caracteres lidos
   * @param start incio dos caracteres lidos no array
   * @param length nmero de caracteres lidos
   * @throws XMLException se houver algum problema no processamento
   */
  public void characters(final char[] ch, final int start, final int length)
    throws XMLException;

  /**
   * Evento de nova tag. Delega  factory a responsabilidade de criar o objeto
   * correto. Coleta os atributos do elemento e os armazena em um mapa no
   * prprio elemento. Se for o primeiro elemento a ser criado, identifica-o
   * como raiz da rvore.
   * 
   * FIXME comentar parmetros
   * 
   * @param uri .
   * @param localName .
   * @param qName .
   * @param attributes lista de objetos do tipo <code>XMLAttribute</code>.
   * 
   * @throws XMLException caso tenha havido algum problema no processamento
   */
  public void startElement(final String uri, final String localName,
    final String qName, final List<XMLAttribute> attributes)
    throws XMLException;

  /**
   * Evento de tag de fechamento de um elemento. Neste momento, temos que
   * desempilhar quaisquer elementos que existam na pilha e defini-los como
   * filhos do elemento em questo.  nesta hora tambm que o objeto da
   * aplicao correspondente ser criado (se for o caso).
   * 
   * FIXME comentar parmetros
   * 
   * @param uri .
   * @param localName .
   * @param qName .
   * 
   * @throws XMLException caso tenha havido algum problema no processamento
   */
  public void endElement(final String uri, final String localName,
    final String qName) throws XMLException;

  /**
   * Define o inputSource do handler a partir de um <code>InputStream</code>,
   * usando um determinado encoding.
   * 
   * @param input stream de entrada
   */
  public void setInputSource(final Reader input);

  /**
   * Retorna o <code>inputSource</code> associado ao handler.
   * 
   * @return <code>inputSource</code> associado ao handler.
   */
  public InputSource getInputSource();

  /**
   * Retorna o DTD associado ao documento.
   * 
   * @return DTD associado ao documento.
   */
  public String getDTD();

  /**
   * Define o DTD associado ao elemento. Usado na escrita do XML.
   * 
   * @param dtd o DTD
   */
  public void setDTD(final String dtd);

  /**
   * Callback que permite  aplicao resolver "entidades externas". O uso mais
   * comum para este mtodo  interceptar a definio do DTD no documento (a URL
   * do DTD  passada como parmetro <code>systemId</code>)
   * 
   * @param publicId - o "identificador pblico", ou null se nenhum tiver sido
   *        definido (normalmente,  null).
   * @param systemId - o <i>system identifier</i> como definido no documento
   *        XML (p.ex. a URL do DTD)
   * @return um novo {@link InputSource} para a entidade externa, ou null
   *         indicando que o tratamento default deve ser aplicado.
   * @throws XMLException
   * @see org.xml.sax.EntityResolver#resolveEntity(java.lang.String,
   *      java.lang.String)
   * @see InputSource
   */
  public InputSource resolveEntity(final String publicId, final String systemId)
    throws XMLException;

  /**
   * Retorna a fbrica de elementos XML associada ao handler.
   * 
   * @return um objeto que implemente <code>XMLElementFactoryInterface</code>
   */
  public XMLElementFactoryInterface getXMLFactory();

  /**
   * Callback executada ao final do processamento do documento.
   * 
   * @throws XMLException
   */
  public void endDocument() throws XMLException;

  /**
   * Callback executada ao final do processamento do documento. Recebe como
   * parmetro o elemento-raiz do XML. Aplicaes que precisem ser notificadas
   * quando a leitura do documento for concluda devem redefinir este mtodo.
   * 
   * @param rootElement raiz do XML
   */
  public void endDocument(final XMLElementInterface rootElement);

  /**
   * Evento associado a um warning. Deve ser redefinido pelas subclasses.
   * 
   * @param e - exceo
   * @throws XMLException se a exceo precisar ser relanada
   */
  public void warning(final XMLException e) throws XMLException;

  /**
   * Evento associado a um erro fatal. Deve ser redefinido pelas subclasses.
   * 
   * @param e - exceo
   * @throws XMLException se a exceo precisar ser relanada
   */
  public void fatalError(final XMLException e) throws XMLException;

  /**
   * Evento associado a um erro. Deve ser redefinido pelas subclasses.
   * 
   * @param e - exceo
   * @throws XMLException se a exceo precisar ser relanada
   */
  public void error(final XMLException e) throws XMLException;

  /**
   * Retorna o handler interno (SAX) usado por este handler.
   * 
   * @return handler interno (SAX) usado por este handler
   */
  public DefaultHandler getSAXInternalHandler();

}
