/*
 * $Id: XMLListIterator.java 2019 2006-02-10 16:48:38 +0000 (Fri, 10 Feb 2006)
 * costa $
 */

package tecgraf.javautils.xml;

import java.util.Iterator;
import java.util.List;

/**
 * A classe <code>XMLListIterator</code> implementa um iterador sobre uma lista
 * de elementos do tipo XMLElementInterface, retornando a cada iterao o
 * prximo elemento cuja tag  igual  tag especificada no construtor (ou null
 * caso no existam mais elementos com esta tag). Esta implementao assume que
 * os objetos so adjacentes na lista.
 * 
 * @author costa
 * 
 * @deprecated use {@link ImprovedXMLListIterator}
 */
@Deprecated
public class XMLListIterator implements XMLListIteratorInterface {

  /** Tag usada para iterar sobre qualquer elemento */
  public static final String ANY = "*";

  /**
   * Lista sobre a qual estamos iterando.
   */
  private List<XMLElementInterface> list;

  /** Iterador sobre uma lista de elementos */
  private Iterator<XMLElementInterface> iterator;

  /** Tag de referncia */
  private String tag;

  /** ltimo elemento visitado. */
  private XMLElementInterface lastSeen;

  /**
   * Se igual a true, indica que <code>lastSeen</code> deve ser usado na
   * prxima iterao.
   */
  private boolean useLast;

  /** Indica se a tag corrente ANY. */
  private boolean matchAny;

  /**
   * Elemento antecipado da lista pelo mtodo <code>hasNext()</code> mas ainda
   * no considerado.
   */
  private XMLElementInterface pendingElement;

  /**
   * Constri um iterador sobre uma lista, buscando por uma tag especfica.
   * 
   * @param list lista de elementos
   * @param tag tag a ser buscada
   * @throws IllegalArgumentException se a lista  null
   */
  public XMLListIterator(final List<XMLElementInterface> list, final String tag) {
    if (list == null) {
      throw new IllegalArgumentException("lista de elementos == null");
    }
    // TODO deveramos fazer uma cpia read-only da lista?
    this.list = list;
    init(tag);
  }

  /**
   * Efetua as inicializaes comuns  criao do objeto e  operao de
   * reinicializao do mesmo (reset).
   * 
   * @param _tag tag de busca
   */
  private void init(final String _tag) {
    iterator = list.iterator();
    this.tag = _tag;
    this.matchAny = ANY.equals(_tag);
    /*
     * inicializamos o ltimo elemento visitado com o primeiro elemento da lista
     * cuja tag  igual  especificada. Se no houver tal elemento, lastSeen =
     * null
     */
    lastSeen = getNextMatchOnList(false);
    /*
     * se a tag do primeiro elemento no for a que buscamos, sinalizamos que
     * devemos buscar outro elemento (useLast = false)
     */
    this.useLast = (lastSeen != null);
  }

  /**
   * Constri um iterador sobre qualquer elemento (tag de referncia "ANY").
   * 
   * @param list lista de elementos
   */
  public XMLListIterator(final List<XMLElementInterface> list) {
    this(list, ANY);
  }

  /**
   * {@inheritDoc}
   */
  public void reset(final String _tag) {
    init(_tag);
  }

  /**
   * {@inheritDoc}
   */
  public void reset() {
    init(ANY);
  }

  /**
   * Retorna o prximo elemento da lista cuja tag  igual  tag de busca.
   * 
   * @param lookOnlyNext se igual a true, apenas o prximo elemento na lista
   *        ser avaliado; caso contrrio, a lista prosseguir at que um
   *        elemento compatvel seja encontrado ou a lista tenho sido
   *        inteiramente percorrida
   * 
   * @return prximo elemento da lista cuja tag  igual  tag de busca ou
   *         <code>null</code> caso isto no seja possvel
   */
  private XMLElementInterface getNextMatchOnList(final boolean lookOnlyNext) {
    XMLElementInterface match = null;
    while (listHasNext()) {
      match = getNextElementOnList();
      if (tagMatchesOurs(match) || lookOnlyNext == true) {
        break;
      }
    }
    return match;
  }

  /**
   * Retorna o prximo elemento a ser analisado. Se h um elemento pendente, o
   * retorna; caso contrrio, retorna o prximo elemento da lista, ou
   * <code>null</code> caso no haja mais elementos na lista.
   * 
   * @return elemento pendente OU prximo elemento da lista OU <code>null</code>,
   *         nesta ordem
   */
  private XMLElementInterface getNextElementOnList() {
    if (pendingElement != null) {
      XMLElementInterface temp = pendingElement;
      pendingElement = null;
      return temp;
    }
    return iterator.hasNext() ? (XMLElementInterface) iterator.next() : null;
  }

  /**
   * Indica se a tag associada a um determinado elemento  igual  tag de busca.
   * A tag de busca pode ser ANY; neste caso, este mtodo retorna
   * <code>true</code>.
   * 
   * @param element elemento de base para a comparao
   * 
   * @return <code>true</code> se a tag do elemento  igual  tag de busca ou
   *         se esta  ANY; <code>false</code> caso contrrio.
   */
  private boolean tagMatchesOurs(final XMLElementInterface element) {
    if (element == null)
      return false;
    return matchAny || tag.equals(element.getTag());
  }

  /**
   * Indica se h mais um elemento a ser considerado, seja este um elemento
   * pendente ou outro elemento da lista.
   * 
   * @return <code>true</code> se existe um elemento pendente OU se a lista
   *         ainda possui elementos
   */
  private boolean listHasNext() {
    return (pendingElement != null) ? true : iterator.hasNext();
  }

  /**
   * Retorna o prximo elemento da lista cuja tag  igual  solicitada.
   * 
   * @param lookOnlyNext define se apenas o prximo elemento da lista deve ser
   *        investigado (true) ou se a busca deve prosseguir at que um elemento
   *        seja encontrado ou a lista seja inteiramente percorrida
   * 
   * @return prximo elemento da lista cuja tag  igual  solicitada ou
   *         <code>null</code> caso isto no seja possvel
   */
  private XMLElementInterface next(final boolean lookOnlyNext) {
    if (useLast == false) {
      /*
       * no devemos aproveitar o ltimo elemento visitado, pegamos o prximo
       */
      lastSeen = getNextMatchOnList(lookOnlyNext);
    }

    /*
     * se o ltimo elemento visitado  null, encerramos o percorrimento
     */
    if (lastSeen == null) {
      useLast = true;
      return null;
    }

    /*
     * se a tag for ANY retornamos o ltimo elemento visitado
     */
    if (matchAny) {
      useLast = false;
      return lastSeen;
    }

    /*
     * neste ponto o cenrio : - o ltimo elemento visitado  != null - a tag
     * procurada  != ANY
     * 
     * se a tag do ltimo elemento visitado no for a que estamos buscando,
     * sinalizamos que o elemento deve ser reusado na prxima iterao e
     * retornamos null
     */
    if (tag.equals(lastSeen.getTag()) == false) {
      useLast = true;
      return null;
    }
    /*
     * aqui o elemento  diferente de null e a tag  a que buscamos; sinalizamos
     * que o elemento foi "consumido" e retornamos o mesmo.
     */
    useLast = false;
    return lastSeen;
  }

  /**
   * {@inheritDoc}
   */
  public XMLElementInterface next() {
    return next(true);
  }

  /**
   * {@inheritDoc}
   */
  public XMLElementInterface next(final String _tag) {
    if (lastSeen == null)
      return null;
    boolean lookOnlyNext = true;
    if (this.tag.equals(_tag) == false) {
      this.tag = _tag;
      matchAny = ANY.equals(_tag);
      /*
       * reusamos o ltimo elemento apenas se ele  do tipo que nos interessa.
       * se a tag anterior era != ANY e mudamos para ANY, no devemos reusar o
       * ltimo elemento visitado
       */
      useLast = matchAny == false && tagMatchesOurs(lastSeen);
      /*
       * a nova tag de busca  diferente da atual, portanto queremos que a busca
       * continue at encontrar um elemento do tipo desejado ou termos visitado
       * todos os elementos
       */
      lookOnlyNext = false;
    }
    return next(lookOnlyNext);
  }

  /**
   * {@inheritDoc}
   */
  public boolean hasNext() {
    if (lastSeen == null)
      return false;
    if (useLast)
      return tagMatchesOurs(lastSeen);
    if (pendingElement != null)
      return tagMatchesOurs(pendingElement);
    pendingElement = getNextElementOnList();
    if (pendingElement == null)
      return false;
    return tagMatchesOurs(pendingElement);
  }

  /**
   * {@inheritDoc}
   */
  public String getTag() {
    return tag;
  }
}
