/* $Id: XMLParser.java 75314 2008-04-24 18:44:54Z costa $ */

package tecgraf.javautils.xml;

import java.io.IOException;
import java.io.Reader;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.SAXException;

import tecgraf.javautils.xml.exception.XMLParseException;

/**
 * Implementao do parser default (usa um parser SAX).
 * 
 * @author Andr Oliveira da Costa
 */
public class XMLParser {

  /** Parser do SAX */
  private SAXParser parser;

  /** Tratador bsico */
  private XMLHandlerInterface handler;

  /**
   * Cria um novo parser usando o handler definido pela aplicao. O parmetro
   * <code>validate</code> define se o parser validar os arquivos XML contra
   * seus respectivos DTDs.<br>
   * <b>IMPORTANTE:</b> ainda que a validao seja desativada (<code>validate = 
   * false</code>),
   * se o XML tiver a linha de especificao do DTD (<code>&lt;DOCTYPE ... SYSTEM ...&gt;</code>),
   * esta deve apontar para uma URL vlida.
   * 
   * @param handler handler de eventos definido pela aplicao
   * @param validate indica se o parser deve validar o XML contra o DTD
   *        especificado
   */
  public XMLParser(final XMLHandlerInterface handler, final boolean validate) {
    final SAXParserFactory factory = SAXParserFactory.newInstance();

    /*
     * TODO fazer com que o parser ignore a declarao do DTD caso validate seja
     * false
     */
    factory.setValidating(validate);

    try {
      parser = factory.newSAXParser();
    }
    catch (ParserConfigurationException e) {
      System.err.println(e.getMessage());
    }
    catch (SAXException e) {
      System.err.println(e.getMessage());
    }
    this.handler = handler;
  }

  /**
   * Processa um XML lido atravs de um <code>Reader</code>.
   * 
   * @param reader reader a ser usado para leitura do XML
   * @return o elemento XML correspondente  raiz do documento
   * @throws IOException se houve algum erro de I/O
   */
  final public XMLElementInterface parse(final Reader reader)
    throws IOException {
    handler.setInputSource(reader);
    try {
      parser.parse(handler.getInputSource(), handler.getSAXInternalHandler());
    }
    catch (SAXException e) {
      Exception ee = e.getException();
      if (ee instanceof XMLParseException) {
        /*
         * no queremos re-encapsular XMLParseException, queremos relan-la
         * ("desempacotamos" SAXException)
         */
        XMLParseException xpe = (XMLParseException) ee;
        throw xpe;
      }
      else {
        throw new XMLParseException(e.getMessage(), e);
      }
    }
    return handler.getRootElement();
  }

  /**
   * Retorna o DTD associado ao elemento.
   * 
   * @return o DTD.
   */
  final public String getDTD() {
    return handler.getDTD();
  }
}
