/**
 * $Id: XMLValidator.java 123860 2011-11-04 23:11:41Z costa $
 */
package tecgraf.javautils.xml;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.xml.sax.InputSource;

import tecgraf.javautils.xml.exception.XMLException;

/**
 * Validador para documentos XML. Apenas processa o arquivo sem armazenar
 * informaes, para verificar se este  vlido estruturalmente. Opcionalmente,
 * pode-se validar o contedo do XML, usando o DTD.
 * 
 * @author Tecgraf
 */
public class XMLValidator {
  /**
   * Leitor XML.
   */
  private XMLReader xmlReader;
  /**
   * Handler XML.
   */
  private MyHandler handler;
  /**
   * Lista de erros no processamento do XML.
   */
  private ArrayList<String> errors;

  /**
   * Handler XML. Usa o prefixo para o DTD, e registra os erros na lista do
   * validador.
   */
  private static class MyHandler extends XMLBasicHandler {
    /**
     * Prefixo do DTD.
     */
    private final String dtdPrefix;
    /**
     * Lista de erros.
     */
    private final List<String> errors;

    /**
     * Construtor.
     * 
     * @param dtdPrefix prefixo do DTD
     * @param errors lista a ser preenchida com os erros
     */
    MyHandler(String dtdPrefix, List<String> errors) {
      super(new XMLBasicElementFactory(XMLNullDataElement.class));
      this.dtdPrefix = dtdPrefix;
      this.errors = errors;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void warning(XMLException e) throws XMLException {
      logError("AVISO", e);
    }

    /**
     * Registra um erro ({@link XMLException}).
     * 
     * @param prefix prefixo da mensagem
     * @param e exceo
     */
    private void logError(String prefix, XMLException e) {
      String tag = e.getTag();
      String msg;
      if (tag != null) {
        msg = String.format("[%s] tag %s: %s", prefix, tag, e.getMessage());
      }
      else {
        msg = String.format("[%s] %s", prefix, e.getMessage());
      }
      errors.add(msg);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void fatalError(XMLException e) throws XMLException {
      logError("ERRO FATAL", e);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void error(XMLException e) throws XMLException {
      logError("ERRO", e);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public InputSource resolveEntity(String publicId, String systemId)
      throws XMLException {
      if (dtdPrefix == null) {
        return new InputSource(systemId);
      }
      String fixedSystemId =
        systemId.replaceFirst("(file:/|http:)//.*/(.*)$", dtdPrefix + "$2");
      return new InputSource(fixedSystemId);
    }
  }

  /**
   * Construtor.
   * 
   * @param reader leitor do XML
   * @param dtdPrefix prefixo para o DTD (pode ser <code>null</code>))
   * @param validateDTD <code>true</code> se o XML deve ser validado de acordo
   *        com o DTD
   */
  public XMLValidator(Reader reader, String dtdPrefix, boolean validateDTD) {
    errors = new ArrayList<String>();
    handler = new MyHandler(dtdPrefix, errors);
    xmlReader = new XMLReader(reader, handler, validateDTD);
  }

  /**
   * Construtor.
   * 
   * @param inputStream stream para leitura do XML
   * @param dtdPrefix prefixo para o DTD (pode ser <code>null</code>)
   * @param validateDTD <code>true</code> se o XML deve ser validado de acordo
   *        com o DTD
   */
  public XMLValidator(InputStream inputStream, String dtdPrefix,
    boolean validateDTD) {
    this(new InputStreamReader(inputStream), dtdPrefix, validateDTD);
  }

  /**
   * Construtor.
   * 
   * @param file arquivo XML
   * @param dtdPrefix prefixo para o DTD (pode ser <code>null</code>)
   * @param validateDTD <code>true</code> se o XML deve ser validado de acordo
   *        com o DTD
   * @throws FileNotFoundException
   */
  public XMLValidator(File file, String dtdPrefix, boolean validateDTD)
    throws FileNotFoundException {
    this(new FileReader(file), dtdPrefix, validateDTD);
  }

  /**
   * Construtor.
   * 
   * @param path path para o arquivo XML
   * @param dtdPrefix prefixo para o DTD (pode ser <code>null</code>)
   * @param validateDTD <code>true</code> se o XML deve ser validado de acordo
   *        com o DTD
   * @throws FileNotFoundException
   */
  public XMLValidator(String path, String dtdPrefix, boolean validateDTD)
    throws FileNotFoundException {
    this(new File(path), dtdPrefix, validateDTD);
  }

  /**
   * Valida o XML.
   * 
   * @return <code>true</code> se o XML  vlido
   */
  public boolean validate() {
    try {
      xmlReader.read();
      return errors.isEmpty();
    }
    catch (Exception e) {
      return false;
    }
    finally {
      XMLUtils.close(xmlReader);
    }
  }

  /**
   * Obtm a contagem de erros de validao.
   * 
   * @return nmero de erros de validao
   */
  public int getErrorCount() {
    return errors.size();
  }

  /**
   * Obtm a lista de erros de validao.
   * 
   * @return lista de erros de validao
   */
  public List<String> getErrors() {
    return Collections.unmodifiableList(errors);
  }

  /**
   * Mtodo para testes.
   * 
   * @param args
   * @throws FileNotFoundException
   */
  public static void main(String[] args) throws FileNotFoundException {
    XMLValidator validator = new XMLValidator(args[0], args[1], true);
    boolean valid = validator.validate();
    if (valid) {
      System.out.println("XML ok");
    }
    else {
      for (String msg : validator.getErrors()) {
        System.err.println(msg);
      }
      System.err.println(validator.getErrorCount() + " erros");
    }
  }
}
