package tecgraf.javautils.xml.conversion;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import tecgraf.javautils.xml.conversion.exception.XMLConversionException;
import tecgraf.javautils.xml.conversion.exception.XMLConversionException.XMLConversionExceptionType;

/**
 * Implementao bsica de uma converso XML.
 * 
 * @see XMLConversionInterface
 * @see XMLConverter
 */
public abstract class AbstractXMLConversion implements XMLConversionInterface {

  /**
   * instncia do XPath
   */
  final private XPath xpathInstance;

  /**
   * DTD de entrada
   */
  final private String fromDTD;

  /**
   * DTD de sada
   */
  final private String toDTD;

  /**
   * documento de converso.
   */
  protected ConversionDocument document;

  /**
   * Charset de sada
   */
  final private Charset toCharset;

  /**
   * Construtor (que usa o charset padro do ambiente conforme defino de
   * {@link Charset#defaultCharset()})
   * 
   * @param fromDTD DTD de entrada
   * @param toDTD DTD de sada
   * @throws XMLConversionException em caso de erro.
   */
  protected AbstractXMLConversion(String fromDTD, String toDTD)
    throws XMLConversionException {
    this(fromDTD, toDTD, Charset.defaultCharset());
  }

  /**
   * Construtor
   * 
   * @param fromDTD DTD de entrada
   * @param toDTD DTD de sada
   * @param charset charset de sada a ser usado na gravao.
   * @throws XMLConversionException em caso de erro.
   */
  protected AbstractXMLConversion(String fromDTD, String toDTD, Charset charset)
    throws XMLConversionException {
    this.fromDTD = fromDTD;
    this.toDTD = toDTD;
    final String charsetName = charset.name();
    if (!Charset.isSupported(charsetName)) {
      throw new XMLConversionException(
        XMLConversionExceptionType.UNSUPPORTED_CHARSET);
    }
    this.toCharset = charset;
    xpathInstance = XPathFactory.newInstance().newXPath();
  }

  /**
   * Remove os ns que correspondam a uma determinada expresso, a partir de um
   * n.
   * 
   * @param startNode - n inicial a partir de onde ser feita a busca
   * @param xpathQuery - expresso XPath
   * @return lista dos ns removidos
   * @throws XMLConversionException se houver problemas com a expresso
   */
  protected List<Node> remove(Node startNode, String xpathQuery)
    throws XMLConversionException {
    NodeList nodeList;
    try {
      nodeList =
        (NodeList) xpathInstance.evaluate(xpathQuery, startNode,
          XPathConstants.NODESET);
    }
    catch (XPathExpressionException e) {
      throw new XMLConversionException(XMLConversionExceptionType.QUERY, e);
    }
    final int numNodes = nodeList.getLength();
    List<Node> removedNodes = new ArrayList<Node>();
    for (int i = 0; i < numNodes; i++) {
      Node nodeToBeRemoved = nodeList.item(i);
      Node parentNode = nodeToBeRemoved.getParentNode();
      removedNodes.add(parentNode.removeChild(nodeToBeRemoved));
    }
    return removedNodes;
  }

  /**
   * Remove todos os ns que correspodam a uma expresso, em todo o documento.
   * 
   * @param xpathQuery - expresso XPath para o n de referncia (deve comear
   *        da raiz, i.e. com "/")
   * @return lista dos ns removidos
   * @throws XMLConversionException se houver problemas com a expresso
   */
  protected List<Node> remove(String xpathQuery) throws XMLConversionException {
    // FIXME lanar exceo se a expresso no comear com "/"
    return remove(document.getDocumentElement(), xpathQuery);
  }

  /**
   * Renomeia todos os ns que correspondam a uma expresso.
   * 
   * @param xpathQuery - expresso XPath para encontrar o n
   * @param newName - novo nome para os ns
   * @return lista dos ns renomeados
   * @throws XMLConversionException se houver problemas com a expresso
   */
  protected List<Node> renameAll(String xpathQuery, String newName)
    throws XMLConversionException {
    NodeList nodeList;
    try {
      nodeList =
        (NodeList) xpathInstance.evaluate(xpathQuery, document
          .getDocumentElement(), XPathConstants.NODESET);
    }
    catch (XPathExpressionException e) {
      throw new XMLConversionException(XMLConversionExceptionType.QUERY, e);
    }
    final int numNodes = nodeList.getLength();
    List<Node> renamedNodes = new ArrayList<Node>();
    for (int i = 0; i < numNodes; i++) {
      Node nodeToBeRenamed = nodeList.item(i);
      renamedNodes.add(document.renameNode(nodeToBeRenamed, null, newName));
    }
    return renamedNodes;
  }

  /**
   * Renomeia um n especificado por uma expresso.
   * 
   * @param xpathQuery expresso XPath para encontrar o n
   * @param newName novo nome
   * @return n renomeado. Pode ser o prprio n passado como parmetro, ou um
   *         novo n criado para substitui-lo
   * @throws XMLConversionException se houver problemas com a expresso
   * 
   * @see #rename(Node, String)
   * @see Document#renameNode(Node, String, String)
   */
  protected Node rename(String xpathQuery, String newName)
    throws XMLConversionException {
    final Node node = findFirstNode(xpathQuery);
    return document.renameNode(node, null, newName);
  }

  /**
   * Renomeia um n.
   * 
   * @param node n
   * @param newName novo nome
   * @return n renomeado. Pode ser o prprio n passado como parmetro, ou um
   *         novo n criado para substitui-lo
   */
  protected Node rename(Node node, String newName) {
    return document.renameNode(node, null, newName);
  }

  /**
   * Encontra o primeiro n que corresponde a uma expresso, a partir de um n
   * especfico.
   * 
   * @param startNode - n inicial para a busca
   * @param xpathQuery - expresso XPath para encontrar o n
   * @return o n encontrado, ou null caso no exista FIXME confirmar o retorno
   *         em caso de falha
   * @throws XMLConversionException se houver problemas com a expresso
   */
  protected Node findFirstNode(Node startNode, String xpathQuery)
    throws XMLConversionException {
    // TODO testar se XPath.evaluate pode retornar null
    try {
      return (Node) xpathInstance.evaluate(xpathQuery, startNode,
        XPathConstants.NODE);
    }
    catch (XPathExpressionException e) {
      throw new XMLConversionException(XMLConversionExceptionType.QUERY, e);
    }
  }

  /**
   * Encontra o primeiro n de todo o documento que corresponde a uma expresso.
   * 
   * @param xpathQuery expresso XPath para encontrar o n. Deve partir da raiz
   *        (i.e. comear com '/')
   * @return o n encontrado, ou <code>null</code> caso no exista
   * @throws XMLConversionException se houver problemas com a expresso
   */
  protected Node findFirstNode(String xpathQuery) throws XMLConversionException {
    // FIXME confirmar o retorno em caso de falha
    if (xpathQuery.charAt(0) != '/') {
      throw new XMLConversionException(XMLConversionExceptionType.QUERY,
        "path deve comear com '/'");
    }
    return findFirstNode(document.getDocumentElement(), xpathQuery);
  }

  /**
   * Encontra todos os ns que correspondam a uma expresso, a partir de um n
   * especfico.
   * 
   * @param startNode - no inicial para a busca
   * @param xpathQuery - expresso XPath para encontrar o n
   * @return lista com todos os ns encontrados
   * @throws XMLConversionException se houver problemas com a expresso
   */
  protected NodeList findAllNodes(Node startNode, String xpathQuery)
    throws XMLConversionException {
    try {
      return (NodeList) xpathInstance.evaluate(xpathQuery, startNode,
        XPathConstants.NODESET);
    }
    catch (XPathExpressionException e) {
      throw new XMLConversionException(XMLConversionExceptionType.QUERY, e);
    }
  }

  /**
   * Encontra todos os ns que correspondam a uma expresso.
   * <p>
   * Buscas que no envolvam atributos e nem contextos devem ser feitas via
   * mtodo {@link #findAllNodesByTag(String)}, que tende a ser mais eficiente
   * por no precisar usar expresses XPath.
   * 
   * @param xpathQuery - expresso XPath para encontrar o n
   * @return lista com os ns encontrados
   * @throws XMLConversionException se houver problemas com a expresso
   * @see #findAllNodesByTag(String)
   */
  protected NodeList findAllNodes(String xpathQuery)
    throws XMLConversionException {
    return findAllNodes(document.getDocumentElement(), xpathQuery);
  }

  /**
   * Encontra todos os ns com uma determinada tag.
   * 
   * @param tag - tag. O valor "*" retornar todos os ns
   * @return lista com os ns encontrados, na ordem em que foram definidos, ou
   *         uma lista vazia caso nenhum n tenha sido encontrado.
   */
  protected NodeList findAllNodesByTag(String tag) {
    return document.getElementsByTagName(tag);
  }

  /**
   * Cria um novo n vazio com o nome especificado.
   * 
   * @param name - nome do novo elemento (tag)
   * @return novo n
   */
  protected Node createNode(String name) {
    return document.createElement(name);
  }

  /**
   * Cria um novo n com o nome e o valor especificados.
   * 
   * @param name - nome do n
   * @param value - valor do n
   * @return novo n
   */
  protected Node createNode(String name, String value) {
    Node newNode = createNode(name);
    newNode.setTextContent(value);
    return newNode;
  }

  /**
   * Cria um novo n com o nome e o valor especificados.
   * 
   * @param name - nome do n
   * @param value - valor do n (double)
   * @return novo n
   */
  protected Node createNode(String name, double value) {
    return createNode(name, String.valueOf(value));
  }

  /**
   * Define o valor (texto) de um n, referenciado por uma expresso. Apenas
   * ns-folha (<i>data nodes</i>) so afetados por este mtodo.
   * 
   * @param xpathQuery - expresso XPath para encontrar o n. Caso a expresso
   *        aponte mltiplos ns, apenas o primeiro ser afetado
   * @param value - valor do n
   * @return true se o valor foi definido, false se o elemento no possua valor
   *         (p.ex. se no era um <i>data node</i>)
   * @throws XMLConversionException se houver problemas com a expresso
   */
  protected boolean setNodeValue(String xpathQuery, String value)
    throws XMLConversionException {
    return setNodeValue(findFirstNode(xpathQuery), value);
  }

  /**
   * Define o valor (texto) de um n. Apenas ns-folha (<i>data nodes</i>) so
   * afetados por este mtodo.
   * 
   * @param node - n
   * @param value - valor
   * @return true se o valor foi definido, false se o elemento no possua valor
   *         (p.ex. se no era um <i>data node</i>)
   */
  protected boolean setNodeValue(Node node, String value) {
    Node textChild = getFirstChildByType(node, Node.TEXT_NODE);
    if (textChild == null) {
      // o n no possua nenhum filho do tipo Text
      return false;
    }
    textChild.setNodeValue(value);
    return true;
  }

  /**
   * Define o valor (texto) de um n. Apenas ns-folha (<i>data nodes</i>) so
   * afetados por este mtodo.
   * 
   * @param node - n
   * @param value - valor (double)
   * @return true se o valor foi definido, false se o elemento no possua valor
   *         (p.ex. se no era um <i>data node</i>)
   */
  protected boolean setNodeValue(Node node, double value) {
    return setNodeValue(node, String.valueOf(value));
  }

  /**
   * Copia o valor de um n para outro.
   * 
   * @param from n de origem
   * @param to n de destino
   */
  protected void copyNodeValue(Node from, Node to) {
    to.setNodeValue(from.getNodeValue());
  }

  /**
   * Retorna o primeiro filho de um n do tipo especificado, ou null caso o n
   * no possua nenhum filho direto do tipo correto. Para garantir que exista
   * apenas um n-filho <code>TextNode</code>, normalizamos o n de referncia
   * antes de percorr-lo.
   * 
   * @param node - n de referncia
   * @param type - tipo do n (ver {@link Node#getNodeType()})
   * @return primeiro filho do n cujo tipo  o especificado, ou null caso no
   *         exista um filho <b>direto</b> com o tipo correto
   */
  private Node getFirstChildByType(Node node, short type) {
    // colapsamos os filhos TextNode em um s
    node.normalize();
    NodeList children = node.getChildNodes();
    final int numChildren = children.getLength();
    for (int i = 0; i < numChildren; i++) {
      Node child = children.item(i);
      if (child.getNodeType() == type) {
        return child;
      }
    }
    return null;
  }

  /**
   * Retorna o mapa com os atributos de um determinado n. Se o n for na
   * verdade uma subrvore, retorna os atributos do seu primeiro filho que  um
   * n.
   * 
   * @param node - n
   * @return mapa com os atributos do n, ou null se o n no for de um dos
   *         tipos suportados (ELEMENT ou DOCUMENT_FRAGMENT)
   */
  private NamedNodeMap getAttributeMap(Node node) {
    Node targetNode = null;
    switch (node.getNodeType()) {
      case Node.ELEMENT_NODE:
        targetNode = node;
        break;

      case Node.DOCUMENT_FRAGMENT_NODE:
        targetNode = getFirstChildByType(node, Node.ELEMENT_NODE);
        break;

      default:
        // vazio
        break;
    }
    if (targetNode == null) {
      return null;
    }

    return targetNode.getAttributes();
  }

  /**
   * Retorna o valor de um determinado atributo de um n.
   * 
   * @see #getAttributeMap(Node)
   * 
   * @param node - n
   * @param attrName - nome do atributo
   * @return valor do atributo, ou null se o n no possui atributos ou se o n
   *         no  um ELEMENT ou DOCUMENT_FRAGMENT ou se o n no possui o
   *         atributo em questo.
   */
  protected String getAttribute(Node node, String attrName) {
    NamedNodeMap attrMap = getAttributeMap(node);
    if (attrMap == null) {
      return null;
    }
    Attr attr = (Attr) attrMap.getNamedItem(attrName);
    if (attr == null) {
      return null;
    }
    return attr.getNodeValue();
  }

  /**
   * Define um novo atributo para um n, ou substitui o valor de um atributo
   * existente.
   * 
   * @param node - n onde ser definido o atributo
   * @param attrName - nome do atributo a ser criado/alterado
   * @param attrValue - valor do atributo
   * @return true se o atributo foi criado/alterado, false se o n no possui
   *         atributos (i.e. se no for do tipo {@link Element})
   */
  protected boolean setAttribute(Node node, String attrName, String attrValue) {
    NamedNodeMap attrs = getAttributeMap(node);
    if (attrs == null) {
      return false;
    }
    /*
     * no precisamos verificar se o atributo j existe, simplesmente criamos um
     * novo (em caso de coliso, o antigo  substitudo)
     */
    Attr newAttr = document.createAttribute(attrName);
    newAttr.setNodeValue(attrValue);
    attrs.setNamedItem(newAttr);
    return true;
  }

  /**
   * Define um novo atributo booleano para um n, ou substitui o valor de um
   * atributo existente.
   * 
   * @param node - n onde ser definido o atributo
   * @param attrName - nome do atributo a ser criado/alterado
   * @param attrValue - valor do atributo (boolean)
   * @return true se o atributo foi criado/alterado, false se o n no possui
   *         atributos (i.e. se no for do tipo {@link Element})
   */
  protected boolean setAttribute(Node node, String attrName, boolean attrValue) {
    return setAttribute(node, attrName, attrValue ? "TRUE" : "FALSE");
  }

  /**
   * Define um novo atributo inteiro para um n, ou substitui o valor de um
   * atributo existente.
   * 
   * @param node - n onde ser definido o atributo
   * @param attrName - nome do atributo a ser criado/alterado
   * @param attrValue - valor do atributo (int)
   * @return true se o atributo foi criado/alterado, false se o n no possui
   *         atributos (i.e. se no for do tipo {@link Element})
   */
  protected boolean setAttribute(Node node, String attrName, int attrValue) {
    return setAttribute(node, attrName, String.valueOf(attrValue));
  }

  /**
   * Define um novo atributo double para um n, ou substitui o valor de um
   * atributo existente.
   * 
   * @param node - n onde ser definido o atributo
   * @param attrName - nome do atributo a ser criado/alterado
   * @param attrValue - valor do atributo (double)
   * @return true se o atributo foi criado/alterado, false se o n no possui
   *         atributos (i.e. se no for do tipo {@link Element})
   */
  protected boolean setAttribute(Node node, String attrName, double attrValue) {
    return setAttribute(node, attrName, String.valueOf(attrValue));
  }

  /**
   * Remove um atributo de um n. <b>Importante</b>: De acordo com a
   * documentao de NamedNodeMap.removeNamedItem, usado por este mtodo, caso o
   * atributo tenha sido definido com um valor default no dtd/schema, aps a
   * remoo o atributo ser reinserido, dando a impresso de que o mtodo no
   * executou com sucesso.
   * 
   * @see <a
   *      href="http://www.w3.org/2003/01/dom2-javadoc/org/w3c/dom/NamedNodeMap.html#removeNamedItem_java.lang.String_">Documentao
   *      de NamedNodeMap.removeNamedItem</a>
   * @param node - n de onde ser removido o atributo
   * @param attrName - nome do atributo a ser removido
   * @return atributo removido ({@link Attr}) ou null caso o atributo no exista
   */
  protected Node removeAttribute(Node node, String attrName) {
    NamedNodeMap attrs = node.getAttributes();
    if (attrs == null) {
      return null;
    }
    return attrs.removeNamedItem(attrName);
  }

  /**
   * Insere um n antes de outro, especificado atravs de uma expresso. Caso a
   * expresso aponte para mais de um n o primeiro ser usado como referncia.
   * 
   * @param xpathQuery - expresso XPath para encontrar o n
   * @param newNode - n a ser inserido
   * @return true se a insero foi bem-sucedida, false se o n de referncia
   *         no foi encontrado
   * @throws XMLConversionException se houver problemas com a expresso
   */
  protected boolean insertBefore(String xpathQuery, Node newNode)
    throws XMLConversionException {
    return insertBefore(findFirstNode(xpathQuery), newNode);
  }

  /**
   * Insere um n antes de outro.
   * 
   * @param refNode - n de referncia
   * @param newNode - n a ser inserido
   * @return true se a insero foi bem-sucedida, false se o n de referncia
   *         no  um n vlido
   */
  protected boolean insertBefore(Node refNode, Node newNode) {
    if (refNode == null) {
      return false;
    }
    Node parentNode = refNode.getParentNode();
    if (parentNode == null) {
      return false;
    }
    parentNode.insertBefore(newNode, refNode);
    return true;
  }

  /**
   * Insere um n antes de outro, encontrado a partir de um n de referncia +
   * uma expresso XPath.
   * 
   * @param refNode - n de referncia a partir do qual ser feita a busca
   * @param xpathQuery - expresso XPath para encontrar o n que servir de
   *        referncia para a insero
   * @param newNode - n a ser inserido
   * @return true se a insero foi bem-sucedida, false se o n de referncia
   *         no foi encontrado
   * @throws XMLConversionException se houver problemas com a expresso
   */
  protected boolean insertBefore(Node refNode, String xpathQuery, Node newNode)
    throws XMLConversionException {
    return insertBefore(findFirstNode(refNode, xpathQuery), newNode);
  }

  /**
   * Insere um n aps outro, especificado atravs de uma expresso. Caso a
   * expresso aponte para mais de um n o primeiro ser usado como referncia.
   * 
   * @param xpathQuery - quey XPath
   * @param newNode - n a ser inserido
   * @return true se a insero foi bem-sucedida, false se o n de referncia
   *         no foi encontrado
   * @throws XMLConversionException se houver problemas com a expresso
   */
  protected boolean insertAfter(String xpathQuery, Node newNode)
    throws XMLConversionException {
    return insertAfter(findFirstNode(xpathQuery), newNode);
  }

  /**
   * Insere um n aps um outro.
   * 
   * @param refNode - n de referncia
   * @param newNode - n a ser inserido
   * @return true se a insero foi bem-sucedida, false se o n de referncia
   *         no foi encontrado
   */
  protected boolean insertAfter(Node refNode, Node newNode) {
    if (refNode == null) {
      return false;
    }
    final Node nextSibling = refNode.getNextSibling();
    if (nextSibling != null) {
      return insertBefore(nextSibling, newNode);
    }

    /*
     * n de referncia  o ltimo filho do seu pai, no podemos usar
     * insertBefore(); em vez disso, usamos o appendChild() do pai
     */
    final Node parentNode = refNode.getParentNode();
    if (parentNode == null) {
      return false;
    }
    try {
      parentNode.appendChild(newNode);
    }
    catch (Exception e) {
      return false;
    }
    return true;

  }

  /**
   * Insere um n aps um outro, encontrado a partir de um n de referncia +
   * uma expresso XPath.
   * 
   * @param refNode - n de referncia a partir do qual ser feita a busca
   * @param xpathQuery - expresso XPath para encontrar o n que servir de
   *        referncia para a insero
   * @param newNode - n a ser inserido
   * @return true se a insero foi bem-sucedida, false se o n de referncia
   *         no foi encontrado
   * @throws XMLConversionException se houver problemas com a expresso
   */
  protected boolean insertAfter(Node refNode, String xpathQuery, Node newNode)
    throws XMLConversionException {
    return insertAfter(findFirstNode(refNode, xpathQuery), newNode);
  }

  /**
   * Adiciona um n ao final da lista de filhos de um n especificado por uma
   * expresso.
   * 
   * @param xpathQuery - expresso XPath para encontrar o n
   * @param newNode - n a ser inserido
   * @return true se a insero foi bem-sucedida, false se o n de referncia
   *         no foi encontrado
   * @throws XMLConversionException
   */
  protected boolean appendChild(String xpathQuery, Node newNode)
    throws XMLConversionException {
    Node refNode = findFirstNode(xpathQuery);
    if (refNode == null) {
      return false;
    }
    refNode.appendChild(newNode);
    return true;
  }

  /**
   * Insere um novo n em uma posio especfica na lista de filhos de um n
   * especificado por uma expresso.
   * 
   * @param xpathQuery - expresso XPath para encontrar o n para o n de
   *        referncia
   * @param newNode - n a ser inserido
   * @param pos - posio (comea em 0)
   * @return -1 se o n de referncia no foi encontrado, <b>pos</b> caso a
   *         insero tenha sido bem-sucedida
   * @throws XMLConversionException
   */
  protected int insertChild(String xpathQuery, Node newNode, int pos)
    throws XMLConversionException {
    Node parentNode = findFirstNode(xpathQuery);
    if (parentNode == null) {
      return -1;
    }
    NodeList childNodes = parentNode.getChildNodes();
    int nodeCount = childNodes.getLength();
    if (pos > nodeCount) {
      parentNode.appendChild(newNode);
      return nodeCount;
    }
    Node refNode = childNodes.item(pos);
    insertBefore(refNode, newNode);
    return pos;
  }

  /**
   * Escreve o documento como resultado de uma transformao.
   * 
   * @param result - resultado final da transformao
   * @throws XMLConversionException se no houver dados de entrada ou se houver
   *         algum problema durante a gravao
   * 
   * @see Result
   */
  private void writeXML(Result result) throws XMLConversionException {
    if (document == null) {
      throw new XMLConversionException(XMLConversionExceptionType.NO_INPUT);
    }
    /*
     * "normalizar" significa, entre outras coisas, colapsar eventuais ns
     * 'Text' adjacentes em um nico n
     */
    document.normalizeDocument();
    try {
      Transformer transformer =
        TransformerFactory.newInstance().newTransformer();

      // necessrio para gravao da linha <!DOCTYPE ... > com o DTD
      transformer.setOutputProperty(OutputKeys.DOCTYPE_SYSTEM, toDTD);

      // definimos o mtodo usado para gravar o documento
      transformer.setOutputProperty(OutputKeys.METHOD, "xml");

      // as 2 linhas abaixo habilitam pretty-printing
      transformer.setOutputProperty(OutputKeys.INDENT, "yes");
      transformer.setOutputProperty(
        "{ http://xml.apache.org/xslt }indent-amount", "2");

      // definimos o encoding
      final String charsetName = toCharset.name();
      transformer.setOutputProperty(OutputKeys.ENCODING, charsetName);

      Source source = new DOMSource(document);
      transformer.transform(source, result);
    }
    catch (Exception e) {
      throw new XMLConversionException(XMLConversionExceptionType.WRITE, e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeXML(Writer writer) throws XMLConversionException {
    writeXML(new StreamResult(writer));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeXML(String fileName) throws XMLConversionException {
    try {
      writeXML(new FileOutputStream(fileName));
    }
    catch (FileNotFoundException e) {
      throw new XMLConversionException(
        XMLConversionExceptionType.FILE_NOT_FOUND, e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void writeXML(File file) throws XMLConversionException {
    try {
      writeXML(new FileOutputStream(file));
    }
    catch (FileNotFoundException e) {
      throw new XMLConversionException(
        XMLConversionExceptionType.FILE_NOT_FOUND, e);
    }
  }

  /**
   * Escreve o documento em um {@link OutputStream}. O stream  encapsulado por
   * um {@link OutputStreamWriter} para que possamos definir o charset.
   * 
   * @param os - OutputStream
   * @throws XMLConversionException se o arquivo  invlido, se a entrada no
   *         foi definida ou se houve algum problema na gravao
   */
  @Override
  public void writeXML(OutputStream os) throws XMLConversionException {
    final OutputStreamWriter writer = new OutputStreamWriter(os, toCharset);
    final BufferedWriter buffWriter = new BufferedWriter(writer);
    writeXML(new StreamResult(buffWriter));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final boolean canConvertFrom(String dtd) {
    final boolean flag = compareDTDs(dtd, fromDTD);
    return flag;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final boolean convertsTo(String dtd) {
    return compareDTDs(dtd, toDTD);
  }

  /**
   * Compara dois DTDs.
   * 
   * @param dtd1 - primeiro DTD
   * @param dtd2 - segundo DTD
   * @return true se ambos os DTDs so <code>null</code>, ou se ambos so !=
   *         <code>null</code> e as strings so iguais
   *         (<i>case-insensitive</i>). Em todos os outros casos retorna false.
   */
  private boolean compareDTDs(String dtd1, String dtd2) {
    if (dtd1 == null) {
      return dtd2 == null;
    }
    else if (dtd2 == null) {
      // dtd1 != null
      return false;
    }
    else {
      return dtd1.equalsIgnoreCase(dtd2);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final boolean canConvertFrom(XMLConversionInterface conversion) {
    return canConvertFrom(conversion.getTargetDTD());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getTargetDTD() {
    return toDTD;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConversionDocument convert(XMLConversionInterface previousConversion)
    throws XMLConversionException {
    if (canConvertFrom(previousConversion.getTargetDTD()) == false) {
      throw new XMLConversionException(XMLConversionExceptionType.CANT_CONVERT);
    }
    return convert(previousConversion.getDocument());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConversionDocument convert(ConversionDocument doc)
    throws XMLConversionException {
    if (canConvertFrom(doc.getDTD()) == false) {
      throw new XMLConversionException(XMLConversionExceptionType.CANT_CONVERT);
    }
    setDocument(doc);
    ConversionDocument convertedDoc = convert();
    /*
     * converso foi bem-sucedida, atualizamos o DTD do documento para refletir
     * o novo estado
     */
    convertedDoc.setDTD(getTargetDTD());
    return convertedDoc;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConversionDocument convert(File file, boolean validate)
    throws XMLConversionException {
    DocumentBuilderFactory docBuilderFactory =
      DocumentBuilderFactory.newInstance();
    docBuilderFactory.setValidating(validate);
    try {
      DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
      convert(new ConversionDocument(docBuilder.parse(file)));
    }
    catch (Exception e) {
      throw new XMLConversionException(XMLConversionExceptionType.PARSER, e);
    }
    return document;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConversionDocument getDocument() {
    return document;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setDocument(ConversionDocument doc) {
    this.document = doc;
  }

  /**
   * Retorna o valor de um n. O n tem que ser um <i>data node</i>, i.e. um
   * n-folha. A busca pelo n-alvo  feita a partir de um n de referncia.
   * 
   * @param startNode - n de referncia
   * @param xpathQuery - expresso XPath para encontrar o n
   * 
   * @return valor do n como String, ou null caso o n no tenha sido
   *         encontrado
   * @throws XMLConversionException se a houver erro na query, ou se o n no
   *         for um data node
   */
  protected String getNodeValue(Node startNode, String xpathQuery)
    throws XMLConversionException {
    return getNodeValue(findFirstNode(startNode, xpathQuery));
  }

  /**
   * Retorna o valor de um n. O n tem que ser um <i>data node</i>, i.e. um
   * n-folha. A busca pelo n-alvo  feita a partir da raiz do documento.
   * 
   * @param xpathQuery - expresso XPath para encontrar o n. Deve iniciar pela
   *        raiz, i.e. com "/"
   * @return valor do n como String, ou null caso o n no tenha sido
   *         encontrado
   * @throws XMLConversionException se a houver erro na query, ou se o n no
   *         for um data node
   */
  protected String getNodeValue(String xpathQuery)
    throws XMLConversionException {
    return getNodeValue(findFirstNode(xpathQuery));
  }

  /**
   * Retorna o valor de um n. O n tem que possuir um filho direto do tipo
   * <code>TextNode</code>.
   * 
   * @param node - n
   * @return valor do n, ou null se o n for null ou se no possuir um filho
   *         direto do tipo <code>TextNode</code>
   */
  protected String getNodeValue(Node node) {
    if (node == null) {
      return null;
    }
    if (node.getNodeType() == Node.TEXT_NODE) {
      return node.getNodeValue();
    }
    Node textNode = getFirstChildByType(node, Node.TEXT_NODE);
    return textNode == null ? null : textNode.getNodeValue();
  }

  /**
   * Retorna o valor de um n convertido para double.
   * 
   * @param xpathQuery - expresso XPath para encontrar o n
   * @return valor do n convertido para double
   * @throws XMLConversionException se o n no possua valor ou se o valor no
   *         era um nmero
   */
  protected double getDoubleNodeValue(String xpathQuery)
    throws XMLConversionException {
    return getDoubleNodeValue(findFirstNode(xpathQuery));
  }

  /**
   * Retorna o valor de um n convertido para double.
   * 
   * @param startNode - n inicial para busca
   * @param xpathQuery - expresso XPath para encontrar o n
   * @return valor do n convertido para double
   * @throws XMLConversionException se o n no possua valor ou se o valor no
   *         era um nmero
   */
  protected double getDoubleNodeValue(Node startNode, String xpathQuery)
    throws XMLConversionException {
    return getDoubleNodeValue(findFirstNode(startNode, xpathQuery));
  }

  /**
   * Retorna o valor de um n como um double.
   * 
   * @param node - n
   * @return valor do n (double)
   * @throws XMLConversionException se o n no possua valor ou se o valor no
   *         era um nmero
   */
  protected double getDoubleNodeValue(Node node) throws XMLConversionException {
    String strVal = getNodeValue(node);
    if (strVal == null) {
      throw new XMLConversionException(XMLConversionExceptionType.NO_VALUE);
    }
    try {
      return Double.valueOf(strVal);
    }
    catch (NumberFormatException e) {
      throw new XMLConversionException(XMLConversionExceptionType.NOT_NUMBER, e);
    }
  }
}
