/*
 * $Id: XMLConversionException.java 122771 2011-10-04 21:35:02Z costa $
 */
package tecgraf.javautils.xml.conversion.exception;

import tecgraf.javautils.xml.conversion.AbstractXMLConversion;
import tecgraf.javautils.xml.conversion.XMLConverter;

/**
 * A classe <code>XMLConversionException</code> implementa as possveis
 * situaes de erro encontradas durante o processo de converso de documentos
 * XML usando as classes {@link XMLConverter} e subclasses de
 * {@link AbstractXMLConversion}.
 * <p>
 * As diferentes situaes de erro so indicadas pelo tipo associado  exceo.
 * 
 * TODO pode ser interessante recebermos mais dados em alguns casos
 */
public class XMLConversionException extends Exception {

  /**
   * Cdigos de erro para a exceo. Indicam qual foi o erro, evitando a
   * necessidade de criar mltiplas excees para diferentes situaes
   * correlatas.
   */
  public enum XMLConversionExceptionType {
    /**
     * Erro no processamento do XML.
     */
    PARSER,
    /**
     * Query XPath mal-formada.
     */
    QUERY,
    /**
     * Erro na gravao do XML.
     */
    WRITE,
    /**
     * Documento no pode ser convertido. Isto tipicamente acontece quando a
     * converso em questo no aceita o DTD do documento em questo como
     * entrada.
     */
    CANT_CONVERT,
    /**
     * No foi definido uma entrada para a converso.
     */
    NO_INPUT,
    /**
     * Tipo do n no  vlido para a operao solicitada (p.ex. recuperao do
     * valor de um n que no  um TEXT_NODE).
     */
    NODE_TYPE,
    /**
     * Cadeia de converses foi quebrada. O conversor percorre sua lista de
     * converses procurando por uma capaz de converter o documento que tem em
     * mos; uma vez que ache a primeira capaz de faz-lo, todas as converses
     * subsequentes at o final da lista tm que ser capazes de ser encadeadas
     * (processar o documento da converso anterior).
     */
    BROKEN_CHAIN,
    /**
     * N no possui valor.
     */
    NO_VALUE,
    /**
     * Valor do n no  nmero e tentou ser usado como tal.
     */
    NOT_NUMBER,
    /**
     * Arquivo invlido (no encontrado).
     */
    FILE_NOT_FOUND,
    /**
     * No foi definida nenhuma converso.
     */
    NO_CONVERSIONS,
    /**
     * DTD associado ao arquivo no corresponde ao DTD corrente e no 
     * compatvel com nenhuma converso. Isto vale tambm para documentos que
     * no possuem DTD (i.e. sem declarao DOCTYPE).
     */
    INVALID_DTD,
    /**
     * As converses no foram aplicadas no arquivo em questo.
     */
    CONVERSIONS_NOT_APPLIED,
    /**
     * No foi possvel validar o resultado da converso.
     */
    CANT_VALIDATE,
    /**
     * No foi possvel fazer backup do arquivo
     */
    CANT_BACKUP,

    /**
     * Charset utilizado no  suportado pelo ambiente.
     */
    UNSUPPORTED_CHARSET,
  }

  /**
   * Tipo
   */
  private XMLConversionExceptionType type;

  /**
   * Cria uma exceo.
   * 
   * @param type - tipo da exceo
   */
  public XMLConversionException(XMLConversionExceptionType type) {
    this.type = type;
  }

  /**
   * Cria uma exceo com uma mensagem especfica.
   * 
   * @param type tipo da exceo
   * @param details mensagem
   */
  public XMLConversionException(XMLConversionExceptionType type, String details) {
    super(details);
    this.type = type;
  }

  /**
   * Cria uma exceo encapsulando outra.
   * 
   * @param type tipo da exceo
   * @param original exceo original
   */
  public XMLConversionException(XMLConversionExceptionType type,
    Throwable original) {
    super(original);
    this.type = type;
  }

  /**
   * Cria uma exceo encapsulando outra, com uma mensagem especfica.
   * 
   * @param type tipo da exceo
   * @param details mensagem
   * @param original exceo original
   */
  public XMLConversionException(XMLConversionExceptionType type,
    String details, Throwable original) {
    super(details, original);
    this.type = type;
  }

  /**
   * @return o tipo da exceo
   */
  public XMLConversionExceptionType getType() {
    return type;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    // TODO precisamos traduzir esses erros?
    StringBuilder msg = new StringBuilder("Erro na converso do XML: ");
    switch (type) {
      case BROKEN_CHAIN:
        msg
          .append("cadeia de converses foi quebrada, um dos conversores falhou");
        break;

      case CANT_CONVERT:
        msg.append("documento incompatvel com esta converso");
        break;

      case FILE_NOT_FOUND:
        msg.append("arquivo no encontrado");
        break;

      case NO_INPUT:
        msg.append("no h dados de entrada");
        break;

      case NO_VALUE:
        msg.append("elemento no possui valor");
        break;

      case NODE_TYPE:
        msg.append("operao incompatvel com o tipo do n");
        break;

      case NOT_NUMBER:
        msg.append("valor do elemento no  um double");
        break;

      case PARSER:
        msg.append("erro no processamento do documento XML");
        break;

      case QUERY:
        msg.append("query XPath mal-formada ou erro no processamento da mesma");
        break;

      case WRITE:
        msg.append("erro na gravao do XML");
        break;

      case CONVERSIONS_NOT_APPLIED:
        msg.append("converses no foram aplicadas");
        break;

      case INVALID_DTD:
        msg.append("DTD no pode ser tratado por nenhuma converso");
        break;

      case NO_CONVERSIONS:
        msg.append("no foi definida nenhuma converso");
        break;

      case CANT_VALIDATE:
        msg.append("resultado da converso no  vlido de acordo com o DTD");
        break;

      case CANT_BACKUP:
        msg.append("erro fazendo backup do arquivo original");
        break;

      case UNSUPPORTED_CHARSET:
        msg.append("charset no suportado");
        break;

      default:
        final String err = "Tipo de exceo no tratada! " + type.name();
        throw new AssertionError(err);
    }
    String originalMsg = super.toString();
    if ("".equals(originalMsg) == false) {
      msg.append("\nDados adicionais\n\n");
      msg.append(super.toString());
    }
    return msg.toString();
  }

}
