/*
 * $Id: XMLConverter.java 127093 2012-03-08 18:46:10Z costa $
 */
package tecgraf.javautils.xml.conversion;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.xml.sax.InputSource;

import tecgraf.javautils.xml.conversion.exception.XMLConversionException;
import tecgraf.javautils.xml.conversion.exception.XMLConversionException.XMLConversionExceptionType;

/**
 * Implementao do conversor de arquivos XML. O conversor possui uma lista de
 * {@link XMLConversionInterface converses}, que sero aplicadas na ordem em
 * que foram definidas. O objetivo  converter arquivos validados por verses
 * anteriores do DTD para arquivos vlidos de acordo com o DTD mais atual.
 * <p>
 * A lista de converses  percorrida em ordem, at que seja encontrada a
 * primeira converso capaz de lidar com o DTD associado ao XML em questo. A
 * partir da, <b>todas</b> as converses subsequentes tm que ser aplicadas,
 * sem exceo (para que se garanta que o resultado final esteja compatvel com
 * a ltima verso do DTD).
 * 
 * @see XMLConversionInterface
 */
public class XMLConverter {

  /**
   * Enumerao com os possveis estados de uma converso.
   */
  public enum XMLConversionStatus {
    /**
     * Nenhuma converso foi aplicada.
     */
    NOT_APPLIED,
    /**
     * Converses aplicadas com sucesso.
     */
    SUCCESS,
    /**
     * Converses foram tentadas mas houve alguma falha no processo.
     */
    ERROR,
    /**
     * Arquivo j correspondia  ltima verso do DTD.
     */
    NOT_NEEDED,
  }

  /**
   * Lista de converses. As converses sero aplicadas em ordem, e no pode
   * haver "gaps" no processo: uma vez que o processo seja iniciado (i.e. o
   * documento a ser convertido seja "aceito" por uma das converses) todas as
   * converses subsequentes tm que fazer o mesmo, at o final da lista.
   */
  private List<XMLConversionInterface> conversions;
  /**
   * Documento a ser convertido.
   */
  private ConversionDocument document;
  /**
   * ltima converso aplicada.
   */
  private XMLConversionInterface lastConversion;
  /**
   * Status da converso.
   */
  private XMLConversionStatus conversionStatus;
  /**
   * Gerador do documento.
   */
  private ConversionDocumentBuilder docBuilder;
  /**
   * Prefixo do DTD.
   */
  private String dtdPrefix;
  /**
   * Indica se o prefixo para o DTD refere-se ao JAR.
   */
  private boolean fromJar;
  /**
   * Indica se o docmuento deve ser validado pelo seu DTD.
   */
  private boolean validate;
  /**
   * Fonte para leitura do documento.
   */
  private InputSource inputSource;
  /**
   * Charset usado na leitura do documento. Se no for especificado, ser usado
   * o charset default do sistema.
   */
  private Charset charset;

  /**
   * Construtor vazio. Usa o encoding default do sistema, e no define prefixo
   * para os DTDs.
   */
  public XMLConverter() {
    this((String) null);
  }

  /**
   * Instancia um conversor que aponta para DTDs externos (fora do JAR) e usa o
   * encoding default do sistema.
   * 
   * @param dtdPrefix prefixo para o DTD. Pode ser uma URL, um path para o
   *        filesystem local ou <code>null</code>
   */
  public XMLConverter(String dtdPrefix) {
    this(dtdPrefix, false);
  }

  /**
   * Instancia um conversor usando o encoding default do sistema.
   * 
   * @param dtdPrefix prefixo para o DTD. Pode ser uma URL, um path para o
   *        filesystem local ou <code>null</code>. Se
   *        <code>fromJar == true</code>, deve ser relativo ao JAR corrente
   * @param fromJar <code>true</code> para indicar se o prefixo  relativo ao
   *        JAR, <code>false</code> para indicar que  uma referncia externa
   */
  public XMLConverter(String dtdPrefix, boolean fromJar) {
    this(dtdPrefix, fromJar, null);
  }

  /**
   * Instancia um conversor.
   * 
   * @param dtdPrefix prefixo para o DTD. Pode ser uma URL, um path para o
   *        filesystem local ou <code>null</code>. Se
   *        <code>fromJar == true</code>, deve ser relativo ao JAR corrente
   * @param fromJar <code>true</code> para indicar se o prefixo  relativo ao
   *        JAR, <code>false</code> para indicar que  uma referncia externa
   * @param encoding encoding para leitura do XML (<code>null</code> para usar o
   *        encoding default). <b>Vlido apenas para converso via {@link File}
   *        ou {@link InputStream}</b>
   * 
   * @throws IllegalCharsetNameException se o encoding  invlido
   * @throws UnsupportedCharsetException se o encoding no  suportado por esta
   *         JVM
   */
  public XMLConverter(String dtdPrefix, boolean fromJar, String encoding) {
    this(dtdPrefix, fromJar, encoding, new XMLConversionInterface[0]);
  }

  /**
   * Instancia um conversor.
   * 
   * @param dtdPrefix prefixo para o DTD. Pode ser uma URL, um path para o
   *        filesystem local ou <code>null</code>. Se
   *        <code>fromJar == true</code>, deve ser relativo ao JAR corrente
   * @param fromJar <code>true</code> para indicar se o prefixo  relativo ao
   *        JAR, <code>false</code> para indicar que  uma referncia externa
   * @param encoding encoding para leitura do XML (<code>null</code> para usar o
   *        encoding default) <b>Vlido apenas para converso via {@link File}
   *        ou {@link InputStream}</b>
   * @param conversions converses a serem aplicadas, em ordem
   * 
   * @throws IllegalCharsetNameException se o encoding  invlido
   * @throws UnsupportedCharsetException se o encoding no  suportado por esta
   *         JVM
   */
  public XMLConverter(String dtdPrefix, boolean fromJar, String encoding,
    XMLConversionInterface... conversions) {
    setEncoding(encoding);
    setDTDprefix(dtdPrefix, fromJar);
    this.conversions = new ArrayList<>();
    Collections.addAll(this.conversions, conversions);
  }

  /**
   * Insancia um conversor, recebendo a lista de converses a serem aplicadas.
   * No define prefixo para o DTD e usa o encoding default.
   * 
   * @param conversions converses a serem aplicadas. As converses sero
   *        aplicadas na ordem em que foram definidas
   */
  public XMLConverter(XMLConversionInterface... conversions) {
    this(null, false, null, conversions);
  }

  /**
   * Define o prefixo para o DTD, apontando para uma URL ou o <i>filesystem</i>.
   * Para definir um prefixo relativo ao prprio JAR, use
   * {@link #setDTDprefix(String, boolean)}.
   * 
   * @param dtdPrefix prefixo para o DTD. Se no possuir o protocolo (
   *        <code>http://</code> ou <code>file://</code>) ser assumido um path
   *        (absoluto ou relativo) no <i>filesystem</i>
   * @return o prprio conversor, para encadeamento
   * 
   * @see #setDTDprefix(String, boolean)
   */
  public XMLConverter setDTDprefix(String dtdPrefix) {
    return setDTDprefix(dtdPrefix, false);
  }

  /**
   * Define um prefixo para o DTD, opcionalmente apontando para o prprio JAR.
   * 
   * @param dtdPrefix prefixo para o DTD
   * @param fromJar <code>true</code> para indicar que o prefixo se refere ao
   *        prprio JAR (neste caso deve ser um path absoluto ou relativo, sem
   *        protocolo)
   * @return o prprio conversor, para encadeamento
   * 
   * @see #setDTDprefix(String)
   */
  public XMLConverter setDTDprefix(String dtdPrefix, boolean fromJar) {
    this.dtdPrefix = dtdPrefix;
    this.fromJar = fromJar;
    conversionStatus = XMLConversionStatus.NOT_APPLIED;
    return this;
  }

  /**
   * Define o encoding do XML (<b>vlido apenas para leitura via {@link File} ou
   * {@link InputStream}</b>). Caso no seja executado, o encoding padro ser
   * usado.
   * 
   * @param encoding (<code>null</code> para usar o encoding padro)
   * @return o prprio conversor, para encadeamento
   * 
   * @throws IllegalCharsetNameException se o nome do charset  invlido
   * @throws UnsupportedCharsetException se o charset no  suportado por esta
   *         JVM
   */
  public XMLConverter setEncoding(String encoding) {
    if (encoding == null) {
      charset = Charset.defaultCharset();
    }
    else {
      charset = Charset.forName(encoding);
    }
    return this;
  }

  /**
   * Adiciona uma nova converso ao final da lista de converses.
   * 
   * @param conversion - nova converso
   * @return o prprio conversor, para encadeamento
   * @throws XMLConversionException se a nova converso no for capaz de
   *         processar a sada da ltima converso da lista
   */
  public XMLConverter addConversion(XMLConversionInterface conversion)
    throws XMLConversionException {
    if (conversions.isEmpty() == false) {
      XMLConversionInterface lastConv = conversions.get(conversions.size() - 1);
      if (conversion.canConvertFrom(lastConv) == false) {
        throw new XMLConversionException(
          XMLConversionExceptionType.BROKEN_CHAIN);
      }
    }
    conversions.add(conversion);
    return this;
  }

  /**
   * Aplica todas as converses ao documento para torn-lo compatvel com a
   * ltima verso do DTD. Percorre a lista de converses cadastradas em ordem,
   * buscando pela primeira converso capaz de processar o documento. A partir
   * da, aplica todas as converses em ordem, at o final da lista.
   * <p>
   * Uma vez iniciada a cadeia de converses, esta tem que prosseguir at o
   * final da lista. Se alguma converso recusar o processamento, h um erro de
   * construo (o processo de converso de um documento at a ltima verso do
   * documento tem que ser atmico).
   * 
   * @return o prprio conversor, para encadeamento
   * @throws XMLConversionException se houver algum erro no processo de
   *         converso. Diferentes tipos de problema podem acontecer, o tipo da
   *         exceo deve ser consultado para se identificar a causa exata.
   */
  private XMLConverter convert() throws XMLConversionException {
    conversionStatus = XMLConversionStatus.ERROR;
    if (conversions == null || conversions.isEmpty()) {
      throw new XMLConversionException(
        XMLConversionExceptionType.NO_CONVERSIONS);
    }
    try {
      docBuilder = new ConversionDocumentBuilder(validate, dtdPrefix, fromJar);
      document = new ConversionDocument(docBuilder.parse(inputSource));
    }
    catch (FileNotFoundException e) {
      /*
       * provavelmente a referncia para o DTD (considerando o prefixo) est
       * invlida
       */
      throw new XMLConversionException(
        XMLConversionExceptionType.FILE_NOT_FOUND, e);
    }
    catch (Exception e) {
      throw new XMLConversionException(XMLConversionExceptionType.PARSER, e);
    }

    String dtd = document.getDTD();

    /*
     * apenas para agilizar o processo, verificamos inicialmente se o documento
     * j no corresponde  ltima verso do DTD. Se corresponder, "fingimos"
     * que ele foi processado.
     */
    if (documentIsUpToDate()) {
      conversionStatus = XMLConversionStatus.NOT_NEEDED;
      lastConversion = conversions.get(conversions.size() - 1);
      lastConversion.setDocument(document);
      return this;
    }

    lastConversion = null;
    for (XMLConversionInterface conversion : conversions) {
      if (conversion.canConvertFrom(dtd) == false) {
        if (lastConversion != null) {
          /*
           * j iniciamos as converses, no poderia haver uma converso que
           * "quebrasse a corrente"
           */
          lastConversion = null;
          throw new XMLConversionException(
            XMLConversionExceptionType.BROKEN_CHAIN);
        }
        continue;
      }
      document = conversion.convert(document);
      dtd = conversion.getTargetDTD();
      /*
       * a converso foi bem sucedida, atualizamos o DTD de referncia do
       * documento
       */
      document.setDTD(dtd);
      lastConversion = conversion;
    }
    if (lastConversion == null) {
      /*
       * nenhuma converso foi aplicada. Isto quer dizer que apesar do arquivo
       * no ser compatvel com o DTD atual nenhuma das converses foi capaz de
       * convert-lo
       */
      throw new XMLConversionException(XMLConversionExceptionType.INVALID_DTD);
    }
    // tentamos validar o resultado
    validateOutput();

    // a validao foi bem sucedida, indicamos sucesso
    conversionStatus = XMLConversionStatus.SUCCESS;
    return this;
  }

  /**
   * Converte um XML armazenado em uma string.
   * 
   * @param content string contendo o documento XML
   * @param validate <code>true</code> se o XML deve ser validado de acordo com
   *        seu DTD
   * @return o prprio conversor, para encadeamento
   * @throws XMLConversionException se houver algum erro na converso
   */
  public XMLConverter convert(String content, boolean validate)
    throws XMLConversionException {
    return convert(new StringReader(content), validate);
  }

  /**
   * Converte o XML armazenado em um arquivo.
   * 
   * @param src arquivo XML
   * @param validate <code>true</code> se o XML deve ser validado de acordo com
   *        seu DTD
   * @return o prprio conversor, para encadeamento
   * @throws XMLConversionException se houver algum erro na converso
   * @throws FileNotFoundException se o arquivo passado como parmetro era no
   *         existe
   */
  public XMLConverter convert(File src, boolean validate)
    throws XMLConversionException, FileNotFoundException {
    InputStream stream = null;
    try {
      stream = new FileInputStream(src);
      return convert(stream, validate);
    }
    finally {
      close(stream);
    }
  }

  /**
   * Converte um XML acessado a partir de um {@link InputStream}.
   * 
   * @param src stream para leitura do XML
   * @param validate <code>true</code> se o XML deve ser validado de acordo com
   *        seu DTD
   * @return o prprio conversor, para encadeamento
   * @throws XMLConversionException se houver algum erro na converso
   */
  public XMLConverter convert(InputStream src, boolean validate)
    throws XMLConversionException {
    InputStreamReader reader = new InputStreamReader(src, charset);
    return convert(reader, validate);
  }

  /**
   * Converte um XML acessado a partir de um {@link Reader}.
   * 
   * @param src reader para leitura do XML
   * @param validate <code>true</code> se o XML deve ser validado de acordo com
   *        seu DTD
   * @return o prprio conversor, para encadeamento
   * @throws XMLConversionException se houver algum erro na converso
   */
  public XMLConverter convert(Reader src, boolean validate)
    throws XMLConversionException {
    inputSource = new InputSource(src);
    this.validate = validate;
    return convert();
  }

  /**
   * Valida o contedo do documento em memria.
   * 
   * @throws XMLConversionException se ocorrer um erro na validao
   */
  private void validateOutput() throws XMLConversionException {
    document.normalizeDocument();
    try {
      Transformer transformer =
        TransformerFactory.newInstance().newTransformer();

      String dtd = document.getDTD();
      if (dtd != null) {
        // necessrio para gravao da linha <!DOCTYPE ... > com o DTD
        transformer.setOutputProperty(OutputKeys.DOCTYPE_SYSTEM, dtd);
      }
      else {
        // documento no deve ter a declarao do DTD
        transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
      }

      // definimos o mtodo usado para gravar o documento
      transformer.setOutputProperty(OutputKeys.METHOD, "xml");

      // definimos o encoding
      transformer.setOutputProperty(OutputKeys.ENCODING, charset.name());

      /*
       * primeiro gravamos o resultado em um StringWriter
       */
      StringWriter stringWriter = new StringWriter();
      StreamResult result = new StreamResult(stringWriter);
      DOMSource source = new DOMSource(document);
      transformer.transform(source, result);
      String resultAsString = stringWriter.toString();

      /*
       * agora, lemos o resultado e o processamos, validando-o. Criamos um novo
       * DocumentBuilder para garantir que usaremos validao.
       */
      ConversionDocumentBuilder validationDocBuilder =
        new ConversionDocumentBuilder(true, dtdPrefix, fromJar);
      validationDocBuilder.parse(new InputSource(new StringReader(
        resultAsString)));
      resultAsString = null;
    }
    catch (Exception e) {
      throw new XMLConversionException(
        XMLConversionExceptionType.CANT_VALIDATE, e);
    }
  }

  /**
   * Aplica todas as converses e retorna o resultado como uma string.
   * 
   * @return resultado da converso, ou <code>null</code> em caso de erro
   * @throws XMLConversionException se houver algum erro no processo de
   *         converso. Diferentes tipos de problema podem acontecer, o tipo da
   *         exceo deve ser consultado para identificar a causa exata.
   */
  public String saveToString() throws XMLConversionException {
    checkConversionsWereApplied();
    switch (conversionStatus) {
      case NOT_NEEDED:
      case SUCCESS:
        StringWriter writer = new StringWriter();
        lastConversion.writeXML(writer);
        return writer.toString();

      case ERROR:
      case NOT_APPLIED:
        return null;

      default:
        throw new AssertionError("status desconhecido "
          + conversionStatus.name());
    }
  }

  /**
   * Salva o resultado da converso em um arquivo.
   * 
   * @param file arquivo de destino
   * @return <code>true</code> se o resultado foi gravado com sucesso
   * @throws XMLConversionException se houver algum erro no processo de
   *         converso. Diferentes tipos de problema podem acontecer, o tipo da
   *         exceo deve ser consultado para identificar a causa exata.
   * @throws FileNotFoundException se o arquivo passado como parmetro era no
   *         existe
   * 
   * @see #saveTo(OutputStream)
   */
  public boolean saveTo(File file) throws XMLConversionException,
    FileNotFoundException {
    OutputStream stream = null;
    try {
      stream = new FileOutputStream(file);
      return saveTo(stream);
    }
    finally {
      close(stream);
    }
  }

  /**
   * Salva o resultado da converso em um stream de sada, usando o encoding
   * default.
   * <p>
   * IMPORTANTE: o stream <b>no</b> ser fechado as a gravao.
   * 
   * @param stream stream de sada
   * @return <code>true</code> se o resultado foi gravado com sucesso
   * 
   * @throws XMLConversionException se houver algum erro no processo de
   *         converso. Diferentes tipos de problema podem acontecer, o tipo da
   *         exceo deve ser consultado para identificar a causa exata.
   * 
   * @see #saveTo(File)
   */
  public boolean saveTo(OutputStream stream) throws XMLConversionException {
    return saveTo(stream, charset);
  }

  /**
   * Salva o resultado da converso em um stream de sada, usando um encoding
   * especfico.
   * <p>
   * IMPORTANTE: o stream <b>no</b> ser fechado as a gravao.
   * 
   * @param stream stream de sada
   * @param charset
   * @return <code>true</code> se o resultado foi gravado com sucesso
   * @throws XMLConversionException se houver algum erro no processo de
   *         converso. Diferentes tipos de problema podem acontecer, o tipo da
   *         exceo deve ser consultado para identificar a causa exata.
   */
  private boolean saveTo(OutputStream stream, Charset charset)
    throws XMLConversionException {
    return saveTo(new OutputStreamWriter(stream, charset));
  }

  /**
   * Salva o resultado da converso em um {@link Writer}.
   * 
   * @param writer o writer de sada
   * @return <code>true</code> se o resultado foi gravado com sucesso
   * @throws XMLConversionException se houver algum erro no processo de
   *         converso. Diferentes tipos de problema podem acontecer, o tipo da
   *         exceo deve ser consultado para identificar a causa exata.
   */
  public boolean saveTo(Writer writer) throws XMLConversionException {
    checkConversionsWereApplied();
    switch (conversionStatus) {
      case SUCCESS:
      case NOT_NEEDED:
        lastConversion.writeXML(writer);
        return true;

      case ERROR:
      case NOT_APPLIED:
        return false;

      default:
        throw new AssertionError("status desconhecido "
          + conversionStatus.name());
    }
  }

  /**
   * Garante que as converses foram aplicadas sem erros.
   * 
   * @throws XMLConversionException caso tenha havido algum erro
   */
  private void checkConversionsWereApplied() throws XMLConversionException {
    if (conversions == null || conversions.isEmpty()) {
      throw new XMLConversionException(
        XMLConversionExceptionType.NO_CONVERSIONS);
    }
    if (document == null) {
      throw new XMLConversionException(XMLConversionExceptionType.NO_INPUT);
    }
    if (lastConversion == null && documentIsUpToDate() == false) {
      throw new XMLConversionException(
        XMLConversionExceptionType.CONVERSIONS_NOT_APPLIED);
    }
  }

  /**
   * Indica o status da converso.
   * 
   * @return cdigo indicando o status da converso
   */
  public XMLConversionStatus getConversionStatus() {
    return conversionStatus;
  }

  /**
   * Indica se o documento ainda precisa ser convertido.
   * 
   * @see XMLConversionInterface#canConvertFrom(String)
   * 
   * @return flag indicando se o documento corrente no precisa mais ser
   *         convertido
   */
  private boolean documentIsUpToDate() {
    XMLConversionInterface lastRegisteredConversion =
      conversions.get(conversions.size() - 1);
    return lastRegisteredConversion.convertsTo(document.getDTD());
  }

  /**
   * Retorna a lista de erros de processamento do XML.
   * 
   * @return lista com os erros ocorridos durante o processamento do XML. Caso
   *         no tenha ocorrido nenhum erro, a lista estar vazia
   * @throws XMLConversionException se o parser ainda no foi instanciado
   */
  public List<String> getParserErrors() throws XMLConversionException {
    if (docBuilder == null) {
      throw new XMLConversionException(XMLConversionExceptionType.NO_INPUT);
    }
    return docBuilder.getErrorList();
  }

  /**
   * Indica se houve erros no processamento do XML.
   * 
   * @return flag indicando se houve erros no processamento do XML
   * @throws XMLConversionException se o parser ainda no foi instanciado
   */
  public boolean hasParserErrors() throws XMLConversionException {
    if (docBuilder == null) {
      throw new XMLConversionException(XMLConversionExceptionType.NO_INPUT);
    }
    return docBuilder.hasErrors();
  }

  /**
   * Utilitrio que fecha um {@link Closeable}.
   * 
   * @param closeable objeto a ser fechado.
   */
  private void close(Closeable closeable) {
    if (closeable != null) {
      try {
        closeable.close();
      }
      catch (IOException e) {
        //faz nada
      }
    }
  }
}
