package busexplorer.exception.handling;

/**
 * Classe utilitria para reusar os cdigos de tratamento de excees.
 * 
 * @author Tecgraf
 * @param <T> O tipo de exceo que ser tratado por esta classe.
 */
public abstract class ExceptionHandler<T extends HandlingException<?>> {

  /**
   * Mtodo a ser disparado pela aplicao para tratar a exceo ocorrida no
   * contexto indicado.
   * 
   * @param theException a exceo.
   * @param context o contexto de execuo em que o erro ocorreu.
   * @return um enum do tipo da exceo para que a aplicao possa fazer uso de
   *         switch, facilitando o tratamento da exceo que  particular a cada
   *         contexto de execuo (parte do tratamento da exceo que no 
   *         reutilizvel para todos os contextos)
   */
  public T process(Exception theException, ExceptionContext context) {
    T exception = getHandlingException(theException, context);
    handleException(exception);
    return exception;
  }

  /**
   * A idia deste mtodo  chamar o construtor de {@link HandlingException}
   * especfco, que por sua vez ir utilizar o mtodo
   * {@link HandlingException#getTypeFromException(Exception)} para obter a
   * enumerao especfica ({@link ExceptionType}).
   * 
   * @param realException a exceo ocorrida.
   * @param context o contexto da exceo.
   * @return a instncia de {@link HandlingException} a ser tratada.
   */
  protected abstract T getHandlingException(Exception realException,
    ExceptionContext context);

  /**
   * Mtodo a ser definido pela aplicao para de fato explicitar o tratamento
   * da exceo que deve ser reutilizado.
   * 
   * @param exception a exceo a ser tratada.
   */
  protected abstract void handleException(T exception);

}
