package br.pucrio.tecgraf.soma.serviceapi.configuration;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.MissingOptionException;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

public class ServiceConfiguration {

  private Options options;
  private CommandLine commandLine;
  private Map<String, String> defaultValues;
  public static final String HELP_SHORT_NAME = "h";
  private static final String HELP_LONG_NAME = "help";
  public static final char DEFAULT_VALUE_SEPARATOR = '=';

  public ServiceConfiguration() {
    options = new Options();
    options.addOption(HELP_SHORT_NAME, HELP_LONG_NAME, false, "Print the usage help");
    defaultValues = new HashMap<String, String>();
  }

  public void addOption(char shortName, String longName, String description, boolean hasValue) {
    addOption(shortName, longName, description, hasValue, null, hasValue ? 1 : 0, false, true,
        DEFAULT_VALUE_SEPARATOR, null);
  }

  public void addOption(ConfigurationOption argument) {
    addOption(argument.shortName, argument.longName, argument.description, argument.hasValue,
        argument.valueName, argument.numberOfValues, argument.areValuesOptional, argument.required,
        DEFAULT_VALUE_SEPARATOR, argument.defaultValue);
  }

  public void addOption(Character shortName, String longName, String description, boolean hasValue,
      String valueName, int numberOfValues, boolean areValuesOptional, boolean required,
      char valueSeparator, String defaultValue) {
    options.addOption(Option.builder(shortName == null ? null : shortName.toString())
        .argName(valueName).desc(description).hasArg(hasValue).longOpt(longName)
        .numberOfArgs(numberOfValues).optionalArg(areValuesOptional).required(required)
        .valueSeparator(valueSeparator).build());
    if (!required) {
      if (shortName != null)
        defaultValues.put(shortName.toString(), defaultValue);
      if (longName != null)
        defaultValues.put(longName, defaultValue);
    }
  }

  public boolean hasOption(String opt) {
    return commandLine.hasOption(opt);
  }

  public String getValue(String parameterName) {
    if (defaultValues.containsKey(parameterName))
      return commandLine.getOptionValue(parameterName, defaultValues.get(parameterName));
    return commandLine.getOptionValue(parameterName);
  }

  public static ServiceConfiguration build(List<ConfigurationOption> configurations, String[] args,
      boolean exitIfHelp) throws ParseException {
    ServiceConfiguration serviceConfiguration = new ServiceConfiguration();
    for (ConfigurationOption config : configurations)
      serviceConfiguration.addOption(config);
    build(serviceConfiguration, args, exitIfHelp);
    return serviceConfiguration;
  }

  public static void build(ServiceConfiguration serviceConfiguration, String[] args,
      boolean exitIfHelp) throws ParseException {
    HelpFormatter formatter = new HelpFormatter();
    try {
      serviceConfiguration.commandLine =
          new DefaultParser().parse(serviceConfiguration.options, args);
      if (serviceConfiguration.commandLine.hasOption(HELP_SHORT_NAME) && exitIfHelp) {
        formatter.printHelp("COMMAND", serviceConfiguration.options, true);
        System.exit(0);
      }
    } catch (MissingOptionException moe) {
      System.out.println("\n" + moe.getMessage());
      formatter.printHelp("COMMAND", serviceConfiguration.options, true);
      System.exit(0);
    }
  }

}
