package br.pucrio.tecgraf.soma.job.application.appservice;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.List;
import java.util.stream.Collectors;

import org.jboss.logging.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.client.HttpClientErrorException;

import br.pucrio.tecgraf.soma.job.api.model.ReplicaJob;
import br.pucrio.tecgraf.soma.job.api.model.ReplicaJobBatchResponse;
import br.pucrio.tecgraf.soma.job.api.model.ReplicaJobResponse;
import br.pucrio.tecgraf.soma.job.application.service.MultiflowService;
import br.pucrio.tecgraf.soma.job.application.service.ProjectService;
import br.pucrio.tecgraf.soma.job.application.service.ReplicaService;
import br.pucrio.tecgraf.soma.job.domain.model.Replica;
import jakarta.ws.rs.ForbiddenException;

@Service
public class ReplicaAppService {
  private static final Logger LOG = Logger.getLogger(ReplicaAppService.class);
  
  @Autowired
  private ProjectService projectService;
  @Autowired
  private MultiflowService multiflowService;
  @Autowired
  private ReplicaService replicaService;

  /**
   * Persiste os dados da execução de uma lista de réplicas.
   *
   * @param replicaJobs dados das execuções das réplicas.
   * @return lista de informações das execuções das réplicas
   */
  @Transactional
  public List<ReplicaJobBatchResponse> createReplicaJobs(List<ReplicaJob> replicaJobs) {

    br.pucrio.tecgraf.soma.job.domain.model.Multiflow multiflow = multiflowService.findMultiflowById(replicaJobs.get(0)
      .getMultiflowId());

    validateMultiflowId(multiflow);
    
    return replicaJobs.stream().map(replicaJob -> {
      try {
        validateReplicaJobId(replicaJob);
        return convertToBatchResponse(persistReplicaJob(replicaJob, multiflow));
      }
      catch (HttpClientErrorException e) { // Job ID já está associado a alguma réplica
        String errorMsg = "HTTP error creating a Replica. " + e.getMessage();
        LOG.info(errorMsg);
        ReplicaJobBatchResponse response = new ReplicaJobBatchResponse();
        return response.error(e.getStatusText());
      }
      catch (Exception e) { // Demais erros
        String errorMsg = "Unknown error creating a Replica: " + e.getMessage();
        LOG.info(errorMsg);
        ReplicaJobBatchResponse response = new ReplicaJobBatchResponse();
        return response.error(errorMsg);
      }
    }).collect(Collectors.toList());
  }

  /**
   * Persiste os dados de uma execução de uma réplica.
   *
   * @param replicaJobData dados da execução de uma réplica.
   * @return o id da réplica
   */
  @Transactional
  public ReplicaJobResponse createReplicaJob(ReplicaJob replicaJobData) {
    br.pucrio.tecgraf.soma.job.domain.model.Multiflow multiflow = multiflowService.findMultiflowById(replicaJobData
      .getMultiflowId());

    validateMultiflowId(multiflow);
    validateReplicaJobId(replicaJobData);

    return persistReplicaJob(replicaJobData, multiflow);
  }

  private void validateMultiflowId(br.pucrio.tecgraf.soma.job.domain.model.Multiflow multiflow) {
    if (!projectService.hasPermission(multiflow.getProjectId())) {
      throw new ForbiddenException("User has no permission to add a Replica to this project");
    }
  }

  private ReplicaJobResponse persistReplicaJob(ReplicaJob replicaJobData,
    br.pucrio.tecgraf.soma.job.domain.model.Multiflow multiflow) {

    Replica replica = replicaService.findReplicaBy(multiflow.getId(), replicaJobData.getLineNumber().intValue());
    if (replica == null) {
      replica = new Replica();
      replica.setMultiflow(multiflow);
      replica.setLineNumber(replicaJobData.getLineNumber());
      replicaService.createReplica(replica);
    }

    br.pucrio.tecgraf.soma.job.domain.model.ReplicaJob replicaJob =
      new br.pucrio.tecgraf.soma.job.domain.model.ReplicaJob();
    // TODO: eliminar a possibilidade de números de versão repetidos para uma mesma réplica (WEBSNTS-19170)
    replicaJob.setVersion(replicaService.getNextReplicaJobVersion(replica));
    replicaJob.setJobStringId(replicaJobData.getJobId());
    replicaJob.setSubmissionTime(LocalDateTime.now(ZoneOffset.UTC));
    replicaJob.setReplicaDependencyRaw(replicaJobData.getReplicaDependencyRaw());
    replica.addReplicaJob(replicaJob);
    replicaService.updateReplica(replica);

    return convertToBasicRESTModel(replicaService.findReplicaJobBy(replicaJob.getJobStringId()));
  }

  private void validateReplicaJobId(ReplicaJob replicaJobData) {
    br.pucrio.tecgraf.soma.job.domain.model.ReplicaJob replicaJobFound = replicaService.findReplicaJobBy(replicaJobData
      .getJobId());
    if (replicaJobFound != null) {
      throw new HttpClientErrorException(HttpStatus.CONFLICT, "Job ID " + replicaJobData.getJobId()
        + " exists in line number " + replicaJobFound.getReplica().getLineNumber());
    }
  }

  private static ReplicaJobResponse convertToBasicRESTModel(
    br.pucrio.tecgraf.soma.job.domain.model.ReplicaJob replicaJobDbModel) {
    ReplicaJobResponse restModel = new ReplicaJobResponse();
    Replica replica = replicaJobDbModel.getReplica();
    restModel.setLineNumber(replica.getLineNumber());
    restModel.setMultiflowId(replica.getMultiflow().getId());
    restModel.setMultiflowName(replica.getMultiflow().getName());
    restModel.setJobId(replicaJobDbModel.getJobStringId());
    restModel.setVersion(replicaJobDbModel.getVersion());
    restModel.replicaDependencyRaw(replicaJobDbModel.getReplicaDependencyRaw());
    return restModel;
  }
  
  /**
   * Transforma um objeto de resposta simples em um objeto de resposta batch.
   * A diferença é que o objeto de resposta batch contém um campo de erro.
   * 
   * Neste caso, como tem uma resposta válida, NÃO preenche o campo de erro.
   * 
   * @param response
   * @return
   */
  private static ReplicaJobBatchResponse convertToBatchResponse(ReplicaJobResponse response) {
    ReplicaJobBatchResponse batchResponse = new ReplicaJobBatchResponse();
    batchResponse.setVersion(response.getVersion());
    return batchResponse;
  }
}
