package br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository;

import br.pucrio.tecgraf.soma.job.domain.model.Job;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.specification.GroupsInListSpecification;
import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.Sort;
import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.impl.JPARepository;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.JPASpecification;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import jakarta.persistence.TypedQuery;
import jakarta.persistence.criteria.*;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

@Transactional
@Repository
public class JobRepository extends JPARepository<Job> {

  @PersistenceContext
  private EntityManager _entityManager;

  @Override
  public Class<Job> getType() {
    return Job.class;
  }

  @Override
  public EntityManager getEntityManager() {
    return _entityManager;
  }

  public void setEntityManager(EntityManager entityManager) {
    this._entityManager = entityManager;
  }

  public List<Job> getJobs(Collection<Long> ids, Sort... sorts) {
    EntityManager manager = getEntityManager();
    CriteriaBuilder builder = manager.getCriteriaBuilder();
    CriteriaQuery<Job> criteria = builder.createQuery(Job.class);
    Root<Job> root = criteria.from(Job.class);
    root.alias("getJobs");
    criteria.where(root.get("id").in(ids))
        .orderBy(createSorting(builder, root, sorts));
    return manager.createQuery(criteria).getResultList();
  }

  public List<String> findNotSingletonGroupIds(List<String> ids) {
    EntityManager manager = getEntityManager();
    CriteriaBuilder builder = manager.getCriteriaBuilder();
    CriteriaQuery<String> criteria = builder.createQuery(String.class);
    Root<Job> root = criteria.from(Job.class);
    root.alias("findNotSingletonGroupIds");
    Path<String> groupId = root.get("groupId");
    GroupsInListSpecification specification = new GroupsInListSpecification(ids);
    criteria.select(groupId)
        .where(specification.toPredicate(root, builder))
        .having(builder.greaterThan(builder.countDistinct(root.get("id")), 1L))
        .groupBy(groupId);
    return manager.createQuery(criteria).getResultList();
  }

  public List<Long> findJobIds(JPASpecification<Job> specification,
      int maxResult,
      int offset,
      Sort... sorts) {
    EntityManager manager = getEntityManager();
    CriteriaBuilder builder = manager.getCriteriaBuilder();
    CriteriaQuery<Long> criteria = builder.createQuery(Long.class);
    Root<Job> root = criteria.from(Job.class);
    root.alias("findJobIds");
    Predicate predicate = specification.toPredicate(root, builder);
    criteria.select(root.get("id"))
        .where(predicate)
        .orderBy(createSorting(builder, root, sorts));
    TypedQuery<Long> query = manager.createQuery(criteria);
    query.setMaxResults(maxResult);
    if (offset > 0)
      query.setFirstResult(offset);
    return query.getResultList();
  }

  public long countGroups(JPASpecification<Job> specification) {
    CriteriaBuilder builder = getEntityManager().getCriteriaBuilder();
    CriteriaQuery<Long> criteria = builder.createQuery(Long.class);
    Root<Job> jobRoot = criteria.from(Job.class);
    jobRoot.alias("countGroups");
    criteria.select(builder.countDistinct(jobRoot.get("groupId")))
        .where(specification.toPredicate(jobRoot, builder));
    return getEntityManager().createQuery(criteria).getSingleResult();
  }

  public long countJobs(JPASpecification<Job> specification) {
    CriteriaBuilder builder = getEntityManager().getCriteriaBuilder();
    CriteriaQuery<Long> criteria = builder.createQuery(Long.class);
    Root<Job> jobRoot = criteria.from(Job.class);
    jobRoot.alias("countJobs");
    criteria.select(builder.countDistinct(jobRoot.get("id")))
        .where(specification.toPredicate(jobRoot, builder));
    return getEntityManager().createQuery(criteria).getSingleResult();
  }

  public List<String> findGroupIdsWithDescriptionVariation(List<String> groupIds) {
      String hql = """
                  SELECT j.groupId
                  FROM Job j
                  WHERE j.groupId IN :groupIds
                  GROUP BY j.groupId
                  HAVING COUNT(DISTINCT function('md5', j.description)) > 1
              """;

      return getEntityManager()
              .createQuery(hql, String.class)
              .setParameter("groupIds", groupIds)
              .getResultList();
  }

  public List<JobGroupValCount> findGroupedJobs(JPASpecification<Job> specification,
      int maxResult,
      int offset,
      Sort... sorts) {
    CriteriaBuilder builder = getEntityManager().getCriteriaBuilder();

    /*
     * obtem os IDs dos jobs da "página" solicitada.
     */
    CriteriaQuery<JobGroupValCount> grpCriteria = builder.createQuery(JobGroupValCount.class);
    Root<Job> jobRoot = grpCriteria.from(Job.class);
    jobRoot.alias("findGroupedJobs");
    Path<Long> jobId = jobRoot.get("id");
    Path<String> jobGrpId = jobRoot.get("groupId");
    grpCriteria.select(builder.construct(JobGroupValCount.class,
        jobGrpId,
        builder.min(jobId),
        builder.count(jobId),
        builder.countDistinct(jobRoot.get("projectId")),
        builder.countDistinct(jobRoot.get("jobOwner")),
        builder.countDistinct(jobRoot.get("automaticallyMachineSelection")),
        builder.countDistinct(jobRoot.get("numberOfProcesses")),
        builder.countDistinct(jobRoot.get("numberOfProcessesByMachine")),
        builder.countDistinct(jobRoot.get("priority")),
        builder.countDistinct(jobRoot.get("multipleExecution")),
        builder.countDistinct(jobRoot.get("jobType")),
        builder.countDistinct(jobRoot.get("executionMachine")),
        builder.countDistinct(jobRoot.get("exitCode")),
        builder.countDistinct(jobRoot.get("guiltyNodeId")),
        builder.countDistinct(jobRoot.get("exitStatus")),
        builder.countDistinct(jobRoot.get("flowId")),
        builder.countDistinct(jobRoot.get("flowVersion")),
        builder.countDistinct(jobRoot.get("flowName")),
        builder.min(jobRoot.get("submissionTime")),
        builder.max(jobRoot.get("lastModifiedTime"))))
        .where(specification.toPredicate(jobRoot, builder))
        .groupBy(jobGrpId)
        .orderBy(createGroupSorting(builder, jobRoot, sorts));

    // limita os resultados à "página" solicitada.
    TypedQuery<JobGroupValCount> grpQuery = getEntityManager().createQuery(grpCriteria);
    grpQuery.setMaxResults(maxResult);
    if (offset > 0)
      grpQuery.setFirstResult(offset);

    return grpQuery.getResultList();
  }

  private <T> List<Order> createSorting(CriteriaBuilder builder,
      Root<T> root,
      Sort... sorts) {
    List<Order> sorting = new ArrayList<>();
    for (Sort sort : sorts) {
      if (sort.isAscending()) {
        sorting.add(builder.asc(root.get(sort.getAttribute())));
      } else {
        sorting.add(builder.desc(root.get(sort.getAttribute())));
      }
    }
    return sorting;
  }

  private <T> List<Order> createGroupSorting(CriteriaBuilder builder,
      Root<T> root,
      Sort... sorts) {
    List<Order> sorting = new ArrayList<>();
    for (Sort sort : sorts) {
      if (sort.isAscending()) {
        sorting.add(builder.asc(builder.min(root.get(sort.getAttribute()))));
      } else {
        sorting.add(builder.desc(builder.max(root.get(sort.getAttribute()))));
      }
    }
    return sorting;
  }
}
