/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.bag.immutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;

import com.gs.collections.api.BooleanIterable;
import com.gs.collections.api.LazyBooleanIterable;
import com.gs.collections.api.bag.ImmutableBag;
import com.gs.collections.api.bag.primitive.BooleanBag;
import com.gs.collections.api.bag.primitive.ImmutableBooleanBag;
import com.gs.collections.api.bag.primitive.MutableBooleanBag;
import com.gs.collections.api.block.function.primitive.BooleanToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectBooleanToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.BooleanPredicate;
import com.gs.collections.api.block.procedure.primitive.BooleanIntProcedure;
import com.gs.collections.api.block.procedure.primitive.BooleanProcedure;
import com.gs.collections.api.iterator.BooleanIterator;
import com.gs.collections.api.list.primitive.MutableBooleanList;
import com.gs.collections.api.set.primitive.MutableBooleanSet;
import com.gs.collections.impl.bag.mutable.primitive.BooleanHashBag;
import com.gs.collections.impl.block.procedure.checked.primitive.CheckedBooleanIntProcedure;
import com.gs.collections.impl.iterator.UnmodifiableBooleanIterator;

/**
 * ImmutableBooleanHashBag is the non-modifiable equivalent of {@link BooleanHashBag}.
 * This file was automatically generated from template file immutablePrimitiveHashBag.stg.
 *
 * @since 4.0.
 */
final class ImmutableBooleanHashBag implements ImmutableBooleanBag, Serializable
{
    private static final long serialVersionUID = 1L;

    private final MutableBooleanBag delegate;

    private ImmutableBooleanHashBag(boolean[] newElements)
    {
        this.checkOptimizedSize(newElements.length);
        this.delegate = BooleanHashBag.newBagWith(newElements);
    }

    private void checkOptimizedSize(int length)
    {
        if (length <= 1)
        {
            throw new IllegalArgumentException("Use BooleanBags.immutable.with() to instantiate an optimized collection");
        }
    }

    public static ImmutableBooleanHashBag newBagWith(boolean... elements)
    {
        return new ImmutableBooleanHashBag(elements);
    }

    public ImmutableBooleanBag newWith(boolean element)
    {
        return BooleanHashBag.newBag(this.delegate).with(element).toImmutable();
    }

    public ImmutableBooleanBag newWithout(boolean element)
    {
        BooleanHashBag hashBag = BooleanHashBag.newBag(this.delegate);
        hashBag.remove(element);
        return hashBag.toImmutable();
    }

    public ImmutableBooleanBag newWithAll(BooleanIterable elements)
    {
        BooleanHashBag bag = BooleanHashBag.newBag(this.delegate);
        bag.addAll(elements);
        return bag.toImmutable();
    }

    public ImmutableBooleanBag newWithoutAll(BooleanIterable elements)
    {
        BooleanHashBag bag = BooleanHashBag.newBag(this.delegate);
        bag.removeAll(elements);
        return bag.toImmutable();
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return this.delegate.isEmpty();
    }

    public boolean notEmpty()
    {
        return this.delegate.notEmpty();
    }

    public boolean contains(boolean value)
    {
        return this.delegate.contains(value);
    }

    public boolean containsAll(BooleanIterable source)
    {
        return this.delegate.containsAll(source);
    }

    public boolean containsAll(boolean... elements)
    {
        return this.delegate.containsAll(elements);
    }

    public void forEach(BooleanProcedure procedure)
    {
        this.delegate.forEach(procedure);
    }

    public ImmutableBooleanBag select(BooleanPredicate predicate)
    {
        return this.delegate.select(predicate).toImmutable();
    }

    public ImmutableBooleanBag reject(BooleanPredicate predicate)
    {
        return this.delegate.reject(predicate).toImmutable();
    }

    public <V> ImmutableBag<V> collect(BooleanToObjectFunction<? extends V> function)
    {
        return this.delegate.collect(function).toImmutable();
    }

    public MutableBooleanList toList()
    {
        return this.delegate.toList();
    }

    public int sizeDistinct()
    {
        return this.delegate.sizeDistinct();
    }

    public int occurrencesOf(boolean item)
    {
        return this.delegate.occurrencesOf(item);
    }

    public void forEachWithOccurrences(BooleanIntProcedure booleanIntProcedure)
    {
        this.delegate.forEachWithOccurrences(booleanIntProcedure);
    }

    public boolean detectIfNone(BooleanPredicate predicate, boolean ifNone)
    {
        return this.delegate.detectIfNone(predicate, ifNone);
    }

    public int count(BooleanPredicate predicate)
    {
        return this.delegate.count(predicate);
    }

    public boolean anySatisfy(BooleanPredicate predicate)
    {
        return this.delegate.anySatisfy(predicate);
    }

    public boolean noneSatisfy(BooleanPredicate predicate)
    {
        return this.delegate.noneSatisfy(predicate);
    }

    public boolean allSatisfy(BooleanPredicate predicate)
    {
        return this.delegate.allSatisfy(predicate);
    }

    public <T> T injectInto(T injectedValue, ObjectBooleanToObjectFunction<? super T, ? extends T> function)
    {
        return this.delegate.injectInto(injectedValue, function);
    }

    @Override
    public boolean equals(Object obj)
    {
        return this.delegate.equals(obj);
    }

    @Override
    public int hashCode()
    {
        return this.delegate.hashCode();
    }

    public MutableBooleanSet toSet()
    {
        return this.delegate.toSet();
    }

    public MutableBooleanBag toBag()
    {
        return this.delegate.toBag();
    }

    public ImmutableBooleanBag toImmutable()
    {
        return this;
    }

    public LazyBooleanIterable asLazy()
    {
        return this.delegate.asLazy();
    }

    public boolean[] toArray()
    {
        return this.delegate.toArray();
    }

    @Override
    public String toString()
    {
        return this.delegate.toString();
    }

    public String makeString()
    {
        return this.delegate.makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.appendString(appendable, separator);
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        this.delegate.appendString(appendable, start, separator, end);
    }

    public BooleanIterator booleanIterator()
    {
        return new UnmodifiableBooleanIterator(this.delegate.booleanIterator());
    }

    private Object writeReplace()
    {
        return new ImmutableBooleanBagSerializationProxy(this);
    }

    protected static class ImmutableBooleanBagSerializationProxy implements Externalizable
    {
        private static final long serialVersionUID = 1L;

        private BooleanBag bag;

        @SuppressWarnings("UnusedDeclaration")
        public ImmutableBooleanBagSerializationProxy()
        {
            // Empty constructor for Externalizable class
        }

        protected ImmutableBooleanBagSerializationProxy(BooleanBag bag)
        {
            this.bag = bag;
        }

        public void writeExternal(final ObjectOutput out) throws IOException
        {
            out.writeInt(this.bag.sizeDistinct());
            try
            {
                this.bag.forEachWithOccurrences(new CheckedBooleanIntProcedure()
                {
                    @Override
                    public void safeValue(boolean item, int count) throws IOException
                    {
                        out.writeBoolean(item);
                        out.writeInt(count);
                    }
                });
            }
            catch (RuntimeException e)
            {
                if (e.getCause() instanceof IOException)
                {
                    throw (IOException) e.getCause();
                }
                throw e;
            }
        }

        public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
        {
            int size = in.readInt();
            MutableBooleanBag deserializedBag = new BooleanHashBag();

            for (int i = 0; i < size; i++)
            {
                deserializedBag.addOccurrences(in.readBoolean(), in.readInt());
            }

            this.bag = deserializedBag;
        }

        protected Object readResolve()
        {
            return this.bag.toImmutable();
        }
    }
}
